# "Invalid outcome name" Error - Complete Solution

## 🚨 Problem
You're getting the error: **"Invalid outcome name. Please try again or contact support."**

This happens when trying to place bets on multi-outcome markets because the database constraint `bets_outcome_check` is too restrictive and doesn't allow outcome names like "Emmerson Mnangagwa", "Nelson Chamisa", etc.

## 🔍 Root Cause
The database constraint `bets_outcome_check` was designed for binary markets (yes/no) but doesn't support multi-outcome market names like:
- "Emmerson Mnangagwa"
- "Nelson Chamisa" 
- "Douglas Mwonzora"
- "Thokozani Khupe"

## ✅ Solution

### Option 1: Apply Database Fix (Recommended)

**If you have database access:**

1. **Connect to your database** (PostgreSQL/Supabase)
2. **Run this SQL script:**

```sql
-- Drop the restrictive constraint
ALTER TABLE bets DROP CONSTRAINT IF EXISTS bets_outcome_check;

-- Add a flexible constraint that allows any outcome name
ALTER TABLE bets ADD CONSTRAINT bets_outcome_check 
CHECK (
  outcome IS NOT NULL 
  AND length(trim(outcome)) > 0 
  AND length(outcome) <= 100
);

-- Update status constraint
ALTER TABLE bets DROP CONSTRAINT IF EXISTS bets_status_check;
ALTER TABLE bets ADD CONSTRAINT bets_status_check 
CHECK (status IN (
  'pending', 'matched', 'cancelled', 'seeded', 
  'live', 'won', 'lost', 'active', 'inactive'
));

-- Add outcome_id column for multi-outcome markets
ALTER TABLE bets ADD COLUMN IF NOT EXISTS outcome_id UUID;
```

### Option 2: Use the Provided Script

**Run the fix script:**
```bash
# Make sure you're in the project directory
cd /home/tyna/Desktop/bet

# Apply the fix
./apply-outcome-fix.sh
```

### Option 3: Manual Database Connection

**If you're using Supabase:**

1. Go to your Supabase dashboard
2. Navigate to **SQL Editor**
3. Copy and paste the contents of `fix-outcome-constraint-final.sql`
4. Click **Run**

**If you're using local PostgreSQL:**

```bash
# Connect to your database
psql -h localhost -U your_username -d your_database

# Run the fix
\i fix-outcome-constraint-final.sql
```

## 🧪 Testing the Fix

After applying the fix, test with these outcome names:

### ✅ Should Work Now:
- "Emmerson Mnangagwa"
- "Nelson Chamisa"
- "Douglas Mwonzora"
- "Thokozani Khupe"
- "Other Candidate"
- "yes" (for binary markets)
- "no" (for binary markets)

### ❌ Will Still Fail:
- Empty strings
- Null values
- Names longer than 100 characters

## 🔧 Alternative: Temporary Workaround

If you can't access the database immediately, you can temporarily modify the outcome names to be more generic:

1. **Edit the market outcomes** to use shorter names:
   - "Mnangagwa" instead of "Emmerson Mnangagwa"
   - "Chamisa" instead of "Nelson Chamisa"
   - "Mwonzora" instead of "Douglas Mwonzora"

2. **Or use simple identifiers:**
   - "Candidate A"
   - "Candidate B" 
   - "Candidate C"

## 📋 Files Modified

1. **`fix-outcome-constraint-final.sql`** - Database constraint fix
2. **`apply-outcome-fix.sh`** - Automated fix script
3. **`src/components/markets/UnifiedBetForm.tsx`** - Removed client-side workaround

## 🎯 Expected Result

After applying the fix:

✅ **Multi-outcome betting works** with candidate names  
✅ **Binary betting still works** with yes/no  
✅ **No more "Invalid outcome name" errors**  
✅ **All outcome names are supported** (up to 100 characters)  

## 🚀 Next Steps

1. **Apply the database fix** using one of the methods above
2. **Test multi-outcome betting** with real candidate names
3. **Verify binary markets** still work with yes/no
4. **Check that all market types** display correctly

## 📞 Support

If you're still having issues:

1. **Check database connection** - Make sure your database is running
2. **Verify constraint was applied** - Run the test queries in the SQL script
3. **Check Supabase logs** - Look for any constraint violation errors
4. **Contact database administrator** - If you don't have direct database access

The fix is comprehensive and should resolve the "Invalid outcome name" error permanently! 🎉
