const { createClient } = require('@supabase/supabase-js')

// Initialize Supabase client
const supabase = createClient(
  process.env.NEXT_PUBLIC_SUPABASE_URL || 'your_supabase_url',
  process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY || 'your_supabase_anon_key'
)

// Zimbabwe-related markets data
const zimbabweMarkets = [
  {
    title: "Will Zimbabwe's inflation rate drop below 50% by end of 2024?",
    description: "Zimbabwe has been struggling with hyperinflation for years. Will the government's economic reforms finally bring inflation under control?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.35,
    no_price: 0.65,
    total_volume: 15000
  },
  {
    title: "Will Zimbabwe introduce a new currency in 2024?",
    description: "The Zimbabwean dollar has been unstable. Will the government introduce a new currency or adopt a foreign currency as legal tender?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.42,
    no_price: 0.58,
    total_volume: 12000
  },
  {
    title: "Will Zimbabwe's GDP growth exceed 3% in 2024?",
    description: "Economic recovery in Zimbabwe. Will the country achieve GDP growth above 3% this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.28,
    no_price: 0.72,
    total_volume: 8000
  },
  {
    title: "Will Zimbabwe hold free and fair elections in 2024?",
    description: "Political stability in Zimbabwe. Will the upcoming elections be considered free and fair by international observers?",
    category: "politics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.38,
    no_price: 0.62,
    total_volume: 20000
  },
  {
    title: "Will Zimbabwe's opposition win more than 30% of parliamentary seats?",
    description: "Parliamentary elections in Zimbabwe. Will the opposition parties secure more than 30% of the seats?",
    category: "politics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.45,
    no_price: 0.55,
    total_volume: 18000
  },
  {
    title: "Will Zimbabwe's unemployment rate drop below 20% by end of 2024?",
    description: "Job creation in Zimbabwe. Will the unemployment rate fall below 20% this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.33,
    no_price: 0.67,
    total_volume: 10000
  },
  {
    title: "Will Zimbabwe receive IMF funding in 2024?",
    description: "International financial support. Will Zimbabwe secure funding from the International Monetary Fund this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.52,
    no_price: 0.48,
    total_volume: 25000
  },
  {
    title: "Will Zimbabwe's mining sector contribute more than 15% to GDP?",
    description: "Mining industry performance. Will Zimbabwe's mining sector contribute over 15% to the country's GDP in 2024?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.41,
    no_price: 0.59,
    total_volume: 14000
  },
  {
    title: "Will Zimbabwe's tourism arrivals exceed 2 million in 2024?",
    description: "Tourism recovery in Zimbabwe. Will the country welcome more than 2 million tourists this year?",
    category: "tourism",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.36,
    no_price: 0.64,
    total_volume: 9000
  },
  {
    title: "Will Zimbabwe's agricultural output increase by more than 10%?",
    description: "Agricultural sector growth. Will Zimbabwe's agricultural production increase by over 10% compared to last year?",
    category: "agriculture",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.47,
    no_price: 0.53,
    total_volume: 16000
  },
  {
    title: "Will Zimbabwe's internet penetration exceed 60% by end of 2024?",
    description: "Digital connectivity in Zimbabwe. Will internet penetration reach over 60% of the population?",
    category: "technology",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.39,
    no_price: 0.61,
    total_volume: 7000
  },
  {
    title: "Will Zimbabwe's foreign direct investment exceed $1 billion?",
    description: "Foreign investment in Zimbabwe. Will the country attract more than $1 billion in FDI this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.31,
    no_price: 0.69,
    total_volume: 13000
  },
  {
    title: "Will Zimbabwe's education budget exceed 20% of total budget?",
    description: "Education funding in Zimbabwe. Will the government allocate more than 20% of the budget to education?",
    category: "education",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.44,
    no_price: 0.56,
    total_volume: 11000
  },
  {
    title: "Will Zimbabwe's healthcare system improve significantly in 2024?",
    description: "Healthcare reforms in Zimbabwe. Will there be significant improvements in the healthcare system this year?",
    category: "healthcare",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.37,
    no_price: 0.63,
    total_volume: 12000
  },
  {
    title: "Will Zimbabwe's renewable energy capacity exceed 500MW?",
    description: "Clean energy in Zimbabwe. Will the country's renewable energy capacity reach over 500MW by end of 2024?",
    category: "energy",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.43,
    no_price: 0.57,
    total_volume: 8500
  },
  {
    title: "Will Zimbabwe's manufacturing sector grow by more than 5%?",
    description: "Industrial growth in Zimbabwe. Will the manufacturing sector expand by over 5% this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.40,
    no_price: 0.60,
    total_volume: 15000
  },
  {
    title: "Will Zimbabwe's debt-to-GDP ratio fall below 80%?",
    description: "Debt management in Zimbabwe. Will the country's debt-to-GDP ratio drop below 80% by end of 2024?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.29,
    no_price: 0.71,
    total_volume: 17000
  },
  {
    title: "Will Zimbabwe's exports exceed $4 billion in 2024?",
    description: "Export performance in Zimbabwe. Will the country's exports reach over $4 billion this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.46,
    no_price: 0.54,
    total_volume: 19000
  },
  {
    title: "Will Zimbabwe's urban population exceed 40% of total population?",
    description: "Urbanization in Zimbabwe. Will the urban population reach over 40% of the total population?",
    category: "demographics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.34,
    no_price: 0.66,
    total_volume: 6000
  },
  {
    title: "Will Zimbabwe's life expectancy exceed 65 years?",
    description: "Health outcomes in Zimbabwe. Will the average life expectancy reach over 65 years by end of 2024?",
    category: "healthcare",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.48,
    no_price: 0.52,
    total_volume: 10000
  },
  {
    title: "Will Zimbabwe's literacy rate exceed 90%?",
    description: "Education outcomes in Zimbabwe. Will the literacy rate reach over 90% of the population?",
    category: "education",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.42,
    no_price: 0.58,
    total_volume: 8000
  },
  {
    title: "Will Zimbabwe's corruption perception index improve by more than 10 points?",
    description: "Governance in Zimbabwe. Will the country's corruption perception index improve by over 10 points?",
    category: "politics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.35,
    no_price: 0.65,
    total_volume: 14000
  },
  {
    title: "Will Zimbabwe's stock market index gain more than 20%?",
    description: "Financial markets in Zimbabwe. Will the stock market index increase by over 20% this year?",
    category: "finance",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.38,
    no_price: 0.62,
    total_volume: 16000
  },
  {
    title: "Will Zimbabwe's mobile money transactions exceed $10 billion?",
    description: "Digital payments in Zimbabwe. Will mobile money transactions reach over $10 billion this year?",
    category: "technology",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.51,
    no_price: 0.49,
    total_volume: 12000
  },
  {
    title: "Will Zimbabwe's climate change adaptation funding exceed $100 million?",
    description: "Environmental resilience in Zimbabwe. Will the country secure over $100 million in climate adaptation funding?",
    category: "environment",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.33,
    no_price: 0.67,
    total_volume: 9000
  }
]

async function createZimbabweMarkets() {
  try {
    console.log('Creating Zimbabwe markets...')
    
    // First, get the admin user ID
    const { data: adminUser, error: adminError } = await supabase
      .from('profiles')
      .select('id')
      .eq('is_admin', true)
      .limit(1)
      .single()
    
    if (adminError || !adminUser) {
      console.error('Error finding admin user:', adminError)
      return
    }
    
    console.log('Admin user found:', adminUser.id)
    
    // Create markets one by one
    for (let i = 0; i < zimbabweMarkets.length; i++) {
      const market = zimbabweMarkets[i]
      
      console.log(`Creating market ${i + 1}/25: ${market.title}`)
      
      const { data, error } = await supabase
        .from('markets')
        .insert({
          title: market.title,
          description: market.description,
          category: market.category,
          status: 'active',
          end_date: market.end_date,
          yes_price: market.yes_price,
          no_price: market.no_price,
          total_volume: market.total_volume,
          creator_id: adminUser.id,
          created_at: new Date().toISOString()
        })
        .select()
        .single()
      
      if (error) {
        console.error(`Error creating market ${i + 1}:`, error)
        continue
      }
      
      console.log(`✅ Market ${i + 1} created successfully:`, data.id)
      
      // Add some initial liquidity to make the markets more realistic
      const liquidityAmount = Math.floor(Math.random() * 5000) + 1000 // Random between 1000-6000
      
      // Add liquidity for both YES and NO sides
      const { error: liquidityError } = await supabase
        .from('liquidity_providers')
        .insert([
          {
            user_id: adminUser.id,
            market_id: data.id,
            side: 'yes',
            amount: liquidityAmount,
            shares: liquidityAmount * market.yes_price,
            created_at: new Date().toISOString()
          },
          {
            user_id: adminUser.id,
            market_id: data.id,
            side: 'no',
            amount: liquidityAmount,
            shares: liquidityAmount * market.no_price,
            created_at: new Date().toISOString()
          }
        ])
      
      if (liquidityError) {
        console.error(`Error adding liquidity to market ${i + 1}:`, liquidityError)
      } else {
        console.log(`✅ Added liquidity to market ${i + 1}`)
      }
      
      // Small delay to avoid rate limiting
      await new Promise(resolve => setTimeout(resolve, 100))
    }
    
    console.log('✅ All Zimbabwe markets created successfully!')
    
  } catch (error) {
    console.error('Error creating Zimbabwe markets:', error)
  }
}

// Run the script
createZimbabweMarkets()

