// Debug script to test database connection and check for column issues
const { createClient } = require('@supabase/supabase-js')

// You'll need to replace these with your actual Supabase credentials
const supabaseUrl = process.env.NEXT_PUBLIC_SUPABASE_URL || 'your-supabase-url'
const supabaseKey = process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY || 'your-supabase-key'

const supabase = createClient(supabaseUrl, supabaseKey)

async function debugDatabase() {
  console.log('🔍 Debugging database connection...')
  
  try {
    // Test basic connection
    console.log('1. Testing basic connection...')
    const { data: markets, error: marketsError } = await supabase
      .from('markets')
      .select('id, title, market_type')
      .limit(1)
    
    if (marketsError) {
      console.error('❌ Error fetching markets:', marketsError)
      return
    }
    
    console.log('✅ Basic connection works')
    console.log('Sample market:', markets?.[0])
    
    // Test market_outcomes table
    console.log('\n2. Testing market_outcomes table...')
    const { data: outcomes, error: outcomesError } = await supabase
      .from('market_outcomes')
      .select('id, name, price, pool, volume')
      .limit(1)
    
    if (outcomesError) {
      console.error('❌ Error fetching outcomes:', outcomesError)
      return
    }
    
    console.log('✅ market_outcomes table accessible')
    console.log('Sample outcome:', outcomes?.[0])
    
    // Test the problematic query
    console.log('\n3. Testing the problematic query...')
    const { data: marketWithOutcomes, error: queryError } = await supabase
      .from('markets')
      .select(`
        id,
        market_type,
        shares_sold,
        total_shares,
        yes_pool,
        no_pool,
        total_volume,
        live_volume,
        yes_price,
        no_price,
        updated_at,
        outcomes:market_outcomes(
          id,
          name,
          price,
          pool,
          volume,
          color,
          icon,
          sort_order
        )
      `)
      .eq('status', 'active')
      .limit(1)
    
    if (queryError) {
      console.error('❌ Error with problematic query:', queryError)
      console.error('Error details:', {
        message: queryError.message,
        details: queryError.details,
        hint: queryError.hint,
        code: queryError.code
      })
      return
    }
    
    console.log('✅ Problematic query works!')
    console.log('Market with outcomes:', marketWithOutcomes?.[0])
    
  } catch (error) {
    console.error('❌ Unexpected error:', error)
  }
}

debugDatabase()
