// Debug script to check what market data is being received
const { createClient } = require('@supabase/supabase-js')

const supabaseUrl = process.env.NEXT_PUBLIC_SUPABASE_URL || 'your-supabase-url'
const supabaseKey = process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY || 'your-supabase-key'

const supabase = createClient(supabaseUrl, supabaseKey)

async function debugMarketData() {
  console.log('🔍 Debugging market data...')
  
  try {
    // Get a sample market
    const { data: markets, error: marketsError } = await supabase
      .from('markets')
      .select('id, title, market_type, yes_price, no_price, yes_pool, no_pool, total_volume')
      .eq('status', 'active')
      .limit(1)
    
    if (marketsError) {
      console.error('❌ Error fetching markets:', marketsError)
      return
    }
    
    if (markets && markets.length > 0) {
      const market = markets[0]
      console.log('✅ Sample market found:', market)
      
      // Check if it's a multi-outcome market
      if (market.market_type === 'multi_outcome') {
        console.log('\n🔍 Fetching outcomes for multi-outcome market...')
        const { data: outcomes, error: outcomesError } = await supabase
          .from('market_outcomes')
          .select('id, name, price, pool, volume')
          .eq('market_id', market.id)
        
        if (outcomesError) {
          console.error('❌ Error fetching outcomes:', outcomesError)
        } else {
          console.log('✅ Outcomes found:', outcomes)
        }
      }
      
      // Test share price calculation
      console.log('\n🧮 Testing share price calculation:')
      const yesPrice = market.yes_price || 0.5
      const noPrice = market.no_price || 0.5
      
      const yesSharePrice = Math.max(0.50, Math.min(1.00, yesPrice))
      const noSharePrice = Math.max(0.50, Math.min(1.00, noPrice))
      
      console.log(`YES: probability=${yesPrice}, share_price=${yesSharePrice}`)
      console.log(`NO: probability=${noPrice}, share_price=${noSharePrice}`)
      
      // Test currency formatting
      const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
          style: 'currency',
          currency: 'USD',
          minimumFractionDigits: 2,
          maximumFractionDigits: 2,
        }).format(amount)
      }
      
      console.log(`Formatted YES share price: ${formatCurrency(yesSharePrice)}`)
      console.log(`Formatted NO share price: ${formatCurrency(noSharePrice)}`)
      
    } else {
      console.log('❌ No active markets found')
    }
    
  } catch (error) {
    console.error('❌ Unexpected error:', error)
  }
}

debugMarketData()
