import { useTheme } from '@chakra-ui/system';
import { isBrowser, __DEV__, memoizedGet, breakpoints, isArray, fromEntries, arrayToObjectNotation } from '@chakra-ui/utils';
import * as React from 'react';
import React__default from 'react';
import { useEnvironment } from '@chakra-ui/react-env';

var useSafeLayoutEffect = isBrowser ? React.useLayoutEffect : React.useEffect;
/**
 * React hook that tracks state of a CSS media query
 *
 * @param query the media query to match
 */

function useMediaQuery(query) {
  var env = useEnvironment();
  var queries = Array.isArray(query) ? query : [query];
  var isSupported = isBrowser && "matchMedia" in env.window;

  var _React$useState = React.useState(queries.map(function (query) {
    return isSupported ? !!env.window.matchMedia(query).matches : false;
  })),
      matches = _React$useState[0],
      setMatches = _React$useState[1]; // Specifying matches in the dependency list will cause the event listeners
  // to unload and then load each time the dependency changes. This causes
  // Media Query Events to be missed. The event listeners should only be unloaded
  // when the component unloads.


  useSafeLayoutEffect(function () {
    if (!isSupported) return undefined;
    var mediaQueryList = queries.map(function (query) {
      return env.window.matchMedia(query);
    });
    var listenerList = mediaQueryList.map(function (_, index) {
      var listener = function listener(mqlEvent) {
        var queryIndex = mediaQueryList.findIndex(function (mediaQuery) {
          return mediaQuery.media === mqlEvent.media;
        }); // As the event listener is on the media query list, any time the
        // listener is called, we know there is a change. There's no need
        // to compare the previous matches with current. Using
        // setMatches(matches => {...}) provides access to the current matches
        // state.  Trying to access matches outside the setMatches function
        // would provide data from the the time of instantiation (stale).

        setMatches(function (matches) {
          var currentMatches = matches.map(function (x) {
            return x;
          });
          currentMatches[queryIndex] = mqlEvent.matches;
          return currentMatches;
        });
      }; // Listening to the 'change' event on the Media Query List Object
      // is more performant as the callback is only invoked when a specified
      // media query is matched. Using addEventListener on the window object
      // to listen for the resize event will call the callback on every
      // viewport resize.


      if (typeof mediaQueryList[index].addEventListener === "function") {
        mediaQueryList[index].addEventListener("change", listener);
      } else {
        mediaQueryList[index].addListener(listener);
      }

      return listener;
    });
    return function () {
      mediaQueryList.forEach(function (_, index) {
        if (typeof mediaQueryList[index].removeEventListener === "function") {
          mediaQueryList[index].removeEventListener("change", listenerList[index]);
        } else {
          mediaQueryList[index].removeListener(listenerList[index]);
        }
      });
    };
  }, []);
  return matches;
}

/**
 * Visibility
 *
 * React component to control the visibility of its
 * children based on the current breakpoint
 */
var Visibility = function Visibility(props) {
  var breakpoint = props.breakpoint,
      hide = props.hide,
      children = props.children;

  var _useMediaQuery = useMediaQuery(breakpoint),
      show = _useMediaQuery[0];

  var isVisible = hide ? !show : show;
  var rendered = isVisible ? children : null;
  return rendered;
};

var Hide = function Hide(props) {
  var children = props.children;
  var query = useQuery(props);
  return /*#__PURE__*/React.createElement(Visibility, {
    breakpoint: query,
    hide: true
  }, children);
};

if (__DEV__) {
  Hide.displayName = "Hide";
}

var Show = function Show(props) {
  var children = props.children;
  var query = useQuery(props);
  return /*#__PURE__*/React.createElement(Visibility, {
    breakpoint: query
  }, children);
};

if (__DEV__) {
  Show.displayName = "Show";
}

var getBreakpoint = function getBreakpoint(theme, value) {
  return memoizedGet(theme, "breakpoints." + value, value);
};

function useQuery(props) {
  var _props$breakpoint = props.breakpoint,
      breakpoint = _props$breakpoint === void 0 ? "" : _props$breakpoint,
      below = props.below,
      above = props.above;
  var theme = useTheme();
  var bpBelow = getBreakpoint(theme, below);
  var bpAbove = getBreakpoint(theme, above);
  var query = breakpoint;

  if (bpBelow) {
    query = "(max-width: " + bpBelow + ")";
  } else if (bpAbove) {
    query = "(min-width: " + bpAbove + ")";
  }

  return query;
}

/**
 * React hook used to get the user's animation preference.
 */

function usePrefersReducedMotion() {
  var _useMediaQuery = useMediaQuery("(prefers-reduced-motion: reduce)"),
      prefersReducedMotion = _useMediaQuery[0];

  return prefersReducedMotion;
}
/**
 * React hook for getting the user's color mode preference.
 */

function useColorModePreference() {
  var _useMediaQuery2 = useMediaQuery(["(prefers-color-scheme: light)", "(prefers-color-scheme: dark)"]),
      isLight = _useMediaQuery2[0],
      isDark = _useMediaQuery2[1];

  if (isLight) return "light";
  if (isDark) return "dark";
  return undefined;
}

/**
 * React hook used to get the current responsive media breakpoint.
 *
 * @param [defaultBreakpoint="base"] default breakpoint name
 * (in non-window environments like SSR)
 *
 * For SSR, you can use a package like [is-mobile](https://github.com/kaimallea/isMobile)
 * to get the default breakpoint value from the user-agent
 */

function useBreakpoint(defaultBreakpoint // default value ensures SSR+CSR consistency
) {
  if (defaultBreakpoint === void 0) {
    defaultBreakpoint = "base";
  }

  var _useTheme = useTheme(),
      __breakpoints = _useTheme.__breakpoints;

  var env = useEnvironment();
  var queries = React__default.useMemo(function () {
    var _breakpoints$details;

    return (_breakpoints$details = __breakpoints == null ? void 0 : __breakpoints.details.map(function (_ref) {
      var minMaxQuery = _ref.minMaxQuery,
          breakpoint = _ref.breakpoint;
      return {
        breakpoint: breakpoint,
        query: minMaxQuery.replace("@media screen and ", "")
      };
    })) != null ? _breakpoints$details : [];
  }, [__breakpoints]);

  var _React$useState = React__default.useState(function () {
    if (defaultBreakpoint) {
      // use default breakpoint to ensure render consistency in SSR + CSR environments
      // => first render on the client has to match the render on the server
      var fallbackBreakpointDetail = queries.find(function (_ref2) {
        var breakpoint = _ref2.breakpoint;
        return breakpoint === defaultBreakpoint;
      });

      if (fallbackBreakpointDetail) {
        return fallbackBreakpointDetail.breakpoint;
      }
    }

    if (env.window.matchMedia) {
      // set correct breakpoint on first render if no default breakpoint was provided
      var matchingBreakpointDetail = queries.find(function (_ref3) {
        var query = _ref3.query;
        return env.window.matchMedia(query).matches;
      });

      if (matchingBreakpointDetail) {
        return matchingBreakpointDetail.breakpoint;
      }
    }

    return undefined;
  }),
      currentBreakpoint = _React$useState[0],
      setCurrentBreakpoint = _React$useState[1];

  React__default.useEffect(function () {
    var allUnregisterFns = queries.map(function (_ref4) {
      var breakpoint = _ref4.breakpoint,
          query = _ref4.query;
      var mediaQueryList = env.window.matchMedia(query);

      if (mediaQueryList.matches) {
        setCurrentBreakpoint(breakpoint);
      }

      var handleChange = function handleChange(ev) {
        if (ev.matches) {
          setCurrentBreakpoint(breakpoint);
        }
      }; // add media query listener


      if (typeof mediaQueryList.addEventListener === "function") {
        mediaQueryList.addEventListener("change", handleChange);
      } else {
        mediaQueryList.addListener(handleChange);
      } // return unregister fn


      return function () {
        if (typeof mediaQueryList.removeEventListener === "function") {
          mediaQueryList.removeEventListener("change", handleChange);
        } else {
          mediaQueryList.removeListener(handleChange);
        }
      };
    });
    return function () {
      allUnregisterFns.forEach(function (unregister) {
        return unregister();
      });
    };
  }, [queries, __breakpoints, env.window]);
  return currentBreakpoint;
}

function getClosestValue(values, breakpoint, breakpoints$1) {
  if (breakpoints$1 === void 0) {
    breakpoints$1 = breakpoints;
  }

  var index = Object.keys(values).indexOf(breakpoint);

  if (index !== -1) {
    return values[breakpoint];
  }

  var stopIndex = breakpoints$1.indexOf(breakpoint);

  while (stopIndex >= 0) {
    var key = breakpoints$1[stopIndex];

    if (values[key] != null) {
      index = stopIndex;
      break;
    }

    stopIndex -= 1;
  }

  if (index !== -1) {
    var _key = breakpoints$1[index];
    return values[_key];
  }

  return undefined;
}

/**
 * React hook for getting the value for the current breakpoint from the
 * provided responsive values object.
 *
 * @param values
 * @param [defaultBreakpoint] default breakpoint name
 * (in non-window environments like SSR)
 *
 * For SSR, you can use a package like [is-mobile](https://github.com/kaimallea/isMobile)
 * to get the default breakpoint value from the user-agent
 *
 * @example
 * const width = useBreakpointValue({ base: '150px', md: '250px' })
 */

function useBreakpointValue(values, defaultBreakpoint) {
  var _theme$__breakpoints;

  var breakpoint = useBreakpoint(defaultBreakpoint);
  var theme = useTheme();
  if (!breakpoint) return undefined;
  /**
   * Get the sorted breakpoint keys from the provided breakpoints
   */

  var breakpoints = Array.from(((_theme$__breakpoints = theme.__breakpoints) == null ? void 0 : _theme$__breakpoints.keys) || []);
  var obj = isArray(values) ? fromEntries(Object.entries(arrayToObjectNotation(values, breakpoints)).map(function (_ref) {
    var key = _ref[0],
        value = _ref[1];
    return [key, value];
  })) : values;
  return getClosestValue(obj, breakpoint, breakpoints);
}

export { Hide, Show, useBreakpoint, useBreakpointValue, useColorModePreference, useMediaQuery, usePrefersReducedMotion, useQuery };
