import { InitResponse } from './classes/InitResponse';
import Payment from './classes/payment';
import { StatusResponse } from './classes/StatusResponse';
export default class Paynow {
    integrationId: string;
    integrationKey: string;
    resultUrl?: string | undefined;
    returnUrl?: string | undefined;
    constructor(integrationId: string, integrationKey: string, resultUrl?: string | undefined, returnUrl?: string | undefined);
    /**
     * Update the return url
     * @param url {String}
     */
    setReturnUrl: (url: string) => void;
    /**
     * Update the result url
     * @param url {String}
     */
    setResultUrl(url: string): void;
    /**
     * Send a payment to paynow
     * @param payment
     */
    send(payment: Payment): Promise<InitResponse | null>;
    /**
     * Send a mobile money payment to paynow
     * @param payment
     */
    sendMobile(payment: Payment, phone: string, method: string): Promise<InitResponse | null>;
    /**
     * Create a new Paynow payment
     * @param {String} reference This is the unique reference of the transaction
     * @param {String} authEmail This is the email address of the person making payment. Required for mobile transactions
     * @returns {Payment}
     */
    createPayment(reference: string, authEmail?: string): Payment;
    /**
     * Throw an exception with the given message
     * @param message*
     * @returns void
     */
    fail(message: string): Error;
    /**
     * Initialize a new transaction with PayNow
     * @param payment
     * @returns {PromiseLike<InitResponse> | Promise<InitResponse>}
     */
    init(payment: Payment): Promise<InitResponse | null>;
    /**
     * Initialize a new mobile transaction with PayNow
     * @param {Payment} payment
     * @returns {PromiseLike<InitResponse> | Promise<InitResponse>} the response from the initiation of the transaction
     */
    initMobile(payment: Payment, phone: string, method: string): Promise<InitResponse | null>;
    /**
     * Validates whether an email address is valid or not
     *
     * @param {string} emailAddress The email address to validate
     *
     * @returns {boolean} A value indicating an email is valid or not
     */
    isValidEmail(emailAddress: string): boolean;
    /**
     * Parses the response from Paynow
     * @param response
     * @returns {InitResponse}
     */
    parse(response: Response): InitResponse | null;
    /**
     * Creates a SHA512 hash of the transactions
     * @param values
     * @param integrationKey
     * @returns {string}
     */
    generateHash(values: {
        [key: string]: string;
    }, integrationKey: String): any;
    /**
     * Verify hashes at all interactions with server
     * @param {*} values
     */
    verifyHash(values: {
        [key: string]: string;
    }): boolean;
    /**
     * URL encodes the given string
     * @param str {String}
     * @returns {String}
     */
    urlEncode(url: string): string;
    /**
     * URL decodes the given string
     * @param str {String}
     * @returns {String}
     */
    urlDecode(url: string): string;
    /**
     * Parse responses from Paynow
     * @param queryString
     */
    parseQuery(queryString: string): {
        [key: string]: string;
    };
    /**
     * Build up a payment into the format required by Paynow
     * @param payment
     * @returns {{resulturl: String, returnurl: String, reference: String, amount: number, id: String, additionalinfo: String, authemail: String, status: String}}
     */
    build(payment: Payment): {
        [key: string]: string;
    };
    /**
     * Build up a mobile payment into the format required by Paynow
     * @param payment
     * @returns {{resulturl: String, returnurl: String, reference: String, amount: number, id: String, additionalinfo: String, authemail: String, status: String}}
     */
    buildMobile(payment: Payment, phone: string, method: string): Error | {
        [key: string]: string;
    };
    /**
     * Check the status of a transaction
     * @param url
     * @returns {PromiseLike<InitResponse> | Promise<InitResponse>}
     */
    pollTransaction(url: string): Promise<InitResponse | null>;
    /**
     * Parses the response from Paynow
     * @param response
     * @returns {StatusResponse}
     */
    parseStatusUpdate(response: any): StatusResponse;
    /**
     * Validates an outgoing request before sending it to Paynow (data sanity checks)
     * @param payment
     */
    validate(payment: Payment): void;
}
