import { Modal, ModalOverlay, ModalContent, ModalHeader, ModalCloseButton, ModalBody, Box, Image, Divider, Flex, Heading, Text, FormControl, FormLabel, Input, FormHelperText, Button, ChakraProvider } from '@chakra-ui/react';
import React__default, { useState, useEffect, createContext, useContext, createElement, Fragment } from 'react';
import axios from 'axios';
import qs from 'qs';

/*
 * Taken from https://github.com/paynow/Paynow-NodeJS-SDK/blob/master/src/constants.ts
 */

/*
 * Success response from Paynow
 */
var RESPONSE_OK = 'ok';
/**
 * Error response from Paynow
 */

var RESPONSE_ERROR = 'error';
/**
 * API endpoint for initiating normal web-based transactions
 */

var URL_INITIATE_TRANSACTION = '/interface/initiatetransaction';
/**
 * API endpoint for initiating mobile based transactions
 */

var URL_INITIATE_MOBILE_TRANSACTION = '/interface/remotetransaction';

/**
 *
 * @property {boolean} success - indicates if initiate request was successful or not.
 * @property {boolean} hasRedirect - indicates if the response has a URL to redirect to.
 * @property {String} redirectUrl - the URL the user should be redirected to so they can make a payment.
 * @property {String} error - error message sent from Paynow (if any).
 * @property {String} pollUrl  - pollUrl sent from Paynow that can be used to check transaction status.
 * @property {String} instructions - instructions for USSD push for customers to dial incase of mobile money payments.
 * @property {String} status - status from Paynow.
 *
 * @param data - data from the Response.
 *
 */
var InitResponse = function InitResponse(data) {
  this.status = data.status.toLowerCase();
  this.success = this.status === RESPONSE_OK;
  this.hasRedirect = typeof data.browserurl !== 'undefined';

  if (!this.success) {
    this.error = data.error;
  } else {
    this.pollUrl = data.pollurl;

    if (this.hasRedirect) {
      this.redirectUrl = data.browserurl;
    }

    if (typeof data.instructions !== 'undefined') {
      this.instructions = data.instructions;
    }
  }
};

/**
 *  @param title the name of the cart item
 *
 * @param amount the cost of a single unit of the item
 *
 * @param quantity the number of units of the item
 */
var CartItem = function CartItem(title, amount, quantity) {
  if (quantity === void 0) {
    quantity = 1;
  }

  this.title = title;
  this.amount = amount;
  this.quantity = quantity;
}; //#endregion
//#region

var Cart = /*#__PURE__*/function () {
  function Cart(_items) {
    var _this = this;

    this.items = [];

    if (_items) {
      _items.forEach(function (thing) {
        _this.items.push(thing);
      });
    }
  }

  var _proto = Cart.prototype;

  _proto.length = function length() {
    return this.items.length;
  };

  _proto.add = function add(item) {
    this.items.push(item);
    return this.items.length;
  };

  _proto.getTotal = function getTotal() {
    var cartTotal = 0;
    this.items.forEach(function (item) {
      cartTotal += item.amount * item.quantity;
    });
    return cartTotal;
  };

  _proto.summary = function summary() {
    var summary = '';
    this.items.forEach(function (item) {
      summary = summary.concat(item.title + ', ');
    }); // console.log(summary, this.items);

    return summary;
  };

  return Cart;
}();

/**
 * @param reference  unique identifier for the transaction.
 * @param authEmail customer's email address.
 * @param items items inthe user's Cart
 */

var Payment = /*#__PURE__*/function () {
  function Payment(reference, authEmail, items) {
    if (items === void 0) {
      items = new Cart();
    }

    this.reference = reference;
    this.authEmail = authEmail;
    this.items = items;
  }
  /**
   * Adds an item to the 'shopping cart'
   * @param title
   * @param amount
   */


  var _proto = Payment.prototype;

  _proto.add = function add(title, amount, quantity) {
    this.items.add(new CartItem(title, amount, quantity));
    return this;
  };

  _proto.info = function info() {
    return this.items.summary();
  }
  /**
   * Get the total of the items in the cart
   * @returns {*|number}
   */
  ;

  _proto.total = function total() {
    return this.items.getTotal();
  };

  return Payment;
}();

/**
 *
 * @property {String} reference - merchant transaction reference .
 * @property {String} amount - original amount for the transaction.
 * @property {String} paynowReference  - the Paynow transaction reference.
 * @property {String} pollUrl - the URL on Paynow the merchant can poll to confirm the transaction’s status.
 * @property {String} status - transaction status returned from paynow.
 * @property {String} error - error message sent from Paynow  (if any).
 *
 * @param data data from the status response
 */
var StatusResponse = function StatusResponse(data) {
  if (data.status.toLowerCase() === RESPONSE_ERROR) {
    this.error = data.error;
  } else {
    this.reference = data.reference;
    this.amount = data.amount;
    this.paynowReference = data.paynowreference;
    this.pollUrl = data.pollurl;
    this.status = data.status;
  }
};

var paynowAxiosInstance = /*#__PURE__*/axios.create({
  baseURL: 'https://cors-anywhere.herokuapp.com/https://www.paynow.co.zw'
});

var Paynow = /*#__PURE__*/function () {
  function Paynow(integrationId, integrationKey, resultUrl, returnUrl) {
    var _this = this;

    this.integrationId = integrationId;
    this.integrationKey = integrationKey;
    this.resultUrl = resultUrl;
    this.returnUrl = returnUrl;
    /**
     * Update the return url
     * @param url {String}
     */

    this.setReturnUrl = function (url) {
      _this.returnUrl = url;
    };
  }
  /**
   * Update the result url
   * @param url {String}
   */


  var _proto = Paynow.prototype;

  _proto.setResultUrl = function setResultUrl(url) {
    this.resultUrl = url;
  }
  /**
   * Send a payment to paynow
   * @param payment
   */
  ;

  _proto.send = function send(payment) {
    return this.init(payment);
  }
  /**
   * Send a mobile money payment to paynow
   * @param payment
   */
  ;

  _proto.sendMobile = function sendMobile(payment, phone, method) {
    return this.initMobile(payment, phone, method);
  }
  /**
   * Create a new Paynow payment
   * @param {String} reference This is the unique reference of the transaction
   * @param {String} authEmail This is the email address of the person making payment. Required for mobile transactions
   * @returns {Payment}
   */
  ;

  _proto.createPayment = function createPayment(reference, authEmail) {
    return new Payment(reference, authEmail);
  }
  /**
   * Throw an exception with the given message
   * @param message*
   * @returns void
   */
  ;

  _proto.fail = function fail(message) {
    throw new Error(message);
  }
  /**
   * Initialize a new transaction with PayNow
   * @param payment
   * @returns {PromiseLike<InitResponse> | Promise<InitResponse>}
   */
  ;

  _proto.init = function init(payment) {
    var _this2 = this;

    this.validate(payment);
    var data = this.build(payment);
    return paynowAxiosInstance.post(URL_INITIATE_TRANSACTION, qs.stringify(data), {
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        mode: 'no-cors'
      }
    }).then(function (res) {
      return _this2.parse(res.data);
    })["catch"](function (err) {
      console.error(err);
      throw 'An error occured while initiating the transaction';
    });
  }
  /**
   * Initialize a new mobile transaction with PayNow
   * @param {Payment} payment
   * @returns {PromiseLike<InitResponse> | Promise<InitResponse>} the response from the initiation of the transaction
   */
  ;

  _proto.initMobile = function initMobile(payment, phone, method) {
    var _payment$authEmail,
        _this3 = this;

    this.validate(payment);
    if (!this.isValidEmail((_payment$authEmail = payment.authEmail) != null ? _payment$authEmail : '')) this.fail('Invalid email. Please ensure that you pass a valid email address when initiating a mobile payment');
    var data = this.buildMobile(payment, phone, method);
    return paynowAxiosInstance.post(URL_INITIATE_MOBILE_TRANSACTION, qs.stringify(data), {
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        mode: 'no-cors'
      }
    }).then(function (res) {
      return _this3.parse(res.data);
    })["catch"](function (err) {
      var _err$message;

      console.error(err);
      throw (_err$message = err.message) != null ? _err$message : 'An error occured while initiating the transaction';
    });
  }
  /**
   * Validates whether an email address is valid or not
   *
   * @param {string} emailAddress The email address to validate
   *
   * @returns {boolean} A value indicating an email is valid or not
   */
  ;

  _proto.isValidEmail = function isValidEmail(emailAddress) {
    if (!emailAddress || emailAddress.length === 0) return false;
    return /^\w+([\.-]?\w+)*@\w+([\.-]?\w+)*(\.\w{2,3})+$/.test(emailAddress);
  }
  /**
   * Parses the response from Paynow
   * @param response
   * @returns {InitResponse}
   */
  ;

  _proto.parse = function parse(response) {
    if (typeof response === 'undefined') {
      return null;
    }

    if (response) {
      var parsedResponseURL = this.parseQuery(response);

      if (parsedResponseURL.error === 'Insufficient balance') {
        throw new Error('Insufficient balance');
      }

      if (parsedResponseURL.status.toString() !== 'error' && !this.verifyHash(parsedResponseURL)) {
        throw new Error('Hashes do not match!');
      }

      return new InitResponse(parsedResponseURL);
    } else {
      throw new Error('An unknown error occurred');
    }
  }
  /**
   * Creates a SHA512 hash of the transactions
   * @param values
   * @param integrationKey
   * @returns {string}
   */
  ;

  _proto.generateHash = function generateHash(values, integrationKey) {
    var sha512 = require('js-sha512').sha512;

    var string = '';

    for (var _i = 0, _Object$keys = Object.keys(values); _i < _Object$keys.length; _i++) {
      var key = _Object$keys[_i];

      if (key !== 'hash') {
        string += values[key];
      }
    }

    string += integrationKey.toLowerCase();
    return sha512(string).toUpperCase();
  }
  /**
   * Verify hashes at all interactions with server
   * @param {*} values
   */
  ;

  _proto.verifyHash = function verifyHash(values) {
    if (typeof values['hash'] === 'undefined') {
      return false;
    } else {
      return values['hash'] === this.generateHash(values, this.integrationKey);
    }
  }
  /**
   * URL encodes the given string
   * @param str {String}
   * @returns {String}
   */
  ;

  _proto.urlEncode = function urlEncode(url) {
    return encodeURI(url);
  }
  /**
   * URL decodes the given string
   * @param str {String}
   * @returns {String}
   */
  ;

  _proto.urlDecode = function urlDecode(url) {
    return decodeURIComponent((url + '').replace(/%(?![\da-f]{2})/gi, function () {
      return '%25';
    }).replace(/\+/g, '%20'));
  }
  /**
   * Parse responses from Paynow
   * @param queryString
   */
  ;

  _proto.parseQuery = function parseQuery(queryString) {
    var query = {};
    var pairs = (queryString[0] === '?' ? queryString.substr(1) : queryString).split('&');

    for (var i = 0; i < pairs.length; i++) {
      var pair = pairs[i].split('=');
      query[this.urlDecode(pair[0])] = this.urlDecode(pair[1] || '');
    } // if(!this.verifyHash(query))
    //         throw new Error("Hash mismatch");


    return query;
  }
  /**
   * Build up a payment into the format required by Paynow
   * @param payment
   * @returns {{resulturl: String, returnurl: String, reference: String, amount: number, id: String, additionalinfo: String, authemail: String, status: String}}
   */
  ;

  _proto.build = function build(payment) {
    var _this$resultUrl, _this$returnUrl;

    var data = {
      resulturl: (_this$resultUrl = this.resultUrl) != null ? _this$resultUrl : '',
      returnurl: (_this$returnUrl = this.returnUrl) != null ? _this$returnUrl : '',
      reference: payment.reference,
      amount: payment.total().toString(),
      id: this.integrationId,
      additionalinfo: payment.info(),
      authemail: typeof payment.authEmail === 'undefined' ? '' : payment.authEmail,
      status: 'Message'
    };

    for (var _i2 = 0, _Object$keys2 = Object.keys(data); _i2 < _Object$keys2.length; _i2++) {
      var key = _Object$keys2[_i2];
      if (key === 'hash') continue;
      data[key] = this.urlEncode(data[key]);
    }

    data['hash'] = this.generateHash(data, this.integrationKey);
    return data;
  }
  /**
   * Build up a mobile payment into the format required by Paynow
   * @param payment
   * @returns {{resulturl: String, returnurl: String, reference: String, amount: number, id: String, additionalinfo: String, authemail: String, status: String}}
   */
  ;

  _proto.buildMobile = function buildMobile(payment, phone, method) {
    var _this$resultUrl2, _this$returnUrl2, _payment$authEmail2;

    var data = {
      resulturl: (_this$resultUrl2 = this.resultUrl) != null ? _this$resultUrl2 : '',
      returnurl: (_this$returnUrl2 = this.returnUrl) != null ? _this$returnUrl2 : '',
      reference: payment.reference,
      amount: payment.total().toString(),
      id: this.integrationId,
      additionalinfo: payment.info(),
      authemail: (_payment$authEmail2 = payment.authEmail) != null ? _payment$authEmail2 : '',
      phone: phone,
      method: method,
      status: 'Message'
    };

    for (var _i3 = 0, _Object$keys3 = Object.keys(data); _i3 < _Object$keys3.length; _i3++) {
      var key = _Object$keys3[_i3];
      if (key === 'hash') continue;
      data[key] = this.urlEncode(data[key]);
    }

    data['hash'] = this.generateHash(data, this.integrationKey);
    return data;
  }
  /**
   * Check the status of a transaction
   * @param url
   * @returns {PromiseLike<InitResponse> | Promise<InitResponse>}
   */
  ;

  _proto.pollTransaction = function pollTransaction(url) {
    var _this4 = this;

    return axios.post("https://cors-anywhere.herokuapp.com/" + url).then(function (res) {
      return _this4.parse(res.data);
    });
  }
  /**
   * Parses the response from Paynow
   * @param response
   * @returns {StatusResponse}
   */
  ;

  _proto.parseStatusUpdate = function parseStatusUpdate(response) {
    if (response.length > 0) {
      response = this.parseQuery(response);

      if (!this.verifyHash(response)) {
        throw new Error('Hashes do not match!');
      }

      return new StatusResponse(response);
    } else {
      throw new Error('An unknown error occurred');
    }
  }
  /**
   * Validates an outgoing request before sending it to Paynow (data sanity checks)
   * @param payment
   */
  ;

  _proto.validate = function validate(payment) {
    if (payment.items.length() <= 0) {
      this.fail('You need to have at least one item in cart');
    }

    if (payment.total() <= 0) {
      this.fail('The total should be greater than zero');
    }
  };

  return Paynow;
}();

var PaynowContext = /*#__PURE__*/createContext({
  paynow: null,
  setData: function setData() {}
});
var PaynowContextProvider = function PaynowContextProvider(_ref) {
  var children = _ref.children;

  var _useState = useState({
    integration_id: '',
    integration_key: '',
    result_url: '',
    return_url: ''
  }),
      data = _useState[0],
      setData = _useState[1];

  var _useState2 = useState(null),
      paynow = _useState2[0],
      setPaynow = _useState2[1];

  var updateValues = function updateValues(data) {
    setData(data);
  };

  useEffect(function () {
    var integration_id = data.integration_id,
        integration_key = data.integration_key,
        result_url = data.result_url,
        return_url = data.return_url;

    if (integration_id && integration_key && result_url && return_url) {
      setPaynow(new Paynow(integration_id, integration_key, result_url, return_url));
    }
  }, [data]);
  return React__default.createElement(PaynowContext.Provider, {
    value: {
      paynow: paynow,
      setData: updateValues
    }
  }, children);
};

var getMobileNetworkForNumber = function getMobileNetworkForNumber(phone) {
  var prefix = phone.substring(0, 3);
  if (prefix === '077') return 'ecocash';
  if (prefix === '078') return 'ecocash';
  return 'onemoney';
};

function PaymentModal(_ref) {
  var items = _ref.items,
      label = _ref.label,
      paymentMode = _ref.paymentMode,
      isOpen = _ref.isOpen,
      onClose = _ref.onClose;

  var _useContext = useContext(PaynowContext),
      paynow = _useContext.paynow;

  var _useState = useState(items),
      myItems = _useState[0];

  var _useState2 = useState(false),
      loading = _useState2[0],
      setLoading = _useState2[1];

  var _useState3 = useState(''),
      error = _useState3[0],
      setError = _useState3[1]; // Get total of items in cart


  var totalAmount = myItems.reduce(function (acc, item) {
    return acc + item.amount * item.quantity;
  }, 0);
  var isMobilePayment = paymentMode === 'mobile';
  useEffect(function () {}, [items, label, paymentMode]);
  /*
  const subtract = (index: number) => {
    if (myItems[index].quantity > 1) {
      myItems[index].quantity -= 1;
      setMyItems(myItems);
    }
  };
      const add = (index: number) => {
    myItems[index].quantity += 1;
    setMyItems(myItems);
  };
  */

  var submitMobilePayment = function submitMobilePayment(e) {
    e.preventDefault();
    var phone = e.target.phone.value;
    var email = e.target.email.value;

    if (phone.length < 10) {
      setError('Please enter a valid phone number');
      return;
    }

    var paidOrError = false;
    setLoading(true);
    var payment = paynow == null ? void 0 : paynow.createPayment(label, email);
    items.map(function (item) {
      return payment == null ? void 0 : payment.add(item.title, item.amount, item.quantity);
    });
    paynow == null ? void 0 : paynow.sendMobile(payment, phone, getMobileNetworkForNumber(phone)).then(function (data) {
      console.log(data);
      var pollInterval = setInterval(function () {
        paynow.pollTransaction(data == null ? void 0 : data.pollUrl).then(function (data) {
          if ((data == null ? void 0 : data.status) === 'paid' && !paidOrError) {
            setLoading(false);
            paidOrError = true;
            closeModal({
              paid: true,
              phone: phone,
              email: email
            });
            return clearInterval(pollInterval);
          } else if ((data == null ? void 0 : data.status) === 'cancelled' && !paidOrError) {
            setLoading(false);
            setError('The payment has been cancelled by the user. Please try again.');
            paidOrError = true;
            closeModal({
              paid: false,
              phone: phone,
              email: email
            });
            return clearInterval(pollInterval);
          }
        })["catch"](function (err) {
          setError(err.message);
        });
      }, 1000);
    })["catch"](function (err) {
      setLoading(false);
      setError(err);
    });
  };

  var submitWebPayment = function submitWebPayment(e) {
    e.preventDefault();
    setLoading(true);
    var payment = paynow == null ? void 0 : paynow.createPayment(label);
    items.map(function (item) {
      return payment == null ? void 0 : payment.add(item.title, item.amount, item.quantity);
    });
    paynow == null ? void 0 : paynow.send(payment).then(function (data) {
      window.location.href = data.redirectUrl;
    })["catch"](function (err) {
      return setError(err != null ? err : 'A network error occured');
    });
  };

  var closeModal = function closeModal(data) {
    if (data) {
      onClose(data);
    } else {
      onClose({
        paid: false,
        info: 'Modal closed by user'
      });
    }
  };

  return React__default.createElement(Modal, {
    size: "xl",
    isOpen: isOpen,
    onClose: closeModal
  }, React__default.createElement(ModalOverlay, null), React__default.createElement(ModalContent, null, React__default.createElement(ModalHeader, null, label), React__default.createElement(ModalCloseButton, null), React__default.createElement(ModalBody, null, React__default.createElement(Box, {
    mt: 2,
    mb: 8
  }, React__default.createElement(Image, {
    mb: 4,
    src: "https://developers.paynow.co.zw/docs/assets/Paynow%20Badge-vector-hires%20DARK.svg",
    alt: "Paynow",
    objectFit: 'fill'
  }), React__default.createElement(Divider, {
    my: 2
  }), React__default.createElement(Box, {
    maxH: 200,
    overflowY: 'auto',
    experimental_spaceY: 2
  }, myItems.map(function (item) {
    return React__default.createElement(Flex, {
      alignItems: 'center',
      key: item.title,
      experimental_spaceX: 4
    }, React__default.createElement(Box, {
      w: 14,
      h: 'auto'
    }, React__default.createElement(Image, {
      rounded: 'xl',
      src: item.image,
      alt: item.title,
      fallbackSrc: "https://via.placeholder.com/150"
    })), React__default.createElement(Box, {
      flex: 1
    }, React__default.createElement(Flex, {
      justifyContent: 'space-between',
      w: "full"
    }, React__default.createElement(Heading, {
      mb: 1,
      fontSize: 'md',
      isTruncated: true
    }, item.title), React__default.createElement(Heading, {
      fontWeight: 'medium',
      fontSize: 'md',
      flex: 1,
      pl: 2,
      textAlign: 'right'
    }, Intl.NumberFormat('en-us', {
      style: 'currency',
      currency: 'USD'
    }).format(item.amount))), React__default.createElement(Flex, {
      alignItems: 'center',
      experimental_spaceX: 2
    }, React__default.createElement(Text, null, "Quantity: ", item.quantity))));
  })), React__default.createElement(Divider, {
    py: 2
  }), isMobilePayment ? React__default.createElement(Box, {
    py: 2,
    mt: 4
  }, React__default.createElement("form", {
    onSubmit: submitMobilePayment
  }, React__default.createElement(FormControl, {
    py: 2
  }, React__default.createElement(FormLabel, null, "Mobile number"), React__default.createElement(Input, {
    type: "number",
    maxLength: 10,
    minLength: 10,
    autoComplete: "off",
    name: "phone",
    variant: 'flushed',
    placeholder: "0777777777",
    required: true
  }), error && React__default.createElement(FormHelperText, {
    color: 'red.500'
  }, error)), React__default.createElement(FormControl, {
    py: 2
  }, React__default.createElement(FormLabel, null, "Email address"), React__default.createElement(Input, {
    type: "email",
    autoComplete: "off",
    name: "email",
    placeholder: "you@example.com",
    variant: 'flushed',
    required: true
  })), React__default.createElement(Box, {
    mt: 4,
    experimental_spaceY: 2
  }, React__default.createElement(FormControl, null, React__default.createElement(Button, {
    isFullWidth: true,
    colorScheme: "blue",
    mr: 3,
    type: "submit",
    isLoading: loading
  }, "Pay RTGS$ " + Intl.NumberFormat('en-us', {
    style: 'currency',
    currency: 'USD'
  }).format(totalAmount))), React__default.createElement(FormControl, null, React__default.createElement(Button, {
    onClick: onClose,
    isFullWidth: true,
    variant: "outline"
  }, "Cancel"))))) : React__default.createElement(Box, {
    experimental_spaceY: 4
  }, error && React__default.createElement(Text, {
    color: "red.500"
  }, error), React__default.createElement(Button, {
    isFullWidth: true,
    colorScheme: "blue",
    mr: 3,
    isLoading: loading,
    onClick: submitWebPayment
  }, 'Checkout'), React__default.createElement(Button, {
    onClick: onClose,
    isFullWidth: true,
    variant: "outline"
  }, "Cancel"))))));
}

function PaynowReactWrapper(props) {
  return createElement(PaynowContextProvider, null, createElement(PaynowWrapperConsumer, Object.assign({}, props), props.children));
}

var PaynowWrapperConsumer = function PaynowWrapperConsumer(props) {
  var _React$useContext = useContext(PaynowContext),
      setData = _React$useContext.setData;

  useEffect(function () {
    var integration_id = props.integration_id,
        integration_key = props.integration_key,
        result_url = props.result_url,
        return_url = props.return_url;
    setData({
      integration_id: integration_id,
      integration_key: integration_key,
      result_url: result_url,
      return_url: return_url
    });
  }, []);
  return createElement(Fragment, null, props.children);
};

var PaynowPayment = function PaynowPayment(props) {
  return createElement(ChakraProvider, null, createElement(PaymentModal, Object.assign({}, props)));
};

export default PaynowReactWrapper;
export { PaynowPayment };
//# sourceMappingURL=paynow-react.esm.js.map
