#!/usr/bin/env node

/**
 * Script to check and process market refunds
 * This should be run periodically (e.g., daily) via cron job
 */

const { createClient } = require('@supabase/supabase-js')
require('dotenv').config({ path: '.env.local' })

const supabaseUrl = process.env.NEXT_PUBLIC_SUPABASE_URL
const supabaseServiceKey = process.env.SUPABASE_SERVICE_ROLE_KEY

if (!supabaseUrl || !supabaseServiceKey) {
  console.error('Missing required environment variables')
  process.exit(1)
}

const supabase = createClient(supabaseUrl, supabaseServiceKey)

async function checkMarketRefunds() {
  try {
    console.log('Checking market refunds...')
    
    // Call the refund check function
    const { data, error } = await supabase.rpc('check_market_refunds')
    
    if (error) {
      console.error('Error checking refunds:', error)
      return
    }
    
    console.log('Refund check completed successfully')
    
    // Get statistics about refunded markets
    const { data: refundedMarkets, error: statsError } = await supabase
      .from('markets')
      .select('id, title, creator_id, refund_amount, updated_at')
      .eq('status', 'refunded')
      .gte('updated_at', new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString()) // Last 24 hours
    
    if (statsError) {
      console.error('Error fetching refund statistics:', statsError)
    } else if (refundedMarkets && refundedMarkets.length > 0) {
      console.log(`Processed ${refundedMarkets.length} refunds in the last 24 hours:`)
      refundedMarkets.forEach(market => {
        console.log(`- Market: "${market.title}" (${market.id}) - Refunded: $${market.refund_amount}`)
      })
    } else {
      console.log('No refunds processed in the last 24 hours')
    }
    
  } catch (error) {
    console.error('Unexpected error:', error)
  }
}

// Run the refund check
checkMarketRefunds()
  .then(() => {
    console.log('Refund check script completed')
    process.exit(0)
  })
  .catch((error) => {
    console.error('Script failed:', error)
    process.exit(1)
  })
