#!/usr/bin/env node

/**
 * Script to create an admin user for the prediction market platform
 * 
 * Usage:
 * 1. Set your Supabase credentials in the environment variables
 * 2. Run: node scripts/create-admin-user.js
 * 
 * Environment variables needed:
 * - SUPABASE_URL
 * - SUPABASE_SERVICE_ROLE_KEY (not anon key - needs service role)
 */

const { createClient } = require('@supabase/supabase-js')

// Configuration
const ADMIN_EMAIL = 'admin@predictionmarket.com'
const ADMIN_PASSWORD = 'AdminPassword123!'
const ADMIN_USERNAME = 'admin'
const ADMIN_FIRST_NAME = 'Platform'
const ADMIN_LAST_NAME = 'Administrator'

async function createAdminUser() {
  // Check for required environment variables
  if (!process.env.SUPABASE_URL || !process.env.SUPABASE_SERVICE_ROLE_KEY) {
    console.error('❌ Missing required environment variables:')
    console.error('   SUPABASE_URL')
    console.error('   SUPABASE_SERVICE_ROLE_KEY')
    console.error('')
    console.error('Please set these in your .env.local file or as environment variables.')
    process.exit(1)
  }

  // Create Supabase client with service role key
  const supabase = createClient(
    process.env.SUPABASE_URL,
    process.env.SUPABASE_SERVICE_ROLE_KEY,
    {
      auth: {
        autoRefreshToken: false,
        persistSession: false
      }
    }
  )

  try {
    console.log('🚀 Creating admin user...')
    console.log(`📧 Email: ${ADMIN_EMAIL}`)
    console.log(`👤 Username: ${ADMIN_USERNAME}`)
    console.log(`👤 Name: ${ADMIN_FIRST_NAME} ${ADMIN_LAST_NAME}`)

    // Step 1: Create auth user
    console.log('\n1️⃣ Creating authentication user...')
    const { data: authData, error: authError } = await supabase.auth.admin.createUser({
      email: ADMIN_EMAIL,
      password: ADMIN_PASSWORD,
      email_confirm: true, // Auto-confirm email
      user_metadata: {
        username: ADMIN_USERNAME,
        first_name: ADMIN_FIRST_NAME,
        last_name: ADMIN_LAST_NAME
      }
    })

    if (authError) {
      if (authError.message.includes('already registered')) {
        console.log('⚠️  User already exists in auth system')
        
        // Try to get existing user
        const { data: existingUser, error: getUserError } = await supabase.auth.admin.getUserByEmail(ADMIN_EMAIL)
        
        if (getUserError) {
          throw getUserError
        }
        
        console.log('✅ Using existing auth user')
        var userId = existingUser.user.id
      } else {
        throw authError
      }
    } else {
      console.log('✅ Auth user created successfully')
      var userId = authData.user.id
    }

    // Step 2: Create/Update profile
    console.log('\n2️⃣ Creating/updating user profile...')
    const { data: profileData, error: profileError } = await supabase
      .from('profiles')
      .upsert({
        id: userId,
        username: ADMIN_USERNAME,
        email: ADMIN_EMAIL,
        first_name: ADMIN_FIRST_NAME,
        last_name: ADMIN_LAST_NAME,
        is_admin: true,
        account_status: 'active',
        kyc_verified: true,
        kyc_status: 'approved',
        created_at: new Date().toISOString(),
        updated_at: new Date().toISOString()
      }, {
        onConflict: 'id'
      })
      .select()

    if (profileError) {
      throw profileError
    }

    console.log('✅ Profile created/updated successfully')

    // Step 3: Verify admin status
    console.log('\n3️⃣ Verifying admin status...')
    const { data: verifyData, error: verifyError } = await supabase
      .from('profiles')
      .select('id, username, email, full_name, is_admin, is_banned, kyc_verified')
      .eq('id', userId)
      .single()

    if (verifyError) {
      throw verifyError
    }

    console.log('✅ Admin user verification:')
    console.log(`   ID: ${verifyData.id}`)
    console.log(`   Username: ${verifyData.username}`)
    console.log(`   Email: ${verifyData.email}`)
    console.log(`   Full Name: ${verifyData.full_name}`)
    console.log(`   Is Admin: ${verifyData.is_admin}`)
    console.log(`   Is Banned: ${verifyData.is_banned}`)
    console.log(`   KYC Verified: ${verifyData.kyc_verified}`)

    // Step 4: Test login
    console.log('\n4️⃣ Testing admin login...')
    const { data: loginData, error: loginError } = await supabase.auth.signInWithPassword({
      email: ADMIN_EMAIL,
      password: ADMIN_PASSWORD
    })

    if (loginError) {
      console.warn('⚠️  Login test failed:', loginError.message)
      console.log('   You may need to test login manually in the application')
    } else {
      console.log('✅ Login test successful')
      console.log(`   Session ID: ${loginData.session?.access_token?.slice(0, 20)}...`)
      
      // Sign out after test
      await supabase.auth.signOut()
    }

    console.log('\n🎉 Admin user created successfully!')
    console.log('\n📋 Login Credentials:')
    console.log(`   Email: ${ADMIN_EMAIL}`)
    console.log(`   Password: ${ADMIN_PASSWORD}`)
    console.log('\n🔐 Security Recommendations:')
    console.log('   1. Change the password after first login')
    console.log('   2. Enable two-factor authentication')
    console.log('   3. Use a strong, unique password')
    console.log('   4. Regularly review admin access')

  } catch (error) {
    console.error('\n❌ Error creating admin user:')
    console.error(error.message)
    
    if (error.details) {
      console.error('Details:', error.details)
    }
    
    if (error.hint) {
      console.error('Hint:', error.hint)
    }
    
    process.exit(1)
  }
}

// Run the script
if (require.main === module) {
  createAdminUser()
}

module.exports = { createAdminUser }
