// Simple script to create Zimbabwe markets
// Run with: node seed-zimbabwe-markets.js

const { createClient } = require('@supabase/supabase-js')

// You need to set these environment variables or replace with your actual values
const supabaseUrl = process.env.NEXT_PUBLIC_SUPABASE_URL || 'https://myrocyiwjvkdpxfpnuyc.supabase.co'
const supabaseKey = process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY || 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6Im15cm9jeWl3anZrZHB4ZnBudXljIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NjAzNjA3OTAsImV4cCI6MjA3NTkzNjc5MH0.soVAUiwBOM2jipWwTFtq5LZi5NrGgomDSlZTdJqNZm4'

// Use service role key to bypass RLS policies
const serviceRoleKey = process.env.SUPABASE_SERVICE_ROLE_KEY || 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6Im15cm9jeWl3anZrZHB4ZnBudXljIiwicm9sZSI6InNlcnZpY2Vfcm9sZSIsImlhdCI6MTc2MDM2MDc5MCwiZXhwIjoyMDc1OTM2NzkwfQ.YourServiceRoleKeyHere'

const supabase = createClient(supabaseUrl, serviceRoleKey)

const markets = [
  {
    title: "Will Zimbabwe's inflation rate drop below 50% by end of 2024?",
    description: "Zimbabwe has been struggling with hyperinflation for years. Will the government's economic reforms finally bring inflation under control?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.35,
    no_price: 0.65,
    total_volume: 15000
  },
  {
    title: "Will Zimbabwe introduce a new currency in 2024?",
    description: "The Zimbabwean dollar has been unstable. Will the government introduce a new currency or adopt a foreign currency as legal tender?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.42,
    no_price: 0.58,
    total_volume: 12000
  },
  {
    title: "Will Zimbabwe's GDP growth exceed 3% in 2024?",
    description: "Economic recovery in Zimbabwe. Will the country achieve GDP growth above 3% this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.28,
    no_price: 0.72,
    total_volume: 8000
  },
  {
    title: "Will Zimbabwe hold free and fair elections in 2024?",
    description: "Political stability in Zimbabwe. Will the upcoming elections be considered free and fair by international observers?",
    category: "politics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.38,
    no_price: 0.62,
    total_volume: 20000
  },
  {
    title: "Will Zimbabwe's opposition win more than 30% of parliamentary seats?",
    description: "Parliamentary elections in Zimbabwe. Will the opposition parties secure more than 30% of the seats?",
    category: "politics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.45,
    no_price: 0.55,
    total_volume: 18000
  },
  {
    title: "Will Zimbabwe's unemployment rate drop below 20% by end of 2024?",
    description: "Job creation in Zimbabwe. Will the unemployment rate fall below 20% this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.33,
    no_price: 0.67,
    total_volume: 10000
  },
  {
    title: "Will Zimbabwe receive IMF funding in 2024?",
    description: "International financial support. Will Zimbabwe secure funding from the International Monetary Fund this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.52,
    no_price: 0.48,
    total_volume: 25000
  },
  {
    title: "Will Zimbabwe's mining sector contribute more than 15% to GDP?",
    description: "Mining industry performance. Will Zimbabwe's mining sector contribute over 15% to the country's GDP in 2024?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.41,
    no_price: 0.59,
    total_volume: 14000
  },
  {
    title: "Will Zimbabwe's tourism arrivals exceed 2 million in 2024?",
    description: "Tourism recovery in Zimbabwe. Will the country welcome more than 2 million tourists this year?",
    category: "tourism",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.36,
    no_price: 0.64,
    total_volume: 9000
  },
  {
    title: "Will Zimbabwe's agricultural output increase by more than 10%?",
    description: "Agricultural sector growth. Will Zimbabwe's agricultural production increase by over 10% compared to last year?",
    category: "agriculture",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.47,
    no_price: 0.53,
    total_volume: 16000
  },
  {
    title: "Will Zimbabwe's internet penetration exceed 60% by end of 2024?",
    description: "Digital connectivity in Zimbabwe. Will internet penetration reach over 60% of the population?",
    category: "technology",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.39,
    no_price: 0.61,
    total_volume: 7000
  },
  {
    title: "Will Zimbabwe's foreign direct investment exceed $1 billion?",
    description: "Foreign investment in Zimbabwe. Will the country attract more than $1 billion in FDI this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.31,
    no_price: 0.69,
    total_volume: 13000
  },
  {
    title: "Will Zimbabwe's education budget exceed 20% of total budget?",
    description: "Education funding in Zimbabwe. Will the government allocate more than 20% of the budget to education?",
    category: "education",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.44,
    no_price: 0.56,
    total_volume: 11000
  },
  {
    title: "Will Zimbabwe's healthcare system improve significantly in 2024?",
    description: "Healthcare reforms in Zimbabwe. Will there be significant improvements in the healthcare system this year?",
    category: "healthcare",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.37,
    no_price: 0.63,
    total_volume: 12000
  },
  {
    title: "Will Zimbabwe's renewable energy capacity exceed 500MW?",
    description: "Clean energy in Zimbabwe. Will the country's renewable energy capacity reach over 500MW by end of 2024?",
    category: "energy",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.43,
    no_price: 0.57,
    total_volume: 8500
  },
  {
    title: "Will Zimbabwe's manufacturing sector grow by more than 5%?",
    description: "Industrial growth in Zimbabwe. Will the manufacturing sector expand by over 5% this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.40,
    no_price: 0.60,
    total_volume: 15000
  },
  {
    title: "Will Zimbabwe's debt-to-GDP ratio fall below 80%?",
    description: "Debt management in Zimbabwe. Will the country's debt-to-GDP ratio drop below 80% by end of 2024?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.29,
    no_price: 0.71,
    total_volume: 17000
  },
  {
    title: "Will Zimbabwe's exports exceed $4 billion in 2024?",
    description: "Export performance in Zimbabwe. Will the country's exports reach over $4 billion this year?",
    category: "economics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.46,
    no_price: 0.54,
    total_volume: 19000
  },
  {
    title: "Will Zimbabwe's urban population exceed 40% of total population?",
    description: "Urbanization in Zimbabwe. Will the urban population reach over 40% of the total population?",
    category: "demographics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.34,
    no_price: 0.66,
    total_volume: 6000
  },
  {
    title: "Will Zimbabwe's life expectancy exceed 65 years?",
    description: "Health outcomes in Zimbabwe. Will the average life expectancy reach over 65 years by end of 2024?",
    category: "healthcare",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.48,
    no_price: 0.52,
    total_volume: 10000
  },
  {
    title: "Will Zimbabwe's literacy rate exceed 90%?",
    description: "Education outcomes in Zimbabwe. Will the literacy rate reach over 90% of the population?",
    category: "education",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.42,
    no_price: 0.58,
    total_volume: 8000
  },
  {
    title: "Will Zimbabwe's corruption perception index improve by more than 10 points?",
    description: "Governance in Zimbabwe. Will the country's corruption perception index improve by over 10 points?",
    category: "politics",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.35,
    no_price: 0.65,
    total_volume: 14000
  },
  {
    title: "Will Zimbabwe's stock market index gain more than 20%?",
    description: "Financial markets in Zimbabwe. Will the stock market index increase by over 20% this year?",
    category: "finance",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.38,
    no_price: 0.62,
    total_volume: 16000
  },
  {
    title: "Will Zimbabwe's mobile money transactions exceed $10 billion?",
    description: "Digital payments in Zimbabwe. Will mobile money transactions reach over $10 billion this year?",
    category: "technology",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.51,
    no_price: 0.49,
    total_volume: 12000
  },
  {
    title: "Will Zimbabwe's climate change adaptation funding exceed $100 million?",
    description: "Environmental resilience in Zimbabwe. Will the country secure over $100 million in climate adaptation funding?",
    category: "environment",
    end_date: "2024-12-31T23:59:59Z",
    yes_price: 0.33,
    no_price: 0.67,
    total_volume: 9000
  }
]

async function createMarkets() {
  console.log('🚀 Starting to create Zimbabwe markets...')
  
  // Check if we have the required environment variables
  if (supabaseUrl.includes('your-project') || supabaseKey.includes('your-anon-key')) {
    console.error('❌ Please set your Supabase environment variables:')
    console.error('   NEXT_PUBLIC_SUPABASE_URL=your_supabase_url')
    console.error('   NEXT_PUBLIC_SUPABASE_ANON_KEY=your_supabase_anon_key')
    return
  }
  
  try {
    // Find admin user
    console.log('🔍 Looking for admin user...')
    const { data: adminUser, error: adminError } = await supabase
      .from('profiles')
      .select('id, email, is_admin')
      .eq('is_admin', true)
      .limit(1)
      .single()
    
    if (adminError || !adminUser) {
      console.error('❌ No admin user found. Please create an admin user first.')
      console.error('Error:', adminError)
      return
    }
    
    console.log('✅ Admin user found:', adminUser.email)
    
    let successCount = 0
    let errorCount = 0
    
    // Create markets
    for (let i = 0; i < markets.length; i++) {
      const market = markets[i]
      
      try {
        console.log(`📝 Creating market ${i + 1}/25: ${market.title.substring(0, 50)}...`)
        
        const { data, error } = await supabase
          .from('markets')
          .insert({
            title: market.title,
            description: market.description,
            category: market.category,
            status: 'active',
            end_date: market.end_date,
            yes_price: market.yes_price,
            no_price: market.no_price,
            total_volume: market.total_volume,
            creator_id: adminUser.id,
            created_at: new Date().toISOString()
          })
          .select()
          .single()
        
        if (error) {
          console.error(`❌ Error creating market ${i + 1}:`, error.message)
          errorCount++
          continue
        }
        
        console.log(`✅ Market ${i + 1} created: ${data.id}`)
        successCount++
        
        // Add some liquidity to make the market more realistic
        const liquidityAmount = Math.floor(Math.random() * 3000) + 1000
        
        try {
          await supabase
            .from('liquidity_providers')
            .insert([
              {
                user_id: adminUser.id,
                market_id: data.id,
                side: 'yes',
                amount: liquidityAmount,
                shares: liquidityAmount * market.yes_price,
                created_at: new Date().toISOString()
              },
              {
                user_id: adminUser.id,
                market_id: data.id,
                side: 'no',
                amount: liquidityAmount,
                shares: liquidityAmount * market.no_price,
                created_at: new Date().toISOString()
              }
            ])
          
          console.log(`💰 Added liquidity to market ${i + 1}`)
        } catch (liquidityError) {
          console.warn(`⚠️  Could not add liquidity to market ${i + 1}:`, liquidityError.message)
        }
        
        // Small delay to avoid overwhelming the database
        await new Promise(resolve => setTimeout(resolve, 200))
        
      } catch (error) {
        console.error(`❌ Unexpected error creating market ${i + 1}:`, error.message)
        errorCount++
      }
    }
    
    console.log('\n🎉 Market creation completed!')
    console.log(`✅ Successfully created: ${successCount} markets`)
    console.log(`❌ Failed to create: ${errorCount} markets`)
    
    if (successCount > 0) {
      console.log('\n🌍 Your Zimbabwe markets are now live!')
      console.log('Visit your homepage to see them in action.')
    }
    
  } catch (error) {
    console.error('❌ Fatal error:', error.message)
  }
}

// Run the script
createMarkets()

