'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { BettingEngine } from '@/lib/betting'
import { LoadingSpinner, PageLoadingSpinner } from '@/components/ui/LoadingSpinner'
import Link from 'next/link'

interface AnalyticsData {
  // Platform Overview
  totalUsers: number
  totalMarkets: number
  totalBets: number
  totalVolume: number
  totalRevenue: number
  
  // Algorithm Performance
  algorithmEfficiency: number
  liquidityUtilization: number
  priceAccuracy: number
  marketStability: number
  
  // User Growth
  userGrowth: {
    daily: number[]
    weekly: number[]
    monthly: number[]
  }
  
  // Trading Volume
  volumeData: {
    daily: number[]
    weekly: number[]
    monthly: number[]
  }
  
  // Market Performance
  marketPerformance: {
    totalMarkets: number
    activeMarkets: number
    resolvedMarkets: number
    averageResolutionTime: number
    successRate: number
  }
  
  // Top Performers
  topUsers: Array<{
    id: string
    username: string
    winRate: number
    totalProfit: number
    totalVolume: number
  }>
  
  topMarkets: Array<{
    id: string
    title: string
    totalVolume: number
    liquidity: number
    efficiency: number
  }>
  
  // Algorithm Metrics
  algorithmMetrics: {
    totalLiquidity: number
    seededVolume: number
    liveVolume: number
    averagePriceVolatility: number
    liquidityDepth: number
    marketMakingEfficiency: number
  }
  
  // Risk Analytics
  riskAnalytics: {
    highRiskUsers: number
    suspiciousActivity: number
    averageRiskScore: number
    riskDistribution: number[]
  }
}

export default function AdminAnalytics() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <AdminAnalyticsContent user={user} />
    </FastSessionWrapper>
  )
}

function AdminAnalyticsContent({ user }: { user: any }) {
  const [analytics, setAnalytics] = useState<AnalyticsData | null>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [timeRange, setTimeRange] = useState<'7d' | '30d' | '90d' | '1y'>('30d')

  useEffect(() => {
    fetchAnalytics()
  }, [timeRange])

  const fetchAnalytics = async () => {
    try {
      setLoading(true)
      setError(null)

      const now = new Date()
      const timeRanges = {
        '7d': new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000),
        '30d': new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000),
        '90d': new Date(now.getTime() - 90 * 24 * 60 * 60 * 1000),
        '1y': new Date(now.getTime() - 365 * 24 * 60 * 60 * 1000)
      }
      const startDate = timeRanges[timeRange]

      // Fetch comprehensive analytics data
      const [
        platformData,
        userGrowthData,
        volumeData,
        marketData,
        topPerformersData,
        algorithmData,
        riskData
      ] = await Promise.all([
        // Platform overview
        Promise.all([
          supabase.from('profiles').select('id', { count: 'exact', head: true }),
          supabase.from('markets').select('id', { count: 'exact', head: true }),
          supabase.from('bets').select('id, amount', { count: 'exact', head: true }),
          supabase.from('bets').select('amount').gte('created_at', startDate.toISOString()),
          supabase.from('markets').select('admin_fee').gte('created_at', startDate.toISOString())
        ]),

        // User growth data
        fetchUserGrowthData(startDate),

        // Volume data
        fetchVolumeData(startDate),

        // Market performance data
        fetchMarketPerformanceData(startDate),

        // Top performers
        fetchTopPerformersData(startDate),

        // Algorithm metrics
        fetchAlgorithmMetrics(startDate),

        // Risk analytics
        fetchRiskAnalytics(startDate)
      ])

      // Process platform data
      const [totalUsers, totalMarkets, totalBets, volumeBets, revenueData] = platformData
      const totalVolume = volumeBets.data?.reduce((sum, bet) => sum + bet.amount, 0) || 0
      const totalRevenue = revenueData.data?.reduce((sum, market) => sum + (market.admin_fee || 0), 0) || 0

      // Calculate algorithm efficiency
      const algorithmEfficiency = await calculateAlgorithmEfficiency(startDate)

      setAnalytics({
        totalUsers: totalUsers.count || 0,
        totalMarkets: totalMarkets.count || 0,
        totalBets: totalBets.count || 0,
        totalVolume,
        totalRevenue,
        algorithmEfficiency,
        liquidityUtilization: algorithmData.liquidityUtilization,
        priceAccuracy: algorithmData.priceAccuracy,
        marketStability: algorithmData.marketStability,
        userGrowth: userGrowthData,
        volumeData,
        marketPerformance: marketData,
        topUsers: topPerformersData.topUsers,
        topMarkets: topPerformersData.topMarkets,
        algorithmMetrics: algorithmData,
        riskAnalytics: riskData
      })

    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch analytics')
      console.error('Error fetching analytics:', err)
    } finally {
      setLoading(false)
    }
  }

  const fetchUserGrowthData = async (startDate: Date) => {
    try {
      const { data } = await supabase
        .from('profiles')
        .select('created_at')
        .gte('created_at', startDate.toISOString())
        .order('created_at', { ascending: true })

      // Group by time periods
      const daily = new Array(7).fill(0)
      const weekly = new Array(4).fill(0)
      const monthly = new Array(12).fill(0)

      data?.forEach(profile => {
        const date = new Date(profile.created_at)
        const daysDiff = Math.floor((Date.now() - date.getTime()) / (1000 * 60 * 60 * 24))
        
        if (daysDiff < 7) daily[6 - daysDiff]++
        if (daysDiff < 28) weekly[3 - Math.floor(daysDiff / 7)]++
        if (daysDiff < 365) monthly[11 - Math.floor(daysDiff / 30)]++
      })

      return { daily, weekly, monthly }
    } catch (error) {
      console.error('Error fetching user growth data:', error)
      return { daily: [], weekly: [], monthly: [] }
    }
  }

  const fetchVolumeData = async (startDate: Date) => {
    try {
      const { data } = await supabase
        .from('bets')
        .select('amount, created_at')
        .gte('created_at', startDate.toISOString())
        .order('created_at', { ascending: true })

      // Group by time periods
      const daily = new Array(7).fill(0)
      const weekly = new Array(4).fill(0)
      const monthly = new Array(12).fill(0)

      data?.forEach(bet => {
        const date = new Date(bet.created_at)
        const daysDiff = Math.floor((Date.now() - date.getTime()) / (1000 * 60 * 60 * 24))
        
        if (daysDiff < 7) daily[6 - daysDiff] += bet.amount
        if (daysDiff < 28) weekly[3 - Math.floor(daysDiff / 7)] += bet.amount
        if (daysDiff < 365) monthly[11 - Math.floor(daysDiff / 30)] += bet.amount
      })

      return { daily, weekly, monthly }
    } catch (error) {
      console.error('Error fetching volume data:', error)
      return { daily: [], weekly: [], monthly: [] }
    }
  }

  const fetchMarketPerformanceData = async (startDate: Date) => {
    try {
      const { data: markets } = await supabase
        .from('markets')
        .select('status, created_at, resolution_date')
        .gte('created_at', startDate.toISOString())

      const totalMarkets = markets?.length || 0
      const activeMarkets = markets?.filter(m => m.status === 'active').length || 0
      const resolvedMarkets = markets?.filter(m => m.status === 'resolved').length || 0
      
      // Calculate average resolution time
      const resolvedWithDates = markets?.filter(m => m.status === 'resolved' && m.resolution_date) || []
      const averageResolutionTime = resolvedWithDates.length > 0 
        ? resolvedWithDates.reduce((sum, market) => {
            const created = new Date(market.created_at)
            const resolved = new Date(market.resolution_date)
            return sum + (resolved.getTime() - created.getTime()) / (1000 * 60 * 60 * 24)
          }, 0) / resolvedWithDates.length
        : 0

      // Calculate success rate (markets that were resolved vs created)
      const successRate = totalMarkets > 0 ? (resolvedMarkets / totalMarkets) * 100 : 0

      return {
        totalMarkets,
        activeMarkets,
        resolvedMarkets,
        averageResolutionTime,
        successRate
      }
    } catch (error) {
      console.error('Error fetching market performance data:', error)
      return {
        totalMarkets: 0,
        activeMarkets: 0,
        resolvedMarkets: 0,
        averageResolutionTime: 0,
        successRate: 0
      }
    }
  }

  const fetchTopPerformersData = async (startDate: Date) => {
    try {
      // Top users by performance
      const { data: topUsers } = await supabase
        .from('profiles')
        .select(`
          id, username,
          bets!inner(id, status, amount, potential_payout)
        `)
        .gte('bets.created_at', startDate.toISOString())

      const userStats = (topUsers || []).map(user => {
        const bets = user.bets || []
        const wonBets = bets.filter(bet => bet.status === 'won').length
        const totalCompletedBets = bets.filter(bet => bet.status === 'won' || bet.status === 'lost').length
        const winRate = totalCompletedBets > 0 ? (wonBets / totalCompletedBets) * 100 : 0
        const totalProfit = bets.reduce((sum, bet) => {
          if (bet.status === 'won') return sum + (bet.potential_payout || 0)
          if (bet.status === 'lost') return sum - bet.amount
          return sum
        }, 0)
        const totalVolume = bets.reduce((sum, bet) => sum + bet.amount, 0)

        return {
          id: user.id,
          username: user.username || 'Unknown',
          winRate,
          totalProfit,
          totalVolume
        }
      }).sort((a, b) => b.winRate - a.winRate).slice(0, 10)

      // Top markets by volume
      const { data: topMarkets } = await supabase
        .from('markets')
        .select('id, title, total_volume, total_liquidity, seeded_volume, live_volume')
        .gte('created_at', startDate.toISOString())
        .order('total_volume', { ascending: false })
        .limit(10)

      const marketStats = (topMarkets || []).map(market => {
        const efficiency = market.total_liquidity > 0 
          ? ((market.seeded_volume || 0) / market.total_liquidity) * 100 
          : 0

        return {
          id: market.id,
          title: market.title,
          totalVolume: market.total_volume || 0,
          liquidity: market.total_liquidity || 0,
          efficiency
        }
      })

      return {
        topUsers: userStats,
        topMarkets: marketStats
      }
    } catch (error) {
      console.error('Error fetching top performers data:', error)
      return { topUsers: [], topMarkets: [] }
    }
  }

  const fetchAlgorithmMetrics = async (startDate: Date) => {
    try {
      const { data: markets } = await supabase
        .from('markets')
        .select('total_liquidity, seeded_volume, live_volume, yes_price, no_price')
        .gte('created_at', startDate.toISOString())

      const { data: priceHistory } = await supabase
        .from('price_history')
        .select('yes_price, no_price')
        .gte('created_at', startDate.toISOString())
        .order('created_at', { ascending: false })
        .limit(1000)

      const totalLiquidity = markets?.reduce((sum, market) => sum + (market.total_liquidity || 0), 0) || 0
      const seededVolume = markets?.reduce((sum, market) => sum + (market.seeded_volume || 0), 0) || 0
      const liveVolume = markets?.reduce((sum, market) => sum + (market.live_volume || 0), 0) || 0

      // Calculate price volatility
      let totalVolatility = 0
      let volatilityCount = 0
      if (priceHistory && priceHistory.length > 1) {
        for (let i = 1; i < priceHistory.length; i++) {
          const prev = priceHistory[i - 1]
          const curr = priceHistory[i]
          const volatility = Math.abs((curr.yes_price || 0.5) - (prev.yes_price || 0.5))
          totalVolatility += volatility
          volatilityCount++
        }
      }
      const averagePriceVolatility = volatilityCount > 0 ? totalVolatility / volatilityCount : 0

      // Calculate liquidity utilization
      const liquidityUtilization = totalLiquidity > 0 ? (seededVolume / totalLiquidity) * 100 : 0

      // Calculate price accuracy
      const balancedMarkets = markets?.filter(market => 
        Math.abs((market.yes_price || 0.5) - 0.5) < 0.1
      ) || []
      const priceAccuracy = markets && markets.length > 0 ? (balancedMarkets.length / markets.length) * 100 : 0

      // Calculate market stability
      const marketStability = Math.max(0, 100 - (averagePriceVolatility * 200))

      // Calculate liquidity depth
      const liquidityDepth = totalLiquidity > 0 ? (liveVolume / totalLiquidity) * 100 : 0

      // Calculate market making efficiency
      const marketMakingEfficiency = Math.min(100, (liquidityUtilization + marketStability) / 2)

      return {
        totalLiquidity,
        seededVolume,
        liveVolume,
        averagePriceVolatility,
        liquidityDepth,
        marketMakingEfficiency,
        liquidityUtilization,
        priceAccuracy,
        marketStability
      }
    } catch (error) {
      console.error('Error fetching algorithm metrics:', error)
      return {
        totalLiquidity: 0,
        seededVolume: 0,
        liveVolume: 0,
        averagePriceVolatility: 0,
        liquidityDepth: 0,
        marketMakingEfficiency: 0,
        liquidityUtilization: 0,
        priceAccuracy: 0,
        marketStability: 0
      }
    }
  }

  const fetchRiskAnalytics = async (startDate: Date) => {
    try {
      const { data: users } = await supabase
        .from('profiles')
        .select(`
          id,
          bets!inner(amount, created_at)
        `)
        .gte('bets.created_at', startDate.toISOString())

      let highRiskUsers = 0
      let suspiciousActivity = 0
      let totalRiskScore = 0
      let userCount = 0
      const riskDistribution = new Array(10).fill(0) // 0-10, 10-20, etc.

      users?.forEach(user => {
        const bets = user.bets || []
        if (bets.length === 0) return

        // Calculate risk score for this user
        const totalVolume = bets.reduce((sum, bet) => sum + bet.amount, 0)
        const avgBet = totalVolume / bets.length
        const variance = bets.reduce((sum, bet) => sum + Math.pow(bet.amount - avgBet, 2), 0) / bets.length
        const volatility = Math.sqrt(variance) / avgBet
        const riskScore = Math.min(100, volatility * 50)

        totalRiskScore += riskScore
        userCount++

        // Categorize risk
        const riskCategory = Math.floor(riskScore / 10)
        if (riskCategory < 10) riskDistribution[riskCategory]++

        if (riskScore > 70) highRiskUsers++
        if (riskScore > 80) suspiciousActivity++
      })

      const averageRiskScore = userCount > 0 ? totalRiskScore / userCount : 0

      return {
        highRiskUsers,
        suspiciousActivity,
        averageRiskScore,
        riskDistribution
      }
    } catch (error) {
      console.error('Error fetching risk analytics:', error)
      return {
        highRiskUsers: 0,
        suspiciousActivity: 0,
        averageRiskScore: 0,
        riskDistribution: []
      }
    }
  }

  const calculateAlgorithmEfficiency = async (startDate: Date): Promise<number> => {
    try {
      // This would integrate with the actual BettingEngine
      // For now, return a calculated efficiency based on market data
      const { data: markets } = await supabase
        .from('markets')
        .select('yes_price, no_price, total_liquidity, seeded_volume')
        .gte('created_at', startDate.toISOString())

      if (!markets || markets.length === 0) return 0

      let totalEfficiency = 0
      markets.forEach(market => {
        const priceBalance = Math.abs((market.yes_price || 0.5) - 0.5)
        const liquidityUtilization = market.total_liquidity > 0 
          ? ((market.seeded_volume || 0) / market.total_liquidity) * 100 
          : 0
        const efficiency = Math.max(0, 100 - (priceBalance * 200) + liquidityUtilization)
        totalEfficiency += efficiency
      })

      return totalEfficiency / markets.length
    } catch (error) {
      console.error('Error calculating algorithm efficiency:', error)
      return 0
    }
  }

  if (loading) {
    return <PageLoadingSpinner />
  }

  if (error || !analytics) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h2 className="text-2xl font-bold text-black mb-4">Error Loading Analytics</h2>
          <p className="text-black mb-4">{error || 'Failed to load analytics data'}</p>
          <button
            onClick={fetchAnalytics}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700"
          >
            Retry
          </button>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-black">Analytics Dashboard</h1>
              <p className="text-black mt-2">Comprehensive platform analytics with algorithm insights</p>
            </div>
            <div className="flex items-center space-x-4">
              <select
                value={timeRange}
                onChange={(e) => setTimeRange(e.target.value as any)}
                className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              >
                <option value="7d">Last 7 days</option>
                <option value="30d">Last 30 days</option>
                <option value="90d">Last 90 days</option>
                <option value="1y">Last year</option>
              </select>
              <Link
                href="/admin"
                className="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition-colors"
              >
                ← Back to Dashboard
              </Link>
            </div>
          </div>
        </div>

        {/* Platform Overview */}
        <div className="grid grid-cols-1 md:grid-cols-5 gap-6 mb-8">
          <div className="bg-white rounded-lg shadow-sm p-6">
            <div className="text-2xl font-bold text-blue-600">{analytics.totalUsers.toLocaleString()}</div>
            <div className="text-sm text-black">Total Users</div>
          </div>
          <div className="bg-white rounded-lg shadow-sm p-6">
            <div className="text-2xl font-bold text-green-600">{analytics.totalMarkets.toLocaleString()}</div>
            <div className="text-sm text-black">Total Markets</div>
          </div>
          <div className="bg-white rounded-lg shadow-sm p-6">
            <div className="text-2xl font-bold text-purple-600">${analytics.totalVolume.toLocaleString()}</div>
            <div className="text-sm text-black">Total Volume</div>
          </div>
          <div className="bg-white rounded-lg shadow-sm p-6">
            <div className="text-2xl font-bold text-orange-600">${analytics.totalRevenue.toLocaleString()}</div>
            <div className="text-sm text-black">Total Revenue</div>
          </div>
          <div className="bg-white rounded-lg shadow-sm p-6">
            <div className="text-2xl font-bold text-indigo-600">{analytics.algorithmEfficiency.toFixed(1)}%</div>
            <div className="text-sm text-black">Algorithm Efficiency</div>
          </div>
        </div>

        {/* Algorithm Performance */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Algorithm Performance</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
            <div className="text-center">
              <div className="text-3xl font-bold text-blue-600">{analytics.liquidityUtilization.toFixed(1)}%</div>
              <div className="text-sm text-black">Liquidity Utilization</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-green-600">{analytics.priceAccuracy.toFixed(1)}%</div>
              <div className="text-sm text-black">Price Accuracy</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-purple-600">{analytics.marketStability.toFixed(1)}%</div>
              <div className="text-sm text-black">Market Stability</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-orange-600">{analytics.algorithmMetrics.marketMakingEfficiency.toFixed(1)}%</div>
              <div className="text-sm text-black">Market Making Efficiency</div>
            </div>
          </div>
        </div>

        {/* Market Performance */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Market Performance</h2>
          <div className="grid grid-cols-1 md:grid-cols-5 gap-6">
            <div className="text-center">
              <div className="text-3xl font-bold text-blue-600">{analytics.marketPerformance.totalMarkets}</div>
              <div className="text-sm text-black">Total Markets</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-green-600">{analytics.marketPerformance.activeMarkets}</div>
              <div className="text-sm text-black">Active Markets</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-purple-600">{analytics.marketPerformance.resolvedMarkets}</div>
              <div className="text-sm text-black">Resolved Markets</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-orange-600">{analytics.marketPerformance.averageResolutionTime.toFixed(1)}d</div>
              <div className="text-sm text-black">Avg Resolution Time</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-indigo-600">{analytics.marketPerformance.successRate.toFixed(1)}%</div>
              <div className="text-sm text-black">Success Rate</div>
            </div>
          </div>
        </div>

        {/* Top Performers */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
          <div className="bg-white rounded-lg shadow-sm p-6">
            <h2 className="text-xl font-semibold text-black mb-4">Top Users</h2>
            <div className="space-y-3">
              {analytics.topUsers.slice(0, 5).map((user, index) => (
                <div key={user.id} className="flex items-center justify-between p-3 border border-gray-200 rounded-lg">
                  <div className="flex items-center">
                    <div className="w-8 h-8 bg-gradient-to-br from-pink-500 via-purple-500 to-white rounded-full flex items-center justify-center mr-3">
                      <span className="text-white text-sm font-bold">{index + 1}</span>
                    </div>
                    <div>
                      <div className="font-medium text-black">{user.username}</div>
                      <div className="text-sm text-black">${user.totalVolume.toLocaleString()} volume</div>
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="font-bold text-green-600">{user.winRate.toFixed(1)}%</div>
                    <div className="text-sm text-black">${user.totalProfit.toFixed(2)} profit</div>
                  </div>
                </div>
              ))}
            </div>
          </div>

          <div className="bg-white rounded-lg shadow-sm p-6">
            <h2 className="text-xl font-semibold text-black mb-4">Top Markets</h2>
            <div className="space-y-3">
              {analytics.topMarkets.slice(0, 5).map((market, index) => (
                <div key={market.id} className="flex items-center justify-between p-3 border border-gray-200 rounded-lg">
                  <div className="flex items-center">
                    <div className="w-8 h-8 bg-gradient-to-br from-blue-500 via-purple-500 to-white rounded-full flex items-center justify-center mr-3">
                      <span className="text-white text-sm font-bold">{index + 1}</span>
                    </div>
                    <div>
                      <div className="font-medium text-black truncate">{market.title}</div>
                      <div className="text-sm text-black">${market.liquidity.toLocaleString()} liquidity</div>
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="font-bold text-blue-600">${market.totalVolume.toLocaleString()}</div>
                    <div className="text-sm text-black">{market.efficiency.toFixed(1)}% efficiency</div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Risk Analytics */}
        <div className="bg-white rounded-lg shadow-sm p-6">
          <h2 className="text-xl font-semibold text-black mb-4">Risk Analytics</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
            <div className="text-center">
              <div className="text-3xl font-bold text-red-600">{analytics.riskAnalytics.highRiskUsers}</div>
              <div className="text-sm text-black">High Risk Users</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-orange-600">{analytics.riskAnalytics.suspiciousActivity}</div>
              <div className="text-sm text-black">Suspicious Activity</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-yellow-600">{analytics.riskAnalytics.averageRiskScore.toFixed(1)}</div>
              <div className="text-sm text-black">Average Risk Score</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-indigo-600">${analytics.algorithmMetrics.totalLiquidity.toLocaleString()}</div>
              <div className="text-sm text-black">Total Liquidity</div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}