'use client'

import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { PageLoadingSpinner } from '@/components/ui/LoadingSpinner'
import { useAdminBetsOptimized } from '@/lib/hooks/useAdminBetsOptimized'
import AdminBetsTable from '@/components/admin/AdminBetsTable'
import Link from 'next/link'

interface BetWithDetails {
  id: string
  user_id: string
  market_id: string
  outcome: string
  amount: number
  price: number
  shares: number
  status: string
  seeded_amount: number
  live_amount: number
  liquidity_contribution: number
  potential_payout: number
  created_at: string
  updated_at: string
  user_profile?: {
    username: string
    email: string
  }
  market?: {
    title: string
    status: string
    yes_price: number
    no_price: number
    total_liquidity: number
    seeded_volume: number
    live_volume: number
  }
  algorithm_metrics?: {
    price_impact: number
    liquidity_utilization: number
    market_efficiency: number
    risk_score: number
  }
}

interface BetMetrics {
  totalBets: number
  totalVolume: number
  seededVolume: number
  liveVolume: number
  matchedBets: number
  pendingBets: number
  wonBets: number
  lostBets: number
  totalPayouts: number
  averageBetSize: number
  algorithmEfficiency: number
  liquidityUtilization: number
}

export default function AdminBets() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <AdminBetsContent user={user} />
    </FastSessionWrapper>
  )
}

function AdminBetsContent({ user }: { user: any }) {
  const {
    bets,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    setSearchTerm,
    loadMore,
    refresh,
    filterByStatus
  } = useAdminBetsOptimized()


  if (loading && bets.length === 0) {
    return <PageLoadingSpinner />
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-black">Bet Management</h1>
              <p className="text-black mt-2">Monitor and manage all bets with optimized performance</p>
            </div>
            <Link
              href="/admin"
              className="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition-colors"
            >
              ← Back to Dashboard
            </Link>
          </div>
        </div>

        {/* Bet Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Betting Overview</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{metrics.totalBets}</div>
              <div className="text-sm text-black">Total Bets</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-green-600">${metrics.totalVolume.toLocaleString()}</div>
              <div className="text-sm text-black">Total Volume</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-purple-600">{metrics.matchedBets}</div>
              <div className="text-sm text-black">Matched Bets</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-orange-600">{metrics.algorithmEfficiency.toFixed(1)}%</div>
              <div className="text-sm text-black">Algorithm Efficiency</div>
            </div>
          </div>
        </div>

        {/* Algorithm Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Algorithm Performance</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">${metrics.seededVolume.toLocaleString()}</div>
              <div className="text-sm text-black">Seeded Volume</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-green-600">${metrics.liveVolume.toLocaleString()}</div>
              <div className="text-sm text-black">Live Volume</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-purple-600">{metrics.liquidityUtilization.toFixed(1)}%</div>
              <div className="text-sm text-black">Liquidity Utilization</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-orange-600">${metrics.averageBetSize.toFixed(2)}</div>
              <div className="text-sm text-black">Average Bet Size</div>
            </div>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <input
                type="text"
                placeholder="Search bets by user or market..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>
            <div className="flex gap-2">
              {['all', 'seeded', 'live', 'matched', 'won', 'lost', 'cancelled'].map((status) => (
                <button
                  key={status}
                  onClick={() => filterByStatus(status)}
                  className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                    'bg-blue-600 text-white'
                  }`}
                >
                  {status.charAt(0).toUpperCase() + status.slice(1)}
                </button>
              ))}
            </div>
          </div>
        </div>

        {/* Bets Table */}
        <AdminBetsTable
          bets={bets}
          loading={loading}
          hasMore={hasMore}
          onLoadMore={loadMore}
          onRefresh={refresh}
        />
      </div>
    </div>
  )
}