'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { LoadingSpinner, PageLoadingSpinner, ButtonLoadingSpinner } from '@/components/ui/LoadingSpinner'
import Link from 'next/link'

interface KYCApplication {
  id: string
  user_id: string
  status: 'pending' | 'approved' | 'rejected'
  document_type: string
  document_number: string
  first_name: string
  last_name: string
  date_of_birth: string
  address: string
  phone_number: string
  document_front_url: string
  document_back_url: string
  selfie_url: string
  submitted_at: string
  reviewed_at: string | null
  reviewed_by: string | null
  rejection_reason: string | null
  user_profile?: {
    username: string
    email: string
    created_at: string
    balance: number
    total_volume: number
  }
  risk_assessment?: {
    risk_score: number
    risk_factors: string[]
    recommendation: 'approve' | 'reject' | 'manual_review'
  }
}

interface KYCMetrics {
  totalApplications: number
  pendingApplications: number
  approvedApplications: number
  rejectedApplications: number
  averageProcessingTime: number
  approvalRate: number
  highRiskApplications: number
  suspiciousApplications: number
}

export default function AdminKYC() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <AdminKYCContent user={user} />
    </FastSessionWrapper>
  )
}

function AdminKYCContent({ user }: { user: any }) {
  const [applications, setApplications] = useState<KYCApplication[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [selectedStatus, setSelectedStatus] = useState<string>('all')
  const [searchTerm, setSearchTerm] = useState('')
  const [selectedApplication, setSelectedApplication] = useState<KYCApplication | null>(null)
  const [showModal, setShowModal] = useState(false)
  const [actionLoading, setActionLoading] = useState(false)
  const [kycMetrics, setKycMetrics] = useState<KYCMetrics>({
    totalApplications: 0,
    pendingApplications: 0,
    approvedApplications: 0,
    rejectedApplications: 0,
    averageProcessingTime: 0,
    approvalRate: 0,
    highRiskApplications: 0,
    suspiciousApplications: 0
  })

  useEffect(() => {
    fetchKYCApplications()
  }, [selectedStatus])

  const fetchKYCApplications = async () => {
    try {
      setLoading(true)
      setError(null)

      // Try using the view first, fallback to direct table query
      let query = supabase
        .from('kyc_applications_with_profiles')
        .select('*')
        .order('submitted_at', { ascending: false })

      if (selectedStatus !== 'all') {
        query = query.eq('status', selectedStatus)
      }

      const { data, error: fetchError } = await query

      if (fetchError) {
        console.error('Error fetching KYC applications:', fetchError)
        
        // Fallback: try direct table query if view fails
        console.log('Trying fallback query...')
        const fallbackQuery = supabase
          .from('kyc_applications')
          .select(`
            *,
            profiles!kyc_applications_user_id_fkey(
              username, 
              email, 
              created_at, 
              balance
            )
          `)
          .order('submitted_at', { ascending: false })

        const fallbackQueryWithFilter = selectedStatus !== 'all' 
          ? fallbackQuery.eq('status', selectedStatus)
          : fallbackQuery

        const { data: fallbackData, error: fallbackError } = await fallbackQueryWithFilter

        if (fallbackError) {
          console.error('Fallback query also failed:', fallbackError)
          throw new Error(`Failed to fetch KYC applications: ${fallbackError.message}`)
        }

        // Process fallback data
        const processedApplications = await Promise.all((fallbackData || []).map(async (application) => {
          const riskAssessment = await calculateRiskAssessment({
            ...application,
            user_profile: application.profiles
          })
          return {
            ...application,
            user_profile: application.profiles,
            risk_assessment: riskAssessment
          }
        }))

        setApplications(processedApplications)
        console.log('Fetched KYC applications via fallback:', processedApplications.length, 'applications')
        return
      }

      console.log('Fetched KYC applications:', data?.length || 0, 'applications')

      // Process KYC applications with risk assessment
      const processedApplications = await Promise.all((data || []).map(async (application) => {
        const riskAssessment = await calculateRiskAssessment(application)
        return {
          ...application,
          user_profile: application.profiles,
          risk_assessment: riskAssessment
        }
      }))

      setApplications(processedApplications)

      // Calculate KYC metrics
      const totalApplications = processedApplications.length
      const pendingApplications = processedApplications.filter(app => app.status === 'pending').length
      const approvedApplications = processedApplications.filter(app => app.status === 'approved').length
      const rejectedApplications = processedApplications.filter(app => app.status === 'rejected').length
      
      // Calculate average processing time
      const processedApps = processedApplications.filter(app => app.reviewed_at)
      const averageProcessingTime = processedApps.length > 0 
        ? processedApps.reduce((sum, app) => {
            const submitted = new Date(app.submitted_at)
            const reviewed = new Date(app.reviewed_at!)
            return sum + (reviewed.getTime() - submitted.getTime()) / (1000 * 60 * 60 * 24)
          }, 0) / processedApps.length
        : 0

      const approvalRate = totalApplications > 0 ? (approvedApplications / totalApplications) * 100 : 0
      const highRiskApplications = processedApplications.filter(app => 
        app.risk_assessment?.risk_score && app.risk_assessment.risk_score > 70
      ).length
      const suspiciousApplications = processedApplications.filter(app => 
        app.risk_assessment?.recommendation === 'reject'
      ).length

      setKycMetrics({
        totalApplications,
        pendingApplications,
        approvedApplications,
        rejectedApplications,
        averageProcessingTime,
        approvalRate,
        highRiskApplications,
        suspiciousApplications
      })

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to fetch KYC applications'
      setError(errorMessage)
      console.error('Error fetching KYC applications:', err)
      
      // If it's a table not found error, provide helpful guidance
      if (errorMessage.includes('relation "kyc_applications" does not exist')) {
        setError('KYC applications table not found. Please run the database migration script: fix-kyc-applications.sql')
      }
    } finally {
      setLoading(false)
    }
  }

  const calculateRiskAssessment = async (application: any) => {
    try {
      const riskFactors: string[] = []
      let riskScore = 0

      // Check user account age
      const userCreated = new Date(application.user_profile?.created_at || application.submitted_at)
      const daysSinceCreation = (Date.now() - userCreated.getTime()) / (1000 * 60 * 60 * 24)
      
      if (daysSinceCreation < 7) {
        riskScore += 30
        riskFactors.push('New account (< 7 days)')
      } else if (daysSinceCreation < 30) {
        riskScore += 15
        riskFactors.push('Recent account (< 30 days)')
      }

      // Check user activity (using balance as proxy since total_volume doesn't exist)
      const userBalance = application.user_profile?.balance || 0
      if (userBalance === 0) {
        riskScore += 20
        riskFactors.push('No account balance')
      } else if (userBalance > 10000) {
        riskScore += 10
        riskFactors.push('High balance account')
      }

      // Check document quality (simplified)
      if (!application.document_front_url || !application.document_back_url) {
        riskScore += 25
        riskFactors.push('Incomplete documentation')
      }

      if (!application.selfie_url) {
        riskScore += 20
        riskFactors.push('Missing selfie verification')
      }

      // Check for suspicious patterns
      const phoneNumber = application.phone_number
      if (phoneNumber && phoneNumber.includes('+1') && !application.address?.includes('United States')) {
        riskScore += 15
        riskFactors.push('Phone/address mismatch')
      }

      // Determine recommendation
      let recommendation: 'approve' | 'reject' | 'manual_review'
      if (riskScore < 30) {
        recommendation = 'approve'
      } else if (riskScore > 70) {
        recommendation = 'reject'
      } else {
        recommendation = 'manual_review'
      }

      return {
        risk_score: Math.min(100, riskScore),
        risk_factors: riskFactors,
        recommendation
      }
    } catch (error) {
      console.error('Error calculating risk assessment:', error)
      return {
        risk_score: 50,
        risk_factors: ['Assessment error'],
        recommendation: 'manual_review' as const
      }
    }
  }

  const handleKYCAction = async (applicationId: string, action: 'approve' | 'reject', rejectionReason?: string) => {
    try {
      setActionLoading(true)

      const updateData: any = {
        status: action,
        reviewed_at: new Date().toISOString(),
        reviewed_by: user?.id
      }

      if (action === 'reject' && rejectionReason) {
        updateData.rejection_reason = rejectionReason
      }

      const { error } = await supabase
        .from('kyc_applications')
        .update(updateData)
        .eq('id', applicationId)

      if (error) {
        console.error('Supabase error:', {
          error,
          message: error.message,
          details: error.details,
          hint: error.hint,
          code: error.code,
          timestamp: new Date().toISOString()
        })
        const errorMessage = error.message || error.details || error.hint || 'Database operation failed'
        throw new Error(errorMessage)
      }

      // Also update the user's profile KYC status
      const application = applications.find(app => app.id === applicationId)
      if (application) {
        const { error: profileError } = await supabase
          .from('profiles')
          .update({
            kyc_status: action === 'approve' ? 'approved' : 'rejected',
            kyc_verified: action === 'approve',
            kyc_verified_at: action === 'approve' ? new Date().toISOString() : null
          })
          .eq('id', application.user_id)

        if (profileError) {
          console.error('Failed to update profile KYC status:', {
            error: profileError,
            message: profileError.message,
            details: profileError.details,
            hint: profileError.hint,
            code: profileError.code,
            timestamp: new Date().toISOString()
          })
          throw new Error(profileError.message || 'Failed to update user profile')
        }
      } else {
        console.error('Application not found for ID:', applicationId)
        throw new Error('Application not found')
      }

      const message = action === 'approve' ? 'KYC application approved successfully' : 'KYC application rejected'
      alert(message)
      await fetchKYCApplications()
      setShowModal(false)

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to perform action'
      console.error('Error performing KYC action:', err)
      alert(`Error: ${errorMessage}`)
    } finally {
      setActionLoading(false)
    }
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'pending':
        return 'bg-yellow-100 text-yellow-800'
      case 'approved':
        return 'bg-green-100 text-green-800'
      case 'rejected':
        return 'bg-red-100 text-red-800'
      default:
        return 'bg-gray-100 text-black'
    }
  }

  const getRiskColor = (riskScore: number) => {
    if (riskScore < 30) return 'text-green-600 bg-green-100'
    if (riskScore < 60) return 'text-yellow-600 bg-yellow-100'
    return 'text-red-600 bg-red-100'
  }

  const getRecommendationColor = (recommendation: string) => {
    switch (recommendation) {
      case 'approve':
        return 'text-green-600 bg-green-100'
      case 'reject':
        return 'text-red-600 bg-red-100'
      case 'manual_review':
        return 'text-yellow-600 bg-yellow-100'
      default:
        return 'text-black bg-gray-100'
    }
  }

  const filteredApplications = applications.filter(application =>
    application.user_profile?.username?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    application.user_profile?.email?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    application.first_name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    application.last_name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    application.document_number?.toLowerCase().includes(searchTerm.toLowerCase())
  )

  if (loading) {
    return <PageLoadingSpinner />
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-black">KYC Management</h1>
              <p className="text-black mt-2">Review and manage KYC applications with risk assessment</p>
            </div>
            <Link
              href="/admin"
              className="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition-colors"
            >
              ← Back to Dashboard
            </Link>
          </div>
        </div>

        {/* KYC Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">KYC Overview</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{kycMetrics.totalApplications}</div>
              <div className="text-sm text-black">Total Applications</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-yellow-600">{kycMetrics.pendingApplications}</div>
              <div className="text-sm text-black">Pending Review</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-green-600">{kycMetrics.approvedApplications}</div>
              <div className="text-sm text-black">Approved</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-red-600">{kycMetrics.rejectedApplications}</div>
              <div className="text-sm text-black">Rejected</div>
            </div>
          </div>
        </div>

        {/* Additional Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Processing Metrics</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{kycMetrics.averageProcessingTime.toFixed(1)}d</div>
              <div className="text-sm text-black">Avg Processing Time</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-green-600">{kycMetrics.approvalRate.toFixed(1)}%</div>
              <div className="text-sm text-black">Approval Rate</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-orange-600">{kycMetrics.highRiskApplications}</div>
              <div className="text-sm text-black">High Risk</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-red-600">{kycMetrics.suspiciousApplications}</div>
              <div className="text-sm text-black">Suspicious</div>
            </div>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <input
                type="text"
                placeholder="Search applications..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>
            <div className="flex gap-2">
              {['all', 'pending', 'approved', 'rejected'].map((status) => (
                <button
                  key={status}
                  onClick={() => setSelectedStatus(status)}
                  className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                    selectedStatus === status
                      ? 'bg-blue-600 text-white'
                      : 'bg-gray-100 text-black hover:bg-gray-200'
                  }`}
                >
                  {status.charAt(0).toUpperCase() + status.slice(1)}
                </button>
              ))}
            </div>
          </div>
        </div>

        {/* Applications Table */}
        <div className="bg-white rounded-lg shadow-sm overflow-hidden">
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Applicant</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Document Info</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Risk Assessment</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Status</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Submitted</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Actions</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {filteredApplications.map((application) => (
                  <tr key={application.id} className="hover:bg-gray-50">
                    <td className="px-6 py-4">
                      <div className="flex items-center">
                        <div className="w-8 h-8 bg-gradient-to-br from-pink-500 via-purple-500 to-white rounded-full flex items-center justify-center mr-3">
                          <span className="text-white text-sm font-bold">
                            {application.first_name?.charAt(0).toUpperCase() || 'U'}
                          </span>
                        </div>
                        <div>
                          <div className="text-sm font-medium text-black">
                            {application.first_name} {application.last_name}
                          </div>
                          <div className="text-sm text-black">{application.user_profile?.username || 'Unknown'}</div>
                          <div className="text-sm text-black">{application.user_profile?.email}</div>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                      <div className="space-y-1">
                        <div className="text-black">{application.document_type}</div>
                        <div className="text-black">#{application.document_number}</div>
                        <div className="text-black">{application.phone_number}</div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                      <div className="space-y-1">
                        <div className="flex items-center">
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getRiskColor(application.risk_assessment?.risk_score || 0)}`}>
                            Risk: {application.risk_assessment?.risk_score || 0}/100
                          </span>
                        </div>
                        <div className="flex items-center">
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getRecommendationColor(application.risk_assessment?.recommendation || 'manual_review')}`}>
                            {application.risk_assessment?.recommendation?.replace('_', ' ') || 'Manual Review'}
                          </span>
                        </div>
                        <div className="text-xs text-black">
                          {application.risk_assessment?.risk_factors?.length || 0} risk factors
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(application.status)}`}>
                        {application.status}
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                      <div>
                        <div className="text-black">{new Date(application.submitted_at).toLocaleDateString()}</div>
                        <div className="text-black">{new Date(application.submitted_at).toLocaleTimeString()}</div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                      <button
                        onClick={() => {
                          setSelectedApplication(application)
                          setShowModal(true)
                        }}
                        className="text-blue-600 hover:text-blue-900"
                      >
                        Review
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>

        {filteredApplications.length === 0 && (
          <div className="text-center py-12">
            <p className="text-black">No KYC applications found matching your criteria.</p>
          </div>
        )}

        {/* KYC Review Modal */}
        {showModal && selectedApplication && (
          <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
            <div className="relative top-10 mx-auto p-5 border w-11/12 max-w-4xl shadow-lg rounded-md bg-white">
              <div className="mt-3">
                <h3 className="text-lg font-medium text-black mb-4">
                  KYC Review: {selectedApplication.first_name} {selectedApplication.last_name}
                </h3>
                
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                  {/* Applicant Information */}
                  <div className="space-y-4">
                    <h4 className="font-medium text-black">Applicant Information</h4>
                    <div className="bg-gray-50 p-4 rounded-lg space-y-2">
                      <div><span className="font-medium text-black">Name:</span> <span className="text-black">{selectedApplication.first_name} {selectedApplication.last_name}</span></div>
                      <div><span className="font-medium text-black">Email:</span> <span className="text-black">{selectedApplication.user_profile?.email}</span></div>
                      <div><span className="font-medium text-black">Phone:</span> <span className="text-black">{selectedApplication.phone_number}</span></div>
                      <div><span className="font-medium text-black">DOB:</span> <span className="text-black">{selectedApplication.date_of_birth}</span></div>
                      <div><span className="font-medium text-black">Address:</span> <span className="text-black">{selectedApplication.address}</span></div>
                      <div><span className="font-medium text-black">Document:</span> <span className="text-black">{selectedApplication.document_type} #{selectedApplication.document_number}</span></div>
                    </div>

                    {/* Risk Assessment */}
                    <h4 className="font-medium text-black">Risk Assessment</h4>
                    <div className="bg-gray-50 p-4 rounded-lg space-y-2">
                      <div className="flex items-center justify-between">
                        <span className="font-medium text-black">Risk Score:</span>
                        <span className={`px-2 py-1 text-xs font-semibold rounded-full ${getRiskColor(selectedApplication.risk_assessment?.risk_score || 0)}`}>
                          {selectedApplication.risk_assessment?.risk_score || 0}/100
                        </span>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="font-medium text-black">Recommendation:</span>
                        <span className={`px-2 py-1 text-xs font-semibold rounded-full ${getRecommendationColor(selectedApplication.risk_assessment?.recommendation || 'manual_review')}`}>
                          {selectedApplication.risk_assessment?.recommendation?.replace('_', ' ') || 'Manual Review'}
                        </span>
                      </div>
                      <div>
                        <span className="font-medium text-black">Risk Factors:</span>
                        <ul className="mt-1 text-sm text-black">
                          {selectedApplication.risk_assessment?.risk_factors?.map((factor, index) => (
                            <li key={index}>• {factor}</li>
                          ))}
                        </ul>
                      </div>
                    </div>
                  </div>

                  {/* Documents */}
                  <div className="space-y-4">
                    <h4 className="font-medium text-black">Documents</h4>
                    <div className="space-y-3">
                      {selectedApplication.document_front_url ? (
                        <div>
                          <div className="font-medium text-sm text-black">Document Front</div>
                          <img 
                            src={selectedApplication.document_front_url} 
                            alt="Document Front"
                            className="w-full h-48 object-contain border border-gray-300 rounded-lg bg-gray-50"
                            onError={(e) => {
                              const target = e.target as HTMLImageElement;
                              target.style.display = 'none';
                              const parent = target.parentElement;
                              if (parent) {
                                parent.innerHTML = '<div class="w-full h-48 border border-gray-300 rounded-lg bg-gray-100 flex items-center justify-center text-black">Image not available</div>';
                              }
                            }}
                          />
                        </div>
                      ) : (
                        <div>
                          <div className="font-medium text-sm text-black">Document Front</div>
                          <div className="w-full h-48 border border-gray-300 rounded-lg bg-gray-100 flex items-center justify-center text-black">
                            No document front uploaded
                          </div>
                        </div>
                      )}
                      
                      {selectedApplication.document_back_url ? (
                        <div>
                          <div className="font-medium text-sm text-black">Document Back</div>
                          <img 
                            src={selectedApplication.document_back_url} 
                            alt="Document Back"
                            className="w-full h-48 object-contain border border-gray-300 rounded-lg bg-gray-50"
                            onError={(e) => {
                              const target = e.target as HTMLImageElement;
                              target.style.display = 'none';
                              const parent = target.parentElement;
                              if (parent) {
                                parent.innerHTML = '<div class="w-full h-48 border border-gray-300 rounded-lg bg-gray-100 flex items-center justify-center text-black">Image not available</div>';
                              }
                            }}
                          />
                        </div>
                      ) : (
                        <div>
                          <div className="font-medium text-sm text-black">Document Back</div>
                          <div className="w-full h-48 border border-gray-300 rounded-lg bg-gray-100 flex items-center justify-center text-black">
                            No document back uploaded
                          </div>
                        </div>
                      )}
                      
                      {selectedApplication.selfie_url ? (
                        <div>
                          <div className="font-medium text-sm text-black">Selfie Verification</div>
                          <img 
                            src={selectedApplication.selfie_url} 
                            alt="Selfie"
                            className="w-full h-48 object-contain border border-gray-300 rounded-lg bg-gray-50"
                            onError={(e) => {
                              const target = e.target as HTMLImageElement;
                              target.style.display = 'none';
                              const parent = target.parentElement;
                              if (parent) {
                                parent.innerHTML = '<div class="w-full h-48 border border-gray-300 rounded-lg bg-gray-100 flex items-center justify-center text-black">Image not available</div>';
                              }
                            }}
                          />
                        </div>
                      ) : (
                        <div>
                          <div className="font-medium text-sm text-black">Selfie Verification</div>
                          <div className="w-full h-48 border border-gray-300 rounded-lg bg-gray-100 flex items-center justify-center text-black">
                            No selfie uploaded
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                {/* Actions */}
                <div className="mt-6 flex justify-between">
                  <button
                    onClick={() => setShowModal(false)}
                    className="bg-gray-300 text-black px-4 py-2 rounded text-sm hover:bg-gray-400"
                  >
                    Close
                  </button>
                  
                  {selectedApplication.status === 'pending' && (
                    <div className="flex gap-2">
                      <button
                        onClick={() => handleKYCAction(selectedApplication.id, 'approve')}
                        disabled={actionLoading}
                        className="bg-green-600 text-white px-4 py-2 rounded text-sm hover:bg-green-700 disabled:opacity-50"
                      >
                        {actionLoading ? <ButtonLoadingSpinner /> : 'Approve'}
                      </button>
                      <button
                        onClick={() => {
                          const reason = prompt('Enter rejection reason:')
                          if (reason) {
                            handleKYCAction(selectedApplication.id, 'reject', reason)
                          }
                        }}
                        disabled={actionLoading}
                        className="bg-red-600 text-white px-4 py-2 rounded text-sm hover:bg-red-700 disabled:opacity-50"
                      >
                        {actionLoading ? <ButtonLoadingSpinner /> : 'Reject'}
                      </button>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}