'use client'

import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { PageLoadingSpinner } from '@/components/ui/LoadingSpinner'
import { useAdminMarketsOptimized } from '@/lib/hooks/useAdminMarketsOptimized'
import AdminMarketsTableEnhanced from '@/components/admin/AdminMarketsTableEnhanced'
import Link from 'next/link'

export default function AdminMarkets() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <AdminMarketsContent user={user} />
    </FastSessionWrapper>
  )
}

function AdminMarketsContent({ user }: { user: any }) {
  const {
    markets,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    setSearchTerm,
    loadMore,
    refresh,
    filterByStatus
  } = useAdminMarketsOptimized()

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  if (loading && markets.length === 0) {
    return <PageLoadingSpinner />
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="max-w-md mx-auto text-center">
          <div className="bg-red-50 border border-red-200 rounded-lg p-6">
            <div className="text-red-600 text-4xl mb-4">⚠️</div>
            <h2 className="text-xl font-semibold text-red-800 mb-2">Error Loading Markets</h2>
            <p className="text-red-600 mb-4">{error}</p>
            <button
              onClick={refresh}
              className="bg-red-600 text-white px-4 py-2 rounded-lg hover:bg-red-700 transition-colors"
            >
              Try Again
            </button>
          </div>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-black">Market Management</h1>
              <p className="text-black mt-2">Monitor and manage all markets with optimized performance</p>
            </div>
            <Link
              href="/admin"
              className="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition-colors"
            >
              ← Back to Dashboard
            </Link>
          </div>
        </div>

        {/* Market Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Market Overview</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{metrics.totalMarkets}</div>
              <div className="text-sm text-black">Total Markets</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-green-600">{metrics.activeMarkets}</div>
              <div className="text-sm text-black">Active Markets</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-purple-600">{metrics.resolvedMarkets}</div>
              <div className="text-sm text-black">Resolved Markets</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-orange-600">{formatCurrency(metrics.totalVolume)}</div>
              <div className="text-sm text-black">Total Volume</div>
            </div>
          </div>
        </div>

        {/* Performance Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Performance Metrics</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{formatCurrency(metrics.averageVolume)}</div>
              <div className="text-sm text-black">Average Volume</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-green-600">{formatCurrency(metrics.totalEarnings)}</div>
              <div className="text-sm text-black">Total Earnings</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-purple-600">{metrics.resolutionRate.toFixed(1)}%</div>
              <div className="text-sm text-black">Resolution Rate</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-orange-600">Live</div>
              <div className="text-sm text-black">Data Updates</div>
            </div>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <input
                type="text"
                placeholder="Search markets..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>
            <div className="flex gap-2">
              {['all', 'active', 'resolved', 'closed'].map((status) => (
                <button
                  key={status}
                  onClick={() => filterByStatus(status)}
                  className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                    'bg-blue-600 text-white'
                  }`}
                >
                  {status.charAt(0).toUpperCase() + status.slice(1)}
                </button>
              ))}
            </div>
          </div>
        </div>

        {/* Markets Table */}
        <AdminMarketsTableEnhanced
          markets={markets}
          loading={loading}
          hasMore={hasMore}
          onLoadMore={loadMore}
          onRefresh={refresh}
        />
      </div>
    </div>
  )
}