'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { BettingEngine } from '@/lib/betting'
import { LoadingSpinner, PageLoadingSpinner } from '@/components/ui/LoadingSpinner'
import Link from 'next/link'

interface AdminStats {
  totalUsers: number
  totalMarkets: number
  totalBets: number
  totalVolume: number
  activeMarkets: number
  pendingMarkets: number
  totalRevenue: number
  averageBetSize: number
  todayUsers: number
  todayVolume: number
  totalLiquidity: number
  seededVolume: number
  liveVolume: number
  platformWinRate: number
  averageMarketLifetime: number
  totalPayouts: number
}

interface RecentActivity {
  id: string
  type: 'user_signup' | 'market_created' | 'bet_placed' | 'market_resolved' | 'liquidity_added' | 'market_seeded'
  description: string
  timestamp: string
  metadata?: any
}

interface AlgorithmMetrics {
  totalLiquidity: number
  seededVolume: number
  liveVolume: number
  averagePriceVolatility: number
  liquidityUtilization: number
  marketEfficiency: number
  priceAccuracy: number
}

export default function AdminDashboard() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <AdminDashboardContent user={user} />
    </FastSessionWrapper>
  )
}

function AdminDashboardContent({ user }: { user: any }) {
  const [stats, setStats] = useState<AdminStats>({
    totalUsers: 0,
    totalMarkets: 0,
    totalBets: 0,
    totalVolume: 0,
    activeMarkets: 0,
    pendingMarkets: 0,
    totalRevenue: 0,
    averageBetSize: 0,
    todayUsers: 0,
    todayVolume: 0,
    totalLiquidity: 0,
    seededVolume: 0,
    liveVolume: 0,
    platformWinRate: 0,
    averageMarketLifetime: 0,
    totalPayouts: 0
  })

  const [algorithmMetrics, setAlgorithmMetrics] = useState<AlgorithmMetrics>({
    totalLiquidity: 0,
    seededVolume: 0,
    liveVolume: 0,
    averagePriceVolatility: 0,
    liquidityUtilization: 0,
    marketEfficiency: 0,
    priceAccuracy: 0
  })

  const [recentActivity, setRecentActivity] = useState<RecentActivity[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    fetchAdminData()
  }, [])

  const fetchAdminData = async () => {
    try {
      setLoading(true)
      setError(null)

      const now = new Date()
      const today = new Date(now.getFullYear(), now.getMonth(), now.getDate())

      // Fetch comprehensive data in parallel
      const [
        usersData,
        marketsData,
        betsData,
        algorithmData,
        activityData
      ] = await Promise.all([
        // Users data
        Promise.all([
          supabase.from('profiles').select('id, created_at', { count: 'exact', head: true }),
          supabase.from('profiles').select('id', { count: 'exact', head: true }).gte('created_at', today.toISOString()),
          supabase.from('bets').select('user_id').gte('created_at', today.toISOString())
        ]),

        // Markets data with algorithm integration
        Promise.all([
          supabase.from('markets').select('id, created_at, status, resolution_date, total_liquidity, seeded_volume, live_volume', { count: 'exact', head: true }),
          supabase.from('markets').select('id', { count: 'exact', head: true }).eq('status', 'active'),
          supabase.from('markets').select('id', { count: 'exact', head: true }).eq('status', 'pending'),
          supabase.from('markets').select('created_at, resolution_date').eq('status', 'resolved'),
          supabase.from('markets').select('admin_fee, creator_earnings, total_liquidity, seeded_volume, live_volume')
        ]),

        // Bets data with algorithm metrics
        Promise.all([
          supabase.from('bets').select('id, amount, status, potential_payout, created_at', { count: 'exact', head: true }),
          supabase.from('bets').select('amount').gte('created_at', today.toISOString()),
          supabase.from('bets').select('amount, status, potential_payout, seeded_amount, live_amount')
        ]),

        // Algorithm-specific data
        Promise.all([
          supabase.from('markets').select('total_liquidity, seeded_volume, live_volume, yes_price, no_price, created_at'),
          supabase.from('bets').select('seeded_amount, live_amount, outcome, created_at'),
          supabase.from('price_history').select('yes_price, no_price, created_at').order('created_at', { ascending: false }).limit(1000)
        ]),

        // Recent activity data
        fetchRecentActivity()
      ])

      // Process users data
      const [totalUsers, newUsersToday, activeUsersToday] = usersData
      const uniqueActiveUsersToday = new Set(activeUsersToday.data?.map(bet => bet.user_id) || []).size

      // Process markets data
      const [totalMarkets, activeMarkets, pendingMarkets, resolvedMarkets, revenueData] = marketsData
      const averageMarketLifetime = resolvedMarkets.data ? 
        resolvedMarkets.data.reduce((sum, market) => {
          const created = new Date(market.created_at)
          const resolved = new Date(market.resolution_date)
          return sum + (resolved.getTime() - created.getTime()) / (1000 * 60 * 60 * 24)
        }, 0) / resolvedMarkets.data.length : 0

      // Process bets data
      const [totalBets, betsToday, allBets] = betsData
      const volumeToday = betsToday.data?.reduce((sum, bet) => sum + bet.amount, 0) || 0
      const totalVolume = allBets.data?.reduce((sum, bet) => sum + bet.amount, 0) || 0
      const totalPayouts = allBets.data?.reduce((sum, bet) => sum + (bet.potential_payout || 0), 0) || 0
      const wonBets = allBets.data?.filter(bet => bet.status === 'won').length || 0
      const platformWinRate = totalBets.count ? (wonBets / totalBets.count) * 100 : 0

      // Process algorithm data
      const [marketsWithLiquidity, betsWithSeeding, priceHistory] = algorithmData
      const totalLiquidity = marketsWithLiquidity.data?.reduce((sum, market) => sum + (market.total_liquidity || 0), 0) || 0
      const seededVolume = marketsWithLiquidity.data?.reduce((sum, market) => sum + (market.seeded_volume || 0), 0) || 0
      const liveVolume = marketsWithLiquidity.data?.reduce((sum, market) => sum + (market.live_volume || 0), 0) || 0

      // Calculate algorithm metrics
      const algorithmMetrics = await calculateAlgorithmMetrics(marketsWithLiquidity.data || [], priceHistory.data || [])

      // Process revenue data
      const totalRevenue = revenueData.data?.reduce((sum, market) => sum + (market.admin_fee || 0), 0) || 0

      setStats({
        totalUsers: totalUsers.count || 0,
        totalMarkets: totalMarkets.count || 0,
        totalBets: totalBets.count || 0,
        totalVolume,
        activeMarkets: activeMarkets.count || 0,
        pendingMarkets: pendingMarkets.count || 0,
        totalRevenue,
        averageBetSize: totalBets.count ? totalVolume / totalBets.count : 0,
        todayUsers: newUsersToday.count || 0,
        todayVolume: volumeToday,
        totalLiquidity,
        seededVolume,
        liveVolume,
        platformWinRate,
        averageMarketLifetime,
        totalPayouts
      })

      setAlgorithmMetrics(algorithmMetrics)
      setRecentActivity(activityData)

    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch admin data')
      console.error('Error fetching admin data:', err)
    } finally {
      setLoading(false)
    }
  }

  const calculateAlgorithmMetrics = async (markets: any[], priceHistory: any[]): Promise<AlgorithmMetrics> => {
    try {
      // Calculate price volatility
      let totalVolatility = 0
      let volatilityCount = 0

      if (priceHistory && priceHistory.length > 1) {
        for (let i = 1; i < priceHistory.length; i++) {
          const prev = priceHistory[i - 1]
          const curr = priceHistory[i]
          const volatility = Math.abs((curr.yes_price || 0.5) - (prev.yes_price || 0.5))
          totalVolatility += volatility
          volatilityCount++
        }
      }

      const averagePriceVolatility = volatilityCount > 0 ? totalVolatility / volatilityCount : 0

      // Calculate liquidity utilization
      const totalLiquidity = markets?.reduce((sum, market) => sum + (market.total_liquidity || 0), 0) || 0
      const totalSeededVolume = markets?.reduce((sum, market) => sum + (market.seeded_volume || 0), 0) || 0
      const liquidityUtilization = totalLiquidity > 0 ? (totalSeededVolume / totalLiquidity) * 100 : 0

      // Calculate market efficiency (based on price accuracy and volatility)
      const marketEfficiency = Math.max(0, 100 - (averagePriceVolatility * 200))

      // Calculate price accuracy (how close prices are to 50/50 when markets are balanced)
      const balancedMarkets = markets?.filter(market => 
        Math.abs((market.yes_price || 0.5) - 0.5) < 0.1
      ) || []
      const priceAccuracy = markets?.length > 0 ? (balancedMarkets.length / markets.length) * 100 : 0

      return {
        totalLiquidity,
        seededVolume: totalSeededVolume,
        liveVolume: markets?.reduce((sum, market) => sum + (market.live_volume || 0), 0) || 0,
        averagePriceVolatility,
        liquidityUtilization,
        marketEfficiency,
        priceAccuracy
      }
    } catch (error) {
      console.error('Error calculating algorithm metrics:', error)
      return {
        totalLiquidity: 0,
        seededVolume: 0,
        liveVolume: 0,
        averagePriceVolatility: 0,
        liquidityUtilization: 0,
        marketEfficiency: 0,
        priceAccuracy: 0
      }
    }
  }

  const fetchRecentActivity = async (): Promise<RecentActivity[]> => {
    try {
      // Fetch recent users
      const { data: recentUsers } = await supabase
        .from('profiles')
        .select('id, username, email, created_at')
        .order('created_at', { ascending: false })
        .limit(3)

      // Fetch recent markets
      const { data: recentMarkets } = await supabase
        .from('markets')
        .select('id, title, status, created_at, seeded_volume, total_liquidity')
        .order('created_at', { ascending: false })
        .limit(3)

      // Fetch recent bets with algorithm data
      const { data: recentBets } = await supabase
        .from('bets')
        .select(`
          id, 
          amount, 
          outcome, 
          created_at,
          market_id,
          seeded_amount,
          live_amount,
          markets!inner(id, title)
        `)
        .order('created_at', { ascending: false })
        .limit(3)

      // Combine and format activity
      const activities: RecentActivity[] = []

      recentUsers?.forEach(user => {
        activities.push({
          id: `user_${user.id}`,
          type: 'user_signup',
          description: `New user registered: ${user.username || user.email}`,
          timestamp: user.created_at
        })
      })

      recentMarkets?.forEach(market => {
        const activityType = market.seeded_volume > 0 ? 'market_seeded' : 'market_created'
        const description = market.seeded_volume > 0 
          ? `Market seeded: ${market.title} ($${market.seeded_volume})`
          : `New market created: ${market.title}`
        
        activities.push({
          id: `market_${market.id}`,
          type: activityType,
          description,
          timestamp: market.created_at,
          metadata: { 
            status: market.status, 
            seeded_volume: market.seeded_volume,
            total_liquidity: market.total_liquidity
          }
        })
      })

      recentBets?.forEach(bet => {
        const marketTitle = (bet.markets as any)?.title || 'Unknown Market'
        const betType = bet.seeded_amount > 0 ? 'seeded' : 'live'
        const amount = bet.seeded_amount > 0 ? bet.seeded_amount : bet.amount
        
        activities.push({
          id: `bet_${bet.id}`,
          type: 'bet_placed',
          description: `${betType.charAt(0).toUpperCase() + betType.slice(1)} bet: $${amount} on ${bet.outcome} in ${marketTitle}`,
          timestamp: bet.created_at,
          metadata: { 
            amount: bet.amount, 
            outcome: bet.outcome,
            seeded_amount: bet.seeded_amount,
            live_amount: bet.live_amount
          }
        })
      })

      // Sort by timestamp and take most recent
      activities.sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime())
      return activities.slice(0, 10)

    } catch (error) {
      console.error('Error fetching recent activity:', error)
      return []
    }
  }

  if (loading) {
    return <PageLoadingSpinner />
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h2 className="text-2xl font-bold text-black mb-4">Error Loading Dashboard</h2>
          <p className="text-black mb-4">{error}</p>
          <button
            onClick={fetchAdminData}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700"
          >
            Retry
          </button>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-black">Admin Dashboard</h1>
          <p className="text-black mt-2">Comprehensive platform management with algorithm insights</p>
        </div>

        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <StatCard
            title="Total Users"
            value={stats.totalUsers.toLocaleString()}
            change={`+${stats.todayUsers} today`}
            icon="👥"
            color="blue"
          />
          <StatCard
            title="Total Markets"
            value={stats.totalMarkets.toLocaleString()}
            change={`${stats.activeMarkets} active, ${stats.pendingMarkets} pending`}
            icon="📊"
            color="green"
          />
          <StatCard
            title="Total Volume"
            value={`$${stats.totalVolume.toLocaleString()}`}
            change={`$${stats.todayVolume.toLocaleString()} today`}
            icon="💰"
            color="purple"
          />
          <StatCard
            title="Platform Win Rate"
            value={`${stats.platformWinRate.toFixed(1)}%`}
            change={`${stats.totalBets} total bets`}
            icon="🎯"
            color="orange"
          />
        </div>

        {/* Algorithm Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Algorithm Performance</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            <MetricCard
              title="Total Liquidity"
              value={`$${algorithmMetrics.totalLiquidity.toLocaleString()}`}
              description="Available for trading"
            />
            <MetricCard
              title="Seeded Volume"
              value={`$${algorithmMetrics.seededVolume.toLocaleString()}`}
              description="Initial market seeding"
            />
            <MetricCard
              title="Live Volume"
              value={`$${algorithmMetrics.liveVolume.toLocaleString()}`}
              description="User trading volume"
            />
            <MetricCard
              title="Market Efficiency"
              value={`${algorithmMetrics.marketEfficiency.toFixed(1)}%`}
              description="Price accuracy & stability"
            />
          </div>
        </div>

        {/* Quick Actions */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
          <div className="bg-white rounded-lg shadow-sm p-6">
            <h2 className="text-xl font-semibold text-black mb-4">Admin Management</h2>
            <div className="grid grid-cols-2 gap-4">
              <Link
                href="/admin/markets"
                className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="text-2xl mr-3">📊</div>
                <div>
                  <div className="font-medium text-black">Manage Markets</div>
                  <div className="text-sm text-black">Approve, resolve, pause</div>
                </div>
              </Link>
              <Link
                href="/admin/users"
                className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="text-2xl mr-3">👥</div>
                <div>
                  <div className="font-medium text-black">Manage Users</div>
                  <div className="text-sm text-black">Ban, promote, KYC</div>
                </div>
              </Link>
              <Link
                href="/admin/bets"
                className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="text-2xl mr-3">🎯</div>
                <div>
                  <div className="font-medium text-black">Monitor Bets</div>
                  <div className="text-sm text-black">Track, resolve, refund</div>
                </div>
              </Link>
              <Link
                href="/admin/kyc"
                className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="text-2xl mr-3">🔐</div>
                <div>
                  <div className="font-medium text-black">KYC Management</div>
                  <div className="text-sm text-black">Review, approve, reject</div>
                </div>
              </Link>
            </div>
          </div>

          <div className="bg-white rounded-lg shadow-sm p-6">
            <h2 className="text-xl font-semibold text-black mb-4">Analytics & Settings</h2>
            <div className="grid grid-cols-2 gap-4">
              <Link
                href="/admin/analytics"
                className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="text-2xl mr-3">📈</div>
                <div>
                  <div className="font-medium text-black">Analytics</div>
                  <div className="text-sm text-black">Platform insights</div>
                </div>
              </Link>
              <Link
                href="/admin/settings"
                className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="text-2xl mr-3">⚙️</div>
                <div>
                  <div className="font-medium text-black">Algorithm Settings</div>
                  <div className="text-sm text-black">Configure, optimize</div>
                </div>
              </Link>
              <Link
                href="/admin/transactions"
                className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="text-2xl mr-3">💳</div>
                <div>
                  <div className="font-medium text-black">Transactions</div>
                  <div className="text-sm text-black">Monitor payments</div>
                </div>
              </Link>
              <Link
                href="/admin/resolve-markets"
                className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="text-2xl mr-3">✅</div>
                <div>
                  <div className="font-medium text-black">Resolve Markets</div>
                  <div className="text-sm text-black">Bulk resolution</div>
                </div>
              </Link>
              <Link
                href="/admin/payouts"
                className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
              >
                <div className="text-2xl mr-3">💰</div>
                <div>
                  <div className="font-medium text-black">Payout Management</div>
                  <div className="text-sm text-black">Monitor payouts</div>
                </div>
              </Link>
            </div>
          </div>
        </div>

        {/* Recent Activity */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Recent Activity</h2>
          <div className="space-y-3">
            {recentActivity.length > 0 ? (
              recentActivity.map((activity) => (
                <div key={activity.id} className="flex items-start space-x-3">
                  <div className="flex-shrink-0">
                    {activity.type === 'user_signup' && <span className="text-lg">👤</span>}
                    {activity.type === 'market_created' && <span className="text-lg">📊</span>}
                    {activity.type === 'market_seeded' && <span className="text-lg">🌱</span>}
                    {activity.type === 'bet_placed' && <span className="text-lg">🎯</span>}
                    {activity.type === 'market_resolved' && <span className="text-lg">✅</span>}
                  </div>
                  <div className="flex-1 min-w-0">
                    <p className="text-sm text-black">{activity.description}</p>
                    <p className="text-xs text-black">
                      {new Date(activity.timestamp).toLocaleString()}
                    </p>
                  </div>
                </div>
              ))
            ) : (
              <p className="text-black text-sm">No recent activity</p>
            )}
          </div>
        </div>

        {/* Performance Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6">
          <h2 className="text-xl font-semibold text-black mb-4">Performance Metrics</h2>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <div className="text-center">
              <div className="text-3xl font-bold text-blue-600">${stats.totalRevenue.toLocaleString()}</div>
              <div className="text-sm text-black">Total Revenue</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-green-600">${stats.averageBetSize.toFixed(2)}</div>
              <div className="text-sm text-black">Average Bet Size</div>
            </div>
            <div className="text-center">
              <div className="text-3xl font-bold text-purple-600">{stats.averageMarketLifetime.toFixed(1)}d</div>
              <div className="text-sm text-black">Avg Market Lifetime</div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}

function StatCard({ title, value, change, icon, color }: {
  title: string
  value: string
  change: string
  icon: string
  color: string
}) {
  const colorClasses = {
    blue: 'bg-blue-50 text-blue-600',
    green: 'bg-green-50 text-green-600',
    purple: 'bg-purple-50 text-purple-600',
    orange: 'bg-orange-50 text-orange-600'
  }

  return (
    <div className="bg-white rounded-lg shadow-sm p-6">
      <div className="flex items-center justify-between">
        <div>
          <p className="text-sm font-medium text-black">{title}</p>
          <p className="text-2xl font-bold text-black">{value}</p>
          <p className="text-sm text-black">{change}</p>
        </div>
        <div className={`p-3 rounded-full ${colorClasses[color as keyof typeof colorClasses]}`}>
          <span className="text-2xl">{icon}</span>
        </div>
      </div>
    </div>
  )
}

function MetricCard({ title, value, description }: {
  title: string
  value: string
  description: string
}) {
  return (
    <div className="text-center p-4 border border-gray-200 rounded-lg">
      <div className="text-2xl font-bold text-black">{value}</div>
      <div className="text-sm font-medium text-black">{title}</div>
      <div className="text-xs text-black">{description}</div>
    </div>
  )
}