'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { LoadingSpinner, PageLoadingSpinner } from '@/components/ui/LoadingSpinner'
import Link from 'next/link'

interface Payout {
  id: string
  market_id: string
  market_title: string
  user_id: string
  username: string
  email: string
  payout_type: 'winner_seeded' | 'winner_live' | 'creator_earnings' | 'admin_fee' | 'refund'
  amount: number
  multiplier: number | null
  status: 'pending' | 'processing' | 'completed' | 'failed' | 'cancelled'
  processed_at: string | null
  created_at: string
  transaction_id: string | null
}

interface PayoutSummary {
  total_payouts: number
  total_amount: number
  completed_payouts: number
  pending_payouts: number
  failed_payouts: number
  winner_count: number
  creator_earnings: number
  admin_fees: number
}

interface MarketPayoutSummary {
  market_id: string
  market_title: string
  total_winners: number
  total_payouts: number
  creator_earnings: number
  admin_fee: number
  winner_details: Array<{
    user_id: string
    username: string
    email: string
    seeded_payout: number
    live_payout: number
    total_payout: number
    multiplier: number
  }>
}

export default function AdminPayouts() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <AdminPayoutsContent user={user} />
    </FastSessionWrapper>
  )
}

function AdminPayoutsContent({ user }: { user: any }) {
  const [payouts, setPayouts] = useState<Payout[]>([])
  const [summary, setSummary] = useState<PayoutSummary>({
    total_payouts: 0,
    total_amount: 0,
    completed_payouts: 0,
    pending_payouts: 0,
    failed_payouts: 0,
    winner_count: 0,
    creator_earnings: 0,
    admin_fees: 0
  })
  const [marketSummaries, setMarketSummaries] = useState<MarketPayoutSummary[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [selectedTab, setSelectedTab] = useState<'overview' | 'payouts' | 'markets'>('overview')
  const [selectedPayoutType, setSelectedPayoutType] = useState<string>('all')
  const [selectedStatus, setSelectedStatus] = useState<string>('all')
  const [searchTerm, setSearchTerm] = useState('')

  useEffect(() => {
    fetchPayoutData()
  }, [])

  const fetchPayoutData = async () => {
    try {
      setLoading(true)
      setError(null)

      // Fetch all payouts
      const { data: payoutsData, error: payoutsError } = await supabase
        .from('payouts')
        .select(`
          *,
          markets!inner(title),
          profiles!inner(username, email)
        `)
        .order('created_at', { ascending: false })

      if (payoutsError) {
        console.error('Error fetching payouts:', payoutsError)
        // If payouts table doesn't exist, show setup message
        if (payoutsError.message.includes('relation "payouts" does not exist')) {
          setError('Payout system not set up. Please run the setup script first.')
          setLoading(false)
          return
        }
        throw payoutsError
      }

      // Transform payout data
      const transformedPayouts = (payoutsData || []).map(payout => ({
        id: payout.id,
        market_id: payout.market_id,
        market_title: (payout.markets as any)?.title || 'Unknown Market',
        user_id: payout.user_id,
        username: (payout.profiles as any)?.username || 'Unknown User',
        email: (payout.profiles as any)?.email || 'Unknown Email',
        payout_type: payout.payout_type,
        amount: payout.amount,
        multiplier: payout.multiplier,
        status: payout.status,
        processed_at: payout.processed_at,
        created_at: payout.created_at,
        transaction_id: payout.transaction_id
      }))

      setPayouts(transformedPayouts)

      // Calculate summary statistics
      const summaryData: PayoutSummary = {
        total_payouts: transformedPayouts.length,
        total_amount: transformedPayouts.reduce((sum, p) => sum + p.amount, 0),
        completed_payouts: transformedPayouts.filter(p => p.status === 'completed').length,
        pending_payouts: transformedPayouts.filter(p => p.status === 'pending').length,
        failed_payouts: transformedPayouts.filter(p => p.status === 'failed').length,
        winner_count: transformedPayouts.filter(p => p.payout_type.includes('winner')).length,
        creator_earnings: transformedPayouts
          .filter(p => p.payout_type === 'creator_earnings')
          .reduce((sum, p) => sum + p.amount, 0),
        admin_fees: transformedPayouts
          .filter(p => p.payout_type === 'admin_fee')
          .reduce((sum, p) => sum + p.amount, 0)
      }

      setSummary(summaryData)

      // Fetch market payout summaries
      await fetchMarketSummaries(transformedPayouts)

    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch payout data')
      console.error('Error fetching payout data:', err)
    } finally {
      setLoading(false)
    }
  }

  const fetchMarketSummaries = async (payoutsData: Payout[]) => {
    try {
      // Get unique market IDs
      const marketIds = [...new Set(payoutsData.map(p => p.market_id))]
      
      const marketSummaries: MarketPayoutSummary[] = []
      
      for (const marketId of marketIds) {
        const marketPayouts = payoutsData.filter(p => p.market_id === marketId)
        const marketTitle = marketPayouts[0]?.market_title || 'Unknown Market'
        
        const winnerDetails = marketPayouts
          .filter(p => p.payout_type.includes('winner'))
          .reduce((acc: any[], payout) => {
            const existing = acc.find(w => w.user_id === payout.user_id)
            if (existing) {
              if (payout.payout_type === 'winner_seeded') {
                existing.seeded_payout = payout.amount
              } else if (payout.payout_type === 'winner_live') {
                existing.live_payout = payout.amount
              }
              existing.total_payout = existing.seeded_payout + existing.live_payout
            } else {
              acc.push({
                user_id: payout.user_id,
                username: payout.username,
                email: payout.email,
                seeded_payout: payout.payout_type === 'winner_seeded' ? payout.amount : 0,
                live_payout: payout.payout_type === 'winner_live' ? payout.amount : 0,
                total_payout: payout.amount,
                multiplier: payout.multiplier || 0
              })
            }
            return acc
          }, [])

        const summary: MarketPayoutSummary = {
          market_id: marketId,
          market_title: marketTitle,
          total_winners: winnerDetails.length,
          total_payouts: marketPayouts.reduce((sum, p) => sum + p.amount, 0),
          creator_earnings: marketPayouts.find(p => p.payout_type === 'creator_earnings')?.amount || 0,
          admin_fee: marketPayouts.find(p => p.payout_type === 'admin_fee')?.amount || 0,
          winner_details: winnerDetails.sort((a, b) => b.total_payout - a.total_payout)
        }

        marketSummaries.push(summary)
      }

      setMarketSummaries(marketSummaries.sort((a, b) => b.total_payouts - a.total_payouts))

    } catch (err) {
      console.error('Error fetching market summaries:', err)
    }
  }

  const filteredPayouts = payouts.filter(payout => {
    const matchesType = selectedPayoutType === 'all' || payout.payout_type === selectedPayoutType
    const matchesStatus = selectedStatus === 'all' || payout.status === selectedStatus
    const matchesSearch = searchTerm === '' || 
      payout.market_title.toLowerCase().includes(searchTerm.toLowerCase()) ||
      payout.username.toLowerCase().includes(searchTerm.toLowerCase()) ||
      payout.email.toLowerCase().includes(searchTerm.toLowerCase())
    
    return matchesType && matchesStatus && matchesSearch
  })

  const getPayoutTypeColor = (type: string) => {
    switch (type) {
      case 'winner_seeded':
      case 'winner_live':
        return 'bg-green-100 text-green-800'
      case 'creator_earnings':
        return 'bg-blue-100 text-blue-800'
      case 'admin_fee':
        return 'bg-purple-100 text-purple-800'
      case 'refund':
        return 'bg-orange-100 text-orange-800'
      default:
        return 'bg-gray-100 text-gray-800'
    }
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'bg-green-100 text-green-800'
      case 'pending':
        return 'bg-yellow-100 text-yellow-800'
      case 'processing':
        return 'bg-blue-100 text-blue-800'
      case 'failed':
        return 'bg-red-100 text-red-800'
      case 'cancelled':
        return 'bg-gray-100 text-gray-800'
      default:
        return 'bg-gray-100 text-gray-800'
    }
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString()
  }

  if (loading) {
    return <PageLoadingSpinner />
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center max-w-md">
          <h2 className="text-2xl font-bold text-black mb-4">Payout System Not Set Up</h2>
          <p className="text-black mb-6">{error}</p>
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
            <h3 className="font-semibold text-black mb-2">To set up the payout system:</h3>
            <ol className="text-sm text-black text-left space-y-1">
              <li>1. Run the SQL setup script: <code className="bg-gray-100 px-1 rounded">setup-payout-system.sql</code></li>
              <li>2. Refresh this page</li>
              <li>3. Start monitoring payouts</li>
            </ol>
          </div>
          <button
            onClick={fetchPayoutData}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700"
          >
            Retry
          </button>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-black">Payout Management</h1>
          <p className="text-black mt-2">Monitor and manage all market payouts and winner distributions</p>
        </div>

        {/* Tabs */}
        <div className="mb-6">
          <div className="border-b border-gray-200">
            <nav className="-mb-px flex space-x-8">
              {[
                { id: 'overview', name: 'Overview', icon: '📊' },
                { id: 'payouts', name: 'All Payouts', icon: '💰' },
                { id: 'markets', name: 'Market Summaries', icon: '🏆' }
              ].map((tab) => (
                <button
                  key={tab.id}
                  onClick={() => setSelectedTab(tab.id as any)}
                  className={`flex items-center py-2 px-1 border-b-2 font-medium text-sm ${
                    selectedTab === tab.id
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  <span className="mr-2">{tab.icon}</span>
                  {tab.name}
                </button>
              ))}
            </nav>
          </div>
        </div>

        {/* Overview Tab */}
        {selectedTab === 'overview' && (
          <div className="space-y-6">
            {/* Summary Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <div className="bg-white rounded-lg shadow-sm p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-black">Total Payouts</p>
                    <p className="text-2xl font-bold text-black">{summary.total_payouts}</p>
                    <p className="text-sm text-black">{formatCurrency(summary.total_amount)}</p>
                  </div>
                  <div className="p-3 rounded-full bg-green-50">
                    <span className="text-2xl">💰</span>
                  </div>
                </div>
              </div>

              <div className="bg-white rounded-lg shadow-sm p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-black">Winners Funded</p>
                    <p className="text-2xl font-bold text-black">{summary.winner_count}</p>
                    <p className="text-sm text-black">Auto-funded accounts</p>
                  </div>
                  <div className="p-3 rounded-full bg-blue-50">
                    <span className="text-2xl">🏆</span>
                  </div>
                </div>
              </div>

              <div className="bg-white rounded-lg shadow-sm p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-black">Completed</p>
                    <p className="text-2xl font-bold text-black">{summary.completed_payouts}</p>
                    <p className="text-sm text-black">{summary.pending_payouts} pending</p>
                  </div>
                  <div className="p-3 rounded-full bg-green-50">
                    <span className="text-2xl">✅</span>
                  </div>
                </div>
              </div>

              <div className="bg-white rounded-lg shadow-sm p-6">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-sm font-medium text-black">Platform Revenue</p>
                    <p className="text-2xl font-bold text-black">{formatCurrency(summary.creator_earnings + summary.admin_fees)}</p>
                    <p className="text-sm text-black">Creators + Admin</p>
                  </div>
                  <div className="p-3 rounded-full bg-purple-50">
                    <span className="text-2xl">🏢</span>
                  </div>
                </div>
              </div>
            </div>

            {/* Recent Payouts */}
            <div className="bg-white rounded-lg shadow-sm">
              <div className="px-6 py-4 border-b border-gray-200">
                <h2 className="text-xl font-semibold text-black">Recent Payouts</h2>
              </div>
              <div className="overflow-x-auto">
                <table className="min-w-full divide-y divide-gray-200">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Market</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">User</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Type</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Amount</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Status</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Date</th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {payouts.slice(0, 10).map((payout) => (
                      <tr key={payout.id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="text-sm font-medium text-black truncate max-w-xs">
                            {payout.market_title}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div>
                            <div className="text-sm font-medium text-black">{payout.username}</div>
                            <div className="text-sm text-black">{payout.email}</div>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getPayoutTypeColor(payout.payout_type)}`}>
                            {payout.payout_type.replace('_', ' ')}
                          </span>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                          <div>
                            <div className="font-medium">{formatCurrency(payout.amount)}</div>
                            {payout.multiplier && (
                              <div className="text-xs text-black">{payout.multiplier.toFixed(2)}x</div>
                            )}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(payout.status)}`}>
                            {payout.status}
                          </span>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                          {formatDate(payout.created_at)}
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
          </div>
        )}

        {/* All Payouts Tab */}
        {selectedTab === 'payouts' && (
          <div className="space-y-6">
            {/* Filters */}
            <div className="bg-white rounded-lg shadow-sm p-6">
              <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div>
                  <label className="block text-sm font-medium text-black mb-2">Search</label>
                  <input
                    type="text"
                    placeholder="Search markets, users..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-black"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-black mb-2">Payout Type</label>
                  <select
                    value={selectedPayoutType}
                    onChange={(e) => setSelectedPayoutType(e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-black"
                  >
                    <option value="all">All Types</option>
                    <option value="winner_seeded">Winner Seeded</option>
                    <option value="winner_live">Winner Live</option>
                    <option value="creator_earnings">Creator Earnings</option>
                    <option value="admin_fee">Admin Fee</option>
                    <option value="refund">Refund</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-black mb-2">Status</label>
                  <select
                    value={selectedStatus}
                    onChange={(e) => setSelectedStatus(e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-black"
                  >
                    <option value="all">All Status</option>
                    <option value="completed">Completed</option>
                    <option value="pending">Pending</option>
                    <option value="processing">Processing</option>
                    <option value="failed">Failed</option>
                    <option value="cancelled">Cancelled</option>
                  </select>
                </div>
                <div className="flex items-end">
                  <button
                    onClick={fetchPayoutData}
                    className="w-full bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors"
                  >
                    Refresh
                  </button>
                </div>
              </div>
            </div>

            {/* Payouts Table */}
            <div className="bg-white rounded-lg shadow-sm overflow-hidden">
              <div className="overflow-x-auto">
                <table className="min-w-full divide-y divide-gray-200">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Market</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">User</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Type</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Amount</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Multiplier</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Status</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Processed</th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Created</th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {filteredPayouts.map((payout) => (
                      <tr key={payout.id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="text-sm font-medium text-black truncate max-w-xs">
                            {payout.market_title}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div>
                            <div className="text-sm font-medium text-black">{payout.username}</div>
                            <div className="text-sm text-black">{payout.email}</div>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getPayoutTypeColor(payout.payout_type)}`}>
                            {payout.payout_type.replace('_', ' ')}
                          </span>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-black">
                          {formatCurrency(payout.amount)}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                          {payout.multiplier ? payout.multiplier.toFixed(2) + 'x' : '-'}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(payout.status)}`}>
                            {payout.status}
                          </span>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                          {payout.processed_at ? formatDate(payout.processed_at) : '-'}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                          {formatDate(payout.created_at)}
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
              
              {filteredPayouts.length === 0 && (
                <div className="text-center py-12">
                  <p className="text-black">No payouts found matching your criteria.</p>
                </div>
              )}
            </div>
          </div>
        )}

        {/* Market Summaries Tab */}
        {selectedTab === 'markets' && (
          <div className="space-y-6">
            {marketSummaries.map((market) => (
              <div key={market.market_id} className="bg-white rounded-lg shadow-sm">
                <div className="px-6 py-4 border-b border-gray-200">
                  <div className="flex items-center justify-between">
                    <div>
                      <h3 className="text-lg font-medium text-black">{market.market_title}</h3>
                      <p className="text-sm text-black">Market ID: {market.market_id}</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-medium text-black">Total Payouts</p>
                      <p className="text-xl font-bold text-black">{formatCurrency(market.total_payouts)}</p>
                    </div>
                  </div>
                </div>
                
                <div className="p-6">
                  <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                    <div className="text-center">
                      <p className="text-2xl font-bold text-blue-600">{market.total_winners}</p>
                      <p className="text-sm text-black">Winners</p>
                    </div>
                    <div className="text-center">
                      <p className="text-2xl font-bold text-green-600">{formatCurrency(market.total_payouts)}</p>
                      <p className="text-sm text-black">Total Payouts</p>
                    </div>
                    <div className="text-center">
                      <p className="text-2xl font-bold text-purple-600">{formatCurrency(market.creator_earnings)}</p>
                      <p className="text-sm text-black">Creator Earnings</p>
                    </div>
                    <div className="text-center">
                      <p className="text-2xl font-bold text-orange-600">{formatCurrency(market.admin_fee)}</p>
                      <p className="text-sm text-black">Admin Fee</p>
                    </div>
                  </div>

                  {market.winner_details.length > 0 && (
                    <div>
                      <h4 className="text-md font-semibold text-black mb-4">Winner Details</h4>
                      <div className="overflow-x-auto">
                        <table className="min-w-full divide-y divide-gray-200">
                          <thead className="bg-gray-50">
                            <tr>
                              <th className="px-4 py-2 text-left text-xs font-medium text-black uppercase tracking-wider">User</th>
                              <th className="px-4 py-2 text-left text-xs font-medium text-black uppercase tracking-wider">Seeded Payout</th>
                              <th className="px-4 py-2 text-left text-xs font-medium text-black uppercase tracking-wider">Live Payout</th>
                              <th className="px-4 py-2 text-left text-xs font-medium text-black uppercase tracking-wider">Total Payout</th>
                              <th className="px-4 py-2 text-left text-xs font-medium text-black uppercase tracking-wider">Multiplier</th>
                            </tr>
                          </thead>
                          <tbody className="bg-white divide-y divide-gray-200">
                            {market.winner_details.map((winner) => (
                              <tr key={winner.user_id}>
                                <td className="px-4 py-2 whitespace-nowrap">
                                  <div>
                                    <div className="text-sm font-medium text-black">{winner.username}</div>
                                    <div className="text-sm text-black">{winner.email}</div>
                                  </div>
                                </td>
                                <td className="px-4 py-2 whitespace-nowrap text-sm text-black">
                                  {formatCurrency(winner.seeded_payout)}
                                </td>
                                <td className="px-4 py-2 whitespace-nowrap text-sm text-black">
                                  {formatCurrency(winner.live_payout)}
                                </td>
                                <td className="px-4 py-2 whitespace-nowrap text-sm font-medium text-black">
                                  {formatCurrency(winner.total_payout)}
                                </td>
                                <td className="px-4 py-2 whitespace-nowrap text-sm text-black">
                                  {winner.multiplier.toFixed(2)}x
                                </td>
                              </tr>
                            ))}
                          </tbody>
                        </table>
                      </div>
                    </div>
                  )}
                </div>
              </div>
            ))}

            {marketSummaries.length === 0 && (
              <div className="text-center py-12">
                <p className="text-black">No market payout summaries available.</p>
              </div>
            )}
          </div>
        )}
      </div>
    </div>
  )
}
