'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { BettingEngine } from '@/lib/betting'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

interface Market {
  id: string
  title: string
  description: string
  end_date: string
  status: string
  outcome: string | null
  total_volume: number
  yes_price: number
  no_price: number
  creator_id: string
  market_type: 'binary' | 'multi_outcome'
  outcomes?: Array<{
    id: string
    name: string
    price: number
    pool: number
    color?: string
    icon?: string
  }>
  profiles: {
    email: string
  }
}

export default function ResolveMarketsPage() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <ResolveMarketsContent user={user} />
    </FastSessionWrapper>
  )
}

function ResolveMarketsContent({ user }: { user: any }) {
  const router = useRouter()
  const [markets, setMarkets] = useState<Market[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState('')
  const [resolving, setResolving] = useState<string | null>(null)
  const [autoResolving, setAutoResolving] = useState(false)
  const [autoResolveResults, setAutoResolveResults] = useState<any>(null)

  const fetchMarkets = async () => {
    try {
      const { data, error } = await supabase
        .from('markets')
        .select(`
          id,
          title,
          description,
          end_date,
          status,
          outcome,
          total_volume,
          yes_price,
          no_price,
          creator_id,
          market_type,
          outcomes:market_outcomes (
            id,
            name,
            price,
            pool,
            color,
            icon
          ),
          profiles!markets_creator_id_fkey (
            email
          )
        `)
        .in('status', ['active', 'closed'])
        .order('end_date', { ascending: true })

      if (error) throw error

      // Transform the data to match our interface
      const transformedData = (data || []).map((market: any) => ({
        id: market.id,
        title: market.title,
        description: market.description,
        end_date: market.end_date,
        status: market.status,
        outcome: market.outcome,
        total_volume: market.total_volume,
        yes_price: market.yes_price,
        no_price: market.no_price,
        creator_id: market.creator_id,
        market_type: market.market_type || 'binary',
        outcomes: market.outcomes || [],
        profiles: market.profiles
      }))

      setMarkets(transformedData)
    } catch (err) {
      console.error('Error fetching markets:', err)
      setError('Failed to load markets')
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    if (user) {
      fetchMarkets()
    }
  }, [user])

  const handleResolveMarket = async (marketId: string, outcome: string) => {
    setResolving(marketId)
    try {
      const result = await BettingEngine.resolveMarket(marketId, outcome)

      if (!result.success) {
        throw new Error(result.error)
      }

      // Refresh the markets list
      await fetchMarkets()
    } catch (err) {
      console.error('Error resolving market:', err)
      setError(err instanceof Error ? err.message : 'Failed to resolve market')
    } finally {
      setResolving(null)
    }
  }

  const handleAutoResolve = async () => {
    setAutoResolving(true)
    setAutoResolveResults(null)
    setError('')
    
    try {
      const result = await BettingEngine.autoResolveExpiredMarkets(70) // 70% minimum confidence
      setAutoResolveResults(result)
      
      if (result.success) {
        // Refresh the markets list to show updated statuses
        await fetchMarkets()
      } else {
        setError(result.error || 'Auto-resolution failed')
      }
    } catch (err) {
      console.error('Error in auto-resolution:', err)
      setError(err instanceof Error ? err.message : 'Failed to auto-resolve markets')
    } finally {
      setAutoResolving(false)
    }
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    })
  }

  const formatPrice = (price: number) => {
    return (price * 100).toFixed(1) + '%'
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  const isMarketExpired = (endDate: string) => {
    return new Date() > new Date(endDate)
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <LoadingSpinner size="lg" />
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-6xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center space-x-4 mb-4">
            <Link href="/admin" className="text-black hover:text-black">
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
              </svg>
            </Link>
            <h1 className="text-3xl font-bold text-black">Resolve Markets</h1>
          </div>
          <p className="text-black">
            Resolve markets that have ended and distribute winnings using the betting algorithm.
          </p>
        </div>

        {error && (
          <div className={`mb-6 p-4 border rounded-lg ${
            error.includes('Payout system not set up') 
              ? 'bg-yellow-50 border-yellow-200' 
              : 'bg-red-50 border-red-200'
          }`}>
            <div className="flex items-center">
              <svg className={`w-5 h-5 mr-2 ${error.includes('Payout system not set up') ? 'text-yellow-400' : 'text-red-400'}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              <div className="flex-1">
                <p className={`text-sm ${error.includes('Payout system not set up') ? 'text-yellow-700' : 'text-red-700'}`}>
                  {error}
                </p>
                {error.includes('Payout system not set up') && (
                  <div className="mt-2 text-xs text-yellow-600">
                    <strong>To fix this:</strong>
                    <ol className="list-decimal list-inside mt-1 space-y-1">
                      <li>Go to your Supabase SQL editor</li>
                      <li>Run the <code className="bg-yellow-100 px-1 rounded">setup-payout-system.sql</code> script</li>
                      <li>Refresh this page</li>
                    </ol>
                  </div>
                )}
              </div>
            </div>
          </div>
        )}

        {/* Auto Resolve Section */}
        <div className="mb-8 bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <div>
              <h2 className="text-lg font-semibold text-black mb-2">Automatic Market Resolution</h2>
              <p className="text-black text-sm">
                Automatically resolve expired markets using AI algorithms based on betting patterns, prices, and volume.
              </p>
            </div>
            <button
              onClick={handleAutoResolve}
              disabled={autoResolving}
              className="px-6 py-3 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center"
            >
              {autoResolving ? (
                <>
                  <svg className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" fill="none" viewBox="0 0 24 24">
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                  </svg>
                  Auto-Resolving...
                </>
              ) : (
                <>
                  <svg className="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
                  </svg>
                  Auto Resolve Markets
                </>
              )}
            </button>
          </div>

          {/* Auto Resolve Results */}
          {autoResolveResults && (
            <div className="border-t pt-4">
              <h3 className="text-md font-semibold text-black mb-3">Auto-Resolution Results</h3>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                <div className="bg-blue-50 p-3 rounded-lg">
                  <div className="text-sm text-blue-600 font-medium">Processed</div>
                  <div className="text-lg font-bold text-blue-700">{autoResolveResults.processed}</div>
                </div>
                <div className="bg-green-50 p-3 rounded-lg">
                  <div className="text-sm text-green-600 font-medium">Resolved</div>
                  <div className="text-lg font-bold text-green-700">{autoResolveResults.resolved}</div>
                </div>
                <div className="bg-red-50 p-3 rounded-lg">
                  <div className="text-sm text-red-600 font-medium">Failed</div>
                  <div className="text-lg font-bold text-red-700">{autoResolveResults.failed}</div>
                </div>
              </div>

              {autoResolveResults.results && autoResolveResults.results.length > 0 && (
                <div className="space-y-2 max-h-64 overflow-y-auto">
                  {autoResolveResults.results.map((result: any, index: number) => (
                    <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg text-sm">
                      <div className="flex-1">
                        <div className="font-medium text-black truncate">{result.marketTitle}</div>
                        {result.outcome && (
                          <div className="text-black">
                            Outcome: <span className="font-medium">{result.outcome.toUpperCase()}</span> 
                            ({result.confidence.toFixed(1)}% confidence)
                          </div>
                        )}
                      </div>
                      <div className="ml-4">
                        {result.success ? (
                          <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">
                            ✅ Resolved
                          </span>
                        ) : (
                          <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">
                            ❌ Failed
                          </span>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>
          )}
        </div>

        {/* Markets List */}
        {markets.length === 0 ? (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8 text-center">
            <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg className="w-8 h-8 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
              </svg>
            </div>
            <h3 className="text-lg font-semibold text-black mb-2">No Markets to Resolve</h3>
            <p className="text-black">
              There are currently no markets that need resolution.
            </p>
          </div>
        ) : (
          <div className="space-y-6">
            {markets.map((market) => (
              <div key={market.id} className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                <div className="flex items-start justify-between mb-4">
                  <div className="flex-1">
                    <h3 className="text-lg font-semibold text-black mb-2">
                      {market.title}
                    </h3>
                    <p className="text-black text-sm mb-3">
                      {market.description}
                    </p>
                    <div className="flex items-center space-x-4 text-sm text-black">
                      <span>Created by: {market.profiles?.email || 'Unknown User'}</span>
                      <span>•</span>
                      <span>Ends: {formatDate(market.end_date)}</span>
                      <span>•</span>
                      <span>Volume: {formatCurrency(market.total_volume)}</span>
                    </div>
                  </div>
                  <Link 
                    href={`/markets/${market.id}`}
                    className="text-blue-600 hover:text-blue-700 text-sm font-medium"
                  >
                    View Market →
                  </Link>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {/* Market Status */}
                  <div className="space-y-3">
                    <div className="text-sm font-medium text-black mb-2">Market Status</div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Status:</span>
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        market.status === 'active' 
                          ? (isMarketExpired(market.end_date) ? 'bg-yellow-100 text-yellow-800' : 'bg-green-100 text-green-800')
                          : 'bg-gray-100 text-black'
                      }`}>
                        {market.status}
                        {market.status === 'active' && isMarketExpired(market.end_date) && ' (Expired)'}
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Current Prices:</span>
                      <div className="text-sm">
                        {market.market_type === 'binary' ? (
                          <>
                            <span className="text-green-600">YES {formatPrice(market.yes_price)}</span>
                            <span className="mx-2">•</span>
                            <span className="text-red-600">NO {formatPrice(market.no_price)}</span>
                          </>
                        ) : (
                          <div className="flex flex-wrap gap-2">
                            {market.outcomes?.map((outcome, index) => (
                              <span 
                                key={outcome.id}
                                className="text-sm"
                                style={{ color: outcome.color || '#6B7280' }}
                              >
                                {outcome.icon && <span className="mr-1">{outcome.icon}</span>}
                                {outcome.name} {formatPrice(outcome.price)}
                              </span>
                            ))}
                          </div>
                        )}
                      </div>
                    </div>
                    {market.outcome && (
                      <div className="flex items-center justify-between">
                        <span className="text-sm text-black">Resolved:</span>
                        <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                          market.outcome === 'yes' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                        }`}>
                          {market.outcome.toUpperCase()}
                        </span>
                      </div>
                    )}
                  </div>

                  {/* Resolution Actions */}
                  <div className="space-y-3">
                    <div className="text-sm font-medium text-black mb-2">Resolution</div>
                    {market.outcome ? (
                      <div className="text-sm text-green-600 bg-green-50 p-3 rounded-lg">
                        ✓ Market already resolved as {market.outcome.toUpperCase()}
                      </div>
                    ) : market.status === 'active' && isMarketExpired(market.end_date) ? (
                      <div className="space-y-2">
                        {market.market_type === 'binary' ? (
                          <>
                            <button
                              onClick={() => handleResolveMarket(market.id, 'yes')}
                              disabled={resolving === market.id}
                              className="w-full bg-green-600 hover:bg-green-700 disabled:bg-green-400 text-white py-2 px-4 rounded-lg font-medium transition-colors text-sm"
                            >
                              {resolving === market.id ? 'Resolving...' : 'Resolve as YES'}
                            </button>
                            <button
                              onClick={() => handleResolveMarket(market.id, 'no')}
                              disabled={resolving === market.id}
                              className="w-full bg-red-600 hover:bg-red-700 disabled:bg-red-400 text-white py-2 px-4 rounded-lg font-medium transition-colors text-sm"
                            >
                              {resolving === market.id ? 'Resolving...' : 'Resolve as NO'}
                            </button>
                          </>
                        ) : (
                          <div className="space-y-2">
                            <div className="text-xs text-gray-600 mb-2">Select winning outcome:</div>
                            {market.outcomes?.map((outcome) => (
                              <button
                                key={outcome.id}
                                onClick={() => handleResolveMarket(market.id, outcome.name)}
                                disabled={resolving === market.id}
                                className="w-full text-white py-2 px-4 rounded-lg font-medium transition-colors text-sm disabled:opacity-50"
                                style={{ 
                                  backgroundColor: outcome.color || '#6B7280',
                                  opacity: resolving === market.id ? 0.5 : 1
                                }}
                              >
                                {resolving === market.id ? 'Resolving...' : (
                                  <>
                                    {outcome.icon && <span className="mr-2">{outcome.icon}</span>}
                                    Resolve as {outcome.name.toUpperCase()}
                                  </>
                                )}
                              </button>
                            ))}
                          </div>
                        )}
                      </div>
                    ) : (
                      <div className="text-sm text-black p-3 bg-gray-50 rounded-lg">
                        Market is still active and has not ended yet
                      </div>
                    )}
                  </div>

                  {/* Algorithm Info */}
                  <div className="space-y-3">
                    <div className="text-sm font-medium text-black mb-2">Betting Algorithm</div>
                    <div className="text-xs text-black space-y-1">
                      <div>• Creators earn 20% of live bets</div>
                      <div>• Admin takes 1% of all bets</div>
                      <div>• Winners share based on seeded amounts</div>
                      <div>• Seeded bets get better payout ratios</div>
                    </div>
                  </div>
                </div>

                {/* Warning for expired markets */}
                {market.status === 'active' && isMarketExpired(market.end_date) && (
                  <div className="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                    <div className="flex items-center">
                      <svg className="w-4 h-4 text-yellow-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                      </svg>
                      <span className="text-yellow-700 text-sm">
                        This market has expired and should be resolved. The betting algorithm will automatically distribute winnings to seeded bet holders and pay creator/admin fees.
                      </span>
                    </div>
                  </div>
                )}
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  )
}
