'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { LoadingSpinner, PageLoadingSpinner, ButtonLoadingSpinner } from '@/components/ui/LoadingSpinner'
import Link from 'next/link'

interface AlgorithmSettings {
  // Liquidity Settings
  defaultLiquidity: number
  minLiquidity: number
  maxLiquidity: number
  liquidityFee: number
  
  // Seeding Settings
  defaultSeedingAmount: number
  minSeedingAmount: number
  maxSeedingAmount: number
  seedingFee: number
  
  // Price Calculation
  priceUpdateInterval: number
  volatilityThreshold: number
  priceStabilityWindow: number
  
  // Refund Settings
  refundThreshold: number
  refundDeadline: number
  autoRefundEnabled: boolean
  
  // Trading Settings
  minBetAmount: number
  maxBetAmount: number
  tradingFee: number
  slippageTolerance: number
  
  // Market Settings
  maxMarketsPerUser: number
  marketCreationFee: number
  marketResolutionFee: number
  
  // Algorithm Parameters
  constantProductK: number
  priceImpactThreshold: number
  liquidityDepth: number
  marketMakingEnabled: boolean
}

interface PlatformSettings {
  // General Settings
  platformName: string
  platformDescription: string
  maintenanceMode: boolean
  registrationEnabled: boolean
  siteUrl: string
  supportEmail: string
  supportPhone: string
  
  // Security Settings
  kycRequired: boolean
  maxDailyBets: number
  maxDailyVolume: number
  suspiciousActivityThreshold: number
  twoFactorRequired: boolean
  sessionTimeout: number
  maxLoginAttempts: number
  passwordMinLength: number
  passwordRequireSpecialChars: boolean
  
  // Notification Settings
  emailNotifications: boolean
  pushNotifications: boolean
  adminAlerts: boolean
  
  // Email Configuration
  smtpHost: string
  smtpPort: number
  smtpSecure: boolean
  smtpUser: string
  smtpPass: string
  emailFromName: string
  emailFromAddress: string
  
  // Email Templates
  welcomeEmailEnabled: boolean
  betResultEmailEnabled: boolean
  depositEmailEnabled: boolean
  withdrawalEmailEnabled: boolean
  kycEmailEnabled: boolean
  marketResolutionEmailEnabled: boolean
  
  // Email Template Content
  welcomeEmailSubject: string
  welcomeEmailContent: string
  betResultEmailSubject: string
  betResultEmailContent: string
  depositEmailSubject: string
  depositEmailContent: string
  withdrawalEmailSubject: string
  withdrawalEmailContent: string
  kycEmailSubject: string
  kycEmailContent: string
  marketResolutionEmailSubject: string
  marketResolutionEmailContent: string
  
  // API Settings
  apiRateLimit: number
  webhookUrl: string
  apiKeyRotation: number
  
  // Payment Integration Settings
  paynowIntegrationId: string
  paynowIntegrationKey: string
  paynowWebhookSecret: string
  paynowBaseUrl: string
  
  // System Announcements
  systemAnnouncement: string
  systemAnnouncementEnabled: boolean
  maintenanceMessage: string
}

export default function AdminSettings() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <AdminSettingsContent user={user} />
    </FastSessionWrapper>
  )
}

function AdminSettingsContent({ user }: { user: any }) {
  const [algorithmSettings, setAlgorithmSettings] = useState<AlgorithmSettings>({
    defaultLiquidity: 1000,
    minLiquidity: 100,
    maxLiquidity: 100000,
    liquidityFee: 0.02,
    defaultSeedingAmount: 500,
    minSeedingAmount: 50,
    maxSeedingAmount: 50000,
    seedingFee: 0.01,
    priceUpdateInterval: 5,
    volatilityThreshold: 0.1,
    priceStabilityWindow: 60,
    refundThreshold: 0.1,
    refundDeadline: 7,
    autoRefundEnabled: true,
    minBetAmount: 1,
    maxBetAmount: 10000,
    tradingFee: 0.02,
    slippageTolerance: 0.05,
    maxMarketsPerUser: 10,
    marketCreationFee: 10,
    marketResolutionFee: 5,
    constantProductK: 1,
    priceImpactThreshold: 0.05,
    liquidityDepth: 0.1,
    marketMakingEnabled: true
  })

  const [platformSettings, setPlatformSettings] = useState<PlatformSettings>({
    // General Settings
    platformName: 'BetFusion Zim',
    platformDescription: 'Zimbabwe\'s premier prediction market platform',
    maintenanceMode: false,
    registrationEnabled: true,
    siteUrl: process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000',
    supportEmail: 'support@betfusionzim.com',
    supportPhone: '+263 77 123 4567',
    
    // Security Settings
    kycRequired: false,
    maxDailyBets: 100,
    maxDailyVolume: 10000,
    suspiciousActivityThreshold: 0.8,
    twoFactorRequired: false,
    sessionTimeout: 24,
    maxLoginAttempts: 5,
    passwordMinLength: 8,
    passwordRequireSpecialChars: true,
    
    // Notification Settings
    emailNotifications: true,
    pushNotifications: true,
    adminAlerts: true,
    
    // Email Configuration
    smtpHost: process.env.SMTP_HOST || 'smtp.gmail.com',
    smtpPort: parseInt(process.env.SMTP_PORT || '587'),
    smtpSecure: process.env.SMTP_SECURE === 'true',
    smtpUser: process.env.SMTP_USER || '',
    smtpPass: process.env.SMTP_PASS || '',
    emailFromName: 'BetFusion Zim',
    emailFromAddress: process.env.SMTP_USER || 'noreply@betfusionzim.com',
    
    // Email Templates
    welcomeEmailEnabled: true,
    betResultEmailEnabled: true,
    depositEmailEnabled: true,
    withdrawalEmailEnabled: true,
    kycEmailEnabled: true,
    marketResolutionEmailEnabled: true,
    
    // API Settings
    apiRateLimit: 1000,
    webhookUrl: '',
    apiKeyRotation: 30,
    
    // Payment Integration Settings
    paynowIntegrationId: process.env.PAYNOW_ZIM_INTEGRATION_ID || '',
    paynowIntegrationKey: process.env.PAYNOW_ZIM_INTEGRATION_KEY || '',
    paynowWebhookSecret: process.env.PAYNOW_ZIM_WEBHOOK_SECRET || '',
    paynowBaseUrl: process.env.PAYNOW_ZIM_BASE_URL || 'https://api.paynow.co.zw',
    
    // System Announcements
    systemAnnouncement: '',
    systemAnnouncementEnabled: false,
    maintenanceMessage: 'We are currently performing scheduled maintenance. Please check back later.'
  })

  const [loading, setLoading] = useState(true)
  const [saving, setSaving] = useState(false)
  const [error, setError] = useState<string | null>(null)
  const [success, setSuccess] = useState<string | null>(null)
  const [activeTab, setActiveTab] = useState<'algorithm' | 'platform' | 'security' | 'notifications' | 'payments' | 'email' | 'system'>('algorithm')

  useEffect(() => {
    fetchSettings()
  }, [])

  const fetchSettings = async () => {
    try {
      setLoading(true)
      setError(null)

      const { supabase } = await import('@/lib/supabase-client')

      // Try to fetch from database
      try {
        const { data: algorithmData, error: algoError } = await supabase
          .from('admin_settings')
          .select('settings')
          .eq('id', 'algorithm_settings')
          .single()

        const { data: platformData, error: platformError } = await supabase
          .from('admin_settings')
          .select('settings')
          .eq('id', 'platform_settings')
          .single()

        // Check if table doesn't exist
        if (algoError && algoError.message.includes('relation "admin_settings" does not exist')) {
          throw new Error('Table does not exist')
        }

        if (algorithmData?.settings) {
          setAlgorithmSettings(algorithmData.settings)
        }

        if (platformData?.settings) {
          setPlatformSettings(platformData.settings)
        }
      } catch (dbError) {
        console.log('Database fetch failed, trying localStorage:', dbError)
        
        // Fallback to localStorage
        try {
          const savedAlgorithmSettings = localStorage.getItem('admin_algorithm_settings')
          const savedPlatformSettings = localStorage.getItem('admin_platform_settings')
          
          if (savedAlgorithmSettings) {
            setAlgorithmSettings(JSON.parse(savedAlgorithmSettings))
          }
          
          if (savedPlatformSettings) {
            setPlatformSettings(JSON.parse(savedPlatformSettings))
          }
        } catch (storageError) {
          console.warn('localStorage fetch failed:', storageError)
        }
      }

      setLoading(false)
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to fetch settings')
      console.error('Error fetching settings:', err)
      setLoading(false)
    }
  }

  const saveSettings = async () => {
    try {
      setSaving(true)
      setError(null)
      setSuccess(null)

      const { supabase } = await import('@/lib/supabase-client')

      try {
        // Try to create the table if it doesn't exist
        const { error: createTableError } = await supabase.rpc('create_admin_settings_table_if_not_exists')
        if (createTableError) {
          console.log('Table creation function not found, trying direct SQL:', createTableError)
          
          // Try to create the table directly
          const { error: directCreateError } = await supabase
            .from('admin_settings')
            .select('id')
            .limit(1)
          
          if (directCreateError && directCreateError.message.includes('relation "admin_settings" does not exist')) {
            console.log('Table does not exist, will use localStorage fallback')
          }
        }
      } catch (tableError) {
        console.log('Table creation skipped or already exists:', tableError)
      }

      // Try to save to database first
      let useLocalStorage = false
      
      try {
        // Save algorithm settings
        const { error: algoError } = await supabase
          .from('admin_settings')
          .upsert({
            id: 'algorithm_settings',
            settings: algorithmSettings,
            updated_at: new Date().toISOString()
          })

        if (algoError) {
          console.error('Algorithm settings error:', algoError)
          if (algoError.message.includes('relation "admin_settings" does not exist')) {
            useLocalStorage = true
          } else {
            throw new Error(`Failed to save algorithm settings: ${algoError.message}`)
          }
        }

        // Save platform settings
        const { error: platformError } = await supabase
          .from('admin_settings')
          .upsert({
            id: 'platform_settings',
            settings: platformSettings,
            updated_at: new Date().toISOString()
          })

        if (platformError) {
          console.error('Platform settings error:', platformError)
          if (platformError.message.includes('relation "admin_settings" does not exist')) {
            useLocalStorage = true
          } else {
            throw new Error(`Failed to save platform settings: ${platformError.message}`)
          }
        }
      } catch (dbError) {
        console.log('Database save failed, using localStorage fallback:', dbError)
        useLocalStorage = true
      }

      // Fallback to localStorage if database fails
      if (useLocalStorage) {
        console.log('Using localStorage fallback for settings')
        localStorage.setItem('admin_algorithm_settings', JSON.stringify(algorithmSettings))
        localStorage.setItem('admin_platform_settings', JSON.stringify(platformSettings))
      }

      setSuccess('Settings saved successfully!')
      setTimeout(() => setSuccess(null), 3000)
      
      // Also save to localStorage as backup
      try {
        localStorage.setItem('admin_algorithm_settings', JSON.stringify(algorithmSettings))
        localStorage.setItem('admin_platform_settings', JSON.stringify(platformSettings))
      } catch (storageError) {
        console.warn('Failed to save to localStorage:', storageError)
      }

    } catch (err) {
      console.error('Error saving settings:', err)
      
      // Better error handling for different error types
      let errorMessage = 'Failed to save settings'
      
      if (err instanceof Error) {
        errorMessage = err.message
      } else if (err && typeof err === 'object') {
        // Handle Supabase errors
        if ('message' in err) {
          errorMessage = String(err.message)
        } else if ('error' in err) {
          errorMessage = String(err.error)
        } else {
          errorMessage = 'Database error occurred'
        }
      }
      
      setError(errorMessage)
    } finally {
      setSaving(false)
    }
  }

  const resetToDefaults = () => {
    if (confirm('Are you sure you want to reset all settings to defaults? This action cannot be undone.')) {
      setAlgorithmSettings({
        defaultLiquidity: 1000,
        minLiquidity: 100,
        maxLiquidity: 100000,
        liquidityFee: 0.02,
        defaultSeedingAmount: 500,
        minSeedingAmount: 50,
        maxSeedingAmount: 50000,
        seedingFee: 0.01,
        priceUpdateInterval: 5,
        volatilityThreshold: 0.1,
        priceStabilityWindow: 60,
        refundThreshold: 0.1,
        refundDeadline: 7,
        autoRefundEnabled: true,
        minBetAmount: 1,
        maxBetAmount: 10000,
        tradingFee: 0.02,
        slippageTolerance: 0.05,
        maxMarketsPerUser: 10,
        marketCreationFee: 10,
        marketResolutionFee: 5,
        constantProductK: 1,
        priceImpactThreshold: 0.05,
        liquidityDepth: 0.1,
        marketMakingEnabled: true
      })
      
      setPlatformSettings({
        platformName: 'Prediction Market',
        platformDescription: 'Decentralized prediction market platform',
        maintenanceMode: false,
        registrationEnabled: true,
        kycRequired: false,
        maxDailyBets: 100,
        maxDailyVolume: 10000,
        suspiciousActivityThreshold: 0.8,
        emailNotifications: true,
        pushNotifications: true,
        adminAlerts: true,
        apiRateLimit: 1000,
        webhookUrl: '',
        apiKeyRotation: 30,
        paynowIntegrationId: '',
        paynowIntegrationKey: '',
        paynowWebhookSecret: '',
        paynowBaseUrl: 'https://api.paynow.co.zw'
      })
      setSuccess('Settings reset to defaults!')
      setTimeout(() => setSuccess(null), 3000)
    }
  }

  if (loading) {
    return <PageLoadingSpinner />
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-black">Admin Settings</h1>
              <p className="text-black mt-2">Configure platform and algorithm parameters</p>
            </div>
            <Link
              href="/admin"
              className="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition-colors"
            >
              ← Back to Dashboard
            </Link>
          </div>
        </div>

        {/* Success/Error Messages */}
        {success && (
          <div className="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg mb-6">
            {success}
          </div>
        )}
        {error && (
          <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6">
            {error}
          </div>
        )}

        {/* Tabs */}
        <div className="bg-white rounded-lg shadow-sm mb-6">
          <div className="border-b border-gray-200">
            <nav className="-mb-px flex space-x-8 px-6">
              {[
                { id: 'algorithm', name: 'Algorithm', icon: '🧮' },
                { id: 'platform', name: 'Platform', icon: '⚙️' },
                { id: 'security', name: 'Security', icon: '🔒' },
                { id: 'notifications', name: 'Notifications', icon: '🔔' },
                { id: 'payments', name: 'Payments', icon: '💳' },
                { id: 'email', name: 'Email', icon: '📧' },
                { id: 'system', name: 'System', icon: '🖥️' }
              ].map((tab) => (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id as any)}
                  className={`py-4 px-1 border-b-2 font-medium text-sm ${
                    activeTab === tab.id
                      ? 'border-blue-500 text-blue-600'
                      : 'border-transparent text-black hover:text-black hover:border-gray-300'
                  }`}
                >
                  <span className="mr-2">{tab.icon}</span>
                  {tab.name}
                </button>
              ))}
            </nav>
          </div>

          <div className="p-6">
            {/* Algorithm Settings */}
            {activeTab === 'algorithm' && (
              <div className="space-y-8">
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Liquidity Management</h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Default Liquidity ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.defaultLiquidity}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, defaultLiquidity: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Min Liquidity ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.minLiquidity}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, minLiquidity: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Max Liquidity ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.maxLiquidity}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, maxLiquidity: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Liquidity Fee (%)
                      </label>
                      <input
                        type="number"
                        step="0.01"
                        value={algorithmSettings.liquidityFee}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, liquidityFee: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                  </div>
                </div>

                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Seeding Configuration</h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Default Seeding Amount ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.defaultSeedingAmount}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, defaultSeedingAmount: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Min Seeding Amount ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.minSeedingAmount}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, minSeedingAmount: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Max Seeding Amount ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.maxSeedingAmount}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, maxSeedingAmount: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Seeding Fee (%)
                      </label>
                      <input
                        type="number"
                        step="0.01"
                        value={algorithmSettings.seedingFee}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, seedingFee: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                  </div>
                </div>

                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Price Calculation</h3>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Price Update Interval (seconds)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.priceUpdateInterval}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, priceUpdateInterval: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Volatility Threshold
                      </label>
                      <input
                        type="number"
                        step="0.01"
                        value={algorithmSettings.volatilityThreshold}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, volatilityThreshold: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Price Stability Window (seconds)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.priceStabilityWindow}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, priceStabilityWindow: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                  </div>
                </div>

                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Refund Settings</h3>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Refund Threshold (%)
                      </label>
                      <input
                        type="number"
                        step="0.01"
                        value={algorithmSettings.refundThreshold}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, refundThreshold: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Refund Deadline (days)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.refundDeadline}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, refundDeadline: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div className="flex items-center">
                      <input
                        type="checkbox"
                        checked={algorithmSettings.autoRefundEnabled}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, autoRefundEnabled: e.target.checked }))}
                        className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                      />
                      <label className="ml-2 block text-sm text-black">
                        Auto Refund Enabled
                      </label>
                    </div>
                  </div>
                </div>

                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Trading Parameters</h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Min Bet Amount ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.minBetAmount}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, minBetAmount: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Max Bet Amount ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.maxBetAmount}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, maxBetAmount: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Trading Fee (%)
                      </label>
                      <input
                        type="number"
                        step="0.01"
                        value={algorithmSettings.tradingFee}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, tradingFee: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Slippage Tolerance (%)
                      </label>
                      <input
                        type="number"
                        step="0.01"
                        value={algorithmSettings.slippageTolerance}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, slippageTolerance: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Platform Settings */}
            {activeTab === 'platform' && (
              <div className="space-y-8">
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">General Settings</h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Platform Name
                      </label>
                      <input
                        type="text"
                        value={platformSettings.platformName}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, platformName: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Max Markets Per User
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.maxMarketsPerUser}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, maxMarketsPerUser: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Market Creation Fee ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.marketCreationFee}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, marketCreationFee: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Market Resolution Fee ($)
                      </label>
                      <input
                        type="number"
                        value={algorithmSettings.marketResolutionFee}
                        onChange={(e) => setAlgorithmSettings(prev => ({ ...prev, marketResolutionFee: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-black mb-2">
                    Platform Description
                  </label>
                  <textarea
                    value={platformSettings.platformDescription}
                    onChange={(e) => setPlatformSettings(prev => ({ ...prev, platformDescription: e.target.value }))}
                    rows={3}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                </div>

                <div className="space-y-4">
                  <div className="flex items-center">
                    <input
                      type="checkbox"
                      checked={platformSettings.maintenanceMode}
                      onChange={(e) => setPlatformSettings(prev => ({ ...prev, maintenanceMode: e.target.checked }))}
                      className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                    />
                    <label className="ml-2 block text-sm text-black">
                      Maintenance Mode
                    </label>
                  </div>
                  <div className="flex items-center">
                    <input
                      type="checkbox"
                      checked={platformSettings.registrationEnabled}
                      onChange={(e) => setPlatformSettings(prev => ({ ...prev, registrationEnabled: e.target.checked }))}
                      className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                    />
                    <label className="ml-2 block text-sm text-black">
                      Registration Enabled
                    </label>
                  </div>
                </div>
              </div>
            )}

            {/* Security Settings */}
            {activeTab === 'security' && (
              <div className="space-y-8">
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Security Configuration</h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Max Daily Bets
                      </label>
                      <input
                        type="number"
                        value={platformSettings.maxDailyBets}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, maxDailyBets: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Max Daily Volume ($)
                      </label>
                      <input
                        type="number"
                        value={platformSettings.maxDailyVolume}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, maxDailyVolume: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Suspicious Activity Threshold
                      </label>
                      <input
                        type="number"
                        step="0.1"
                        value={platformSettings.suspiciousActivityThreshold}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, suspiciousActivityThreshold: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        API Rate Limit (requests/hour)
                      </label>
                      <input
                        type="number"
                        value={platformSettings.apiRateLimit}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, apiRateLimit: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                      />
                    </div>
                  </div>
                </div>

                <div className="space-y-4">
                  <div className="flex items-center">
                    <input
                      type="checkbox"
                      checked={platformSettings.kycRequired}
                      onChange={(e) => setPlatformSettings(prev => ({ ...prev, kycRequired: e.target.checked }))}
                      className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                    />
                    <label className="ml-2 block text-sm text-black">
                      KYC Required for Trading
                    </label>
                  </div>
                </div>
              </div>
            )}

            {/* Notification Settings */}
            {activeTab === 'notifications' && (
              <div className="space-y-8">
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Notification Preferences</h3>
                  <div className="space-y-4">
                    <div className="flex items-center">
                      <input
                        type="checkbox"
                        checked={platformSettings.emailNotifications}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, emailNotifications: e.target.checked }))}
                        className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                      />
                      <label className="ml-2 block text-sm text-black">
                        Email Notifications
                      </label>
                    </div>
                    <div className="flex items-center">
                      <input
                        type="checkbox"
                        checked={platformSettings.pushNotifications}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, pushNotifications: e.target.checked }))}
                        className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                      />
                      <label className="ml-2 block text-sm text-black">
                        Push Notifications
                      </label>
                    </div>
                    <div className="flex items-center">
                      <input
                        type="checkbox"
                        checked={platformSettings.adminAlerts}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, adminAlerts: e.target.checked }))}
                        className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                      />
                      <label className="ml-2 block text-sm text-black">
                        Admin Alerts
                      </label>
                    </div>
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-black mb-2">
                    Webhook URL
                  </label>
                  <input
                    type="url"
                    value={platformSettings.webhookUrl}
                    onChange={(e) => setPlatformSettings(prev => ({ ...prev, webhookUrl: e.target.value }))}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="https://your-webhook-url.com"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-black mb-2">
                    API Key Rotation (days)
                  </label>
                  <input
                    type="number"
                    value={platformSettings.apiKeyRotation}
                    onChange={(e) => setPlatformSettings(prev => ({ ...prev, apiKeyRotation: Number(e.target.value) }))}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                </div>
              </div>
            )}

            {/* Payments Settings */}
            {activeTab === 'payments' && (
              <div className="space-y-8">
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">PayNow Integration Settings</h3>
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                    <div className="flex items-start">
                      <div className="text-blue-600 mr-3 mt-1">💳</div>
                      <div>
                        <h4 className="font-medium text-blue-900 mb-2">PayNow Integration</h4>
                        <p className="text-sm text-blue-800">
                          Configure your PayNow integration credentials. These settings are used for processing USD payments.
                        </p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Integration ID
                      </label>
                      <input
                        type="text"
                        value={platformSettings.paynowIntegrationId}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, paynowIntegrationId: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="Enter your PayNow Integration ID"
                      />
                      <p className="text-xs text-gray-600 mt-1">Your PayNow integration identifier</p>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Integration Key
                      </label>
                      <input
                        type="password"
                        value={platformSettings.paynowIntegrationKey}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, paynowIntegrationKey: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="Enter your PayNow Integration Key"
                      />
                      <p className="text-xs text-gray-600 mt-1">Your PayNow integration secret key</p>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Webhook Secret
                      </label>
                      <input
                        type="password"
                        value={platformSettings.paynowWebhookSecret}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, paynowWebhookSecret: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="Enter your PayNow Webhook Secret"
                      />
                      <p className="text-xs text-gray-600 mt-1">Secret for webhook verification</p>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        Base URL
                      </label>
                      <input
                        type="url"
                        value={platformSettings.paynowBaseUrl}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, paynowBaseUrl: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="https://api.paynow.co.zw"
                      />
                      <p className="text-xs text-gray-600 mt-1">PayNow API base URL</p>
                    </div>
                  </div>

                  <div className="mt-6 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                    <div className="flex items-start">
                      <div className="text-yellow-600 mr-3 mt-1">⚠️</div>
                      <div>
                        <h4 className="font-medium text-yellow-900 mb-2">Security Notice</h4>
                        <p className="text-sm text-yellow-800">
                          These credentials are sensitive and should be kept secure. Make sure to test your integration 
                          in a sandbox environment before going live.
                        </p>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Email Settings Tab */}
            {activeTab === 'email' && (
              <div className="space-y-8">
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Email Configuration</h3>
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                    <div className="flex items-start">
                      <div className="text-blue-600 mr-3 mt-1">📧</div>
                      <div>
                        <h4 className="font-medium text-blue-900 mb-2">SMTP Configuration</h4>
                        <p className="text-sm text-blue-800">
                          Configure your email server settings for sending notifications, confirmations, and updates to users.
                        </p>
                      </div>
                    </div>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        SMTP Host
                      </label>
                      <input
                        type="text"
                        value={platformSettings.smtpHost}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, smtpHost: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="smtp.gmail.com"
                      />
                      <p className="text-xs text-gray-600 mt-1">Your SMTP server hostname</p>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        SMTP Port
                      </label>
                      <input
                        type="number"
                        value={platformSettings.smtpPort}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, smtpPort: Number(e.target.value) }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="587"
                      />
                      <p className="text-xs text-gray-600 mt-1">SMTP server port (usually 587 or 465)</p>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        SMTP Username
                      </label>
                      <input
                        type="text"
                        value={platformSettings.smtpUser}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, smtpUser: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="your-email@gmail.com"
                      />
                      <p className="text-xs text-gray-600 mt-1">Your SMTP username/email</p>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        SMTP Password
                      </label>
                      <input
                        type="password"
                        value={platformSettings.smtpPass}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, smtpPass: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="Your SMTP password or app password"
                      />
                      <p className="text-xs text-gray-600 mt-1">Your SMTP password or app-specific password</p>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        From Name
                      </label>
                      <input
                        type="text"
                        value={platformSettings.emailFromName}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, emailFromName: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="BetFusion Zim"
                      />
                      <p className="text-xs text-gray-600 mt-1">Display name for outgoing emails</p>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-black mb-2">
                        From Email
                      </label>
                      <input
                        type="email"
                        value={platformSettings.emailFromAddress}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, emailFromAddress: e.target.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        placeholder="noreply@betfusionzim.com"
                      />
                      <p className="text-xs text-gray-600 mt-1">Email address for outgoing emails</p>
                    </div>
                  </div>

                  <div className="mt-6">
                    <h4 className="text-md font-medium text-black mb-4">Email Template Settings</h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div className="flex items-center">
                        <input
                          type="checkbox"
                          checked={platformSettings.welcomeEmailEnabled}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, welcomeEmailEnabled: e.target.checked }))}
                          className="mr-3"
                        />
                        <label className="text-sm text-black">Welcome emails for new users</label>
                      </div>
                      <div className="flex items-center">
                        <input
                          type="checkbox"
                          checked={platformSettings.betResultEmailEnabled}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, betResultEmailEnabled: e.target.checked }))}
                          className="mr-3"
                        />
                        <label className="text-sm text-black">Bet result notifications</label>
                      </div>
                      <div className="flex items-center">
                        <input
                          type="checkbox"
                          checked={platformSettings.depositEmailEnabled}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, depositEmailEnabled: e.target.checked }))}
                          className="mr-3"
                        />
                        <label className="text-sm text-black">Deposit confirmations</label>
                      </div>
                      <div className="flex items-center">
                        <input
                          type="checkbox"
                          checked={platformSettings.withdrawalEmailEnabled}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, withdrawalEmailEnabled: e.target.checked }))}
                          className="mr-3"
                        />
                        <label className="text-sm text-black">Withdrawal confirmations</label>
                      </div>
                      <div className="flex items-center">
                        <input
                          type="checkbox"
                          checked={platformSettings.kycEmailEnabled}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, kycEmailEnabled: e.target.checked }))}
                          className="mr-3"
                        />
                        <label className="text-sm text-black">KYC status updates</label>
                      </div>
                      <div className="flex items-center">
                        <input
                          type="checkbox"
                          checked={platformSettings.marketResolutionEmailEnabled}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, marketResolutionEmailEnabled: e.target.checked }))}
                          className="mr-3"
                        />
                        <label className="text-sm text-black">Market resolution notifications</label>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* System Settings Tab */}
            {activeTab === 'system' && (
              <div className="space-y-8">
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">System Configuration</h3>
                  
                  {/* General Settings */}
                  <div className="mb-8">
                    <h4 className="text-md font-medium text-black mb-4">General Settings</h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div>
                        <label className="block text-sm font-medium text-black mb-2">
                          Site URL
                        </label>
                        <input
                          type="url"
                          value={platformSettings.siteUrl}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, siteUrl: e.target.value }))}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="https://betfusionzim.com"
                        />
                        <p className="text-xs text-gray-600 mt-1">Your website URL for email links</p>
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-black mb-2">
                          Support Email
                        </label>
                        <input
                          type="email"
                          value={platformSettings.supportEmail}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, supportEmail: e.target.value }))}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="support@betfusionzim.com"
                        />
                        <p className="text-xs text-gray-600 mt-1">Customer support email address</p>
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-black mb-2">
                          Support Phone
                        </label>
                        <input
                          type="tel"
                          value={platformSettings.supportPhone}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, supportPhone: e.target.value }))}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          placeholder="+263 77 123 4567"
                        />
                        <p className="text-xs text-gray-600 mt-1">Customer support phone number</p>
                      </div>
                    </div>
                  </div>

                  {/* Security Settings */}
                  <div className="mb-8">
                    <h4 className="text-md font-medium text-black mb-4">Security Settings</h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div className="flex items-center">
                        <input
                          type="checkbox"
                          checked={platformSettings.twoFactorRequired}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, twoFactorRequired: e.target.checked }))}
                          className="mr-3"
                        />
                        <label className="text-sm text-black">Require two-factor authentication</label>
                      </div>
                      <div className="flex items-center">
                        <input
                          type="checkbox"
                          checked={platformSettings.passwordRequireSpecialChars}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, passwordRequireSpecialChars: e.target.checked }))}
                          className="mr-3"
                        />
                        <label className="text-sm text-black">Require special characters in passwords</label>
                      </div>
                    </div>
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mt-4">
                      <div>
                        <label className="block text-sm font-medium text-black mb-2">
                          Session Timeout (hours)
                        </label>
                        <input
                          type="number"
                          value={platformSettings.sessionTimeout}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, sessionTimeout: Number(e.target.value) }))}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-black mb-2">
                          Max Login Attempts
                        </label>
                        <input
                          type="number"
                          value={platformSettings.maxLoginAttempts}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, maxLoginAttempts: Number(e.target.value) }))}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-black mb-2">
                          Min Password Length
                        </label>
                        <input
                          type="number"
                          value={platformSettings.passwordMinLength}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, passwordMinLength: Number(e.target.value) }))}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                        />
                      </div>
                    </div>
                  </div>

                  {/* System Announcements */}
                  <div className="mb-8">
                    <h4 className="text-md font-medium text-black mb-4">System Announcements</h4>
                    <div className="space-y-4">
                      <div className="flex items-center">
                        <input
                          type="checkbox"
                          checked={platformSettings.systemAnnouncementEnabled}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, systemAnnouncementEnabled: e.target.checked }))}
                          className="mr-3"
                        />
                        <label className="text-sm text-black">Enable system announcement</label>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-black mb-2">
                          System Announcement
                        </label>
                        <textarea
                          value={platformSettings.systemAnnouncement}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, systemAnnouncement: e.target.value }))}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          rows={3}
                          placeholder="Enter system announcement message..."
                        />
                        <p className="text-xs text-gray-600 mt-1">This message will be displayed to all users</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-black mb-2">
                          Maintenance Message
                        </label>
                        <textarea
                          value={platformSettings.maintenanceMessage}
                          onChange={(e) => setPlatformSettings(prev => ({ ...prev, maintenanceMessage: e.target.value }))}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                          rows={3}
                          placeholder="Enter maintenance message..."
                        />
                        <p className="text-xs text-gray-600 mt-1">Message shown during maintenance mode</p>
                      </div>
                    </div>
                  </div>

                  {/* Maintenance Mode */}
                  <div className="mb-8">
                    <h4 className="text-md font-medium text-black mb-4">Maintenance Mode</h4>
                    <div className="flex items-center">
                      <input
                        type="checkbox"
                        checked={platformSettings.maintenanceMode}
                        onChange={(e) => setPlatformSettings(prev => ({ ...prev, maintenanceMode: e.target.checked }))}
                        className="mr-3"
                      />
                      <label className="text-sm text-black">Enable maintenance mode</label>
                    </div>
                    <p className="text-xs text-gray-600 mt-2">
                      When enabled, only administrators can access the platform
                    </p>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>

        {/* Action Buttons */}
        <div className="flex justify-between items-center">
          <button
            onClick={resetToDefaults}
            className="bg-gray-300 text-black px-4 py-2 rounded-lg hover:bg-gray-400 transition-colors"
          >
            Reset to Defaults
          </button>
          <button
            onClick={saveSettings}
            disabled={saving}
            className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors"
          >
            {saving ? <ButtonLoadingSpinner /> : 'Save Settings'}
          </button>
        </div>
      </div>
    </div>
  )
}