'use client'

import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { PageLoadingSpinner } from '@/components/ui/LoadingSpinner'
import { useAdminTransactionsOptimized } from '@/lib/hooks/useAdminTransactionsOptimized'
import AdminTransactionsTable from '@/components/admin/AdminTransactionsTable'
import Link from 'next/link'

export default function AdminTransactionsPage() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <AdminTransactionsContent user={user} />
    </FastSessionWrapper>
  )
}

function AdminTransactionsContent({ user }: { user: any }) {
  const {
    transactions,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    setSearchTerm,
    loadMore,
    refresh,
    filterByType
  } = useAdminTransactionsOptimized()

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  if (loading && transactions.length === 0) {
    return <PageLoadingSpinner />
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-black">Transaction Management</h1>
              <p className="text-black mt-2">Monitor all financial transactions with optimized performance</p>
            </div>
            <Link
              href="/admin"
              className="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition-colors"
            >
              ← Back to Dashboard
            </Link>
          </div>
        </div>

        {/* Transaction Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Transaction Overview</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{metrics.totalTransactions}</div>
              <div className="text-sm text-black">Total Transactions</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-green-600">{formatCurrency(metrics.totalVolume)}</div>
              <div className="text-sm text-black">Total Volume</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-purple-600">{formatCurrency(metrics.averageAmount)}</div>
              <div className="text-sm text-black">Average Amount</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-orange-600">{metrics.completionRate.toFixed(1)}%</div>
              <div className="text-sm text-black">Completion Rate</div>
            </div>
          </div>
        </div>

        {/* Transaction Type Breakdown */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">Transaction Types</h2>
          <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-green-600">{metrics.deposits}</div>
              <div className="text-sm text-black">Deposits</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-red-600">{metrics.withdrawals}</div>
              <div className="text-sm text-black">Withdrawals</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{metrics.bets}</div>
              <div className="text-sm text-black">Bets</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-yellow-600">{metrics.winnings}</div>
              <div className="text-sm text-black">Winnings</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-purple-600">{formatCurrency(metrics.fees)}</div>
              <div className="text-sm text-black">Total Fees</div>
            </div>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <input
                type="text"
                placeholder="Search transactions..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>
            <div className="flex gap-2">
              {['all', 'deposit', 'withdrawal', 'bet', 'win', 'fee'].map((type) => (
                <button
                  key={type}
                  onClick={() => filterByType(type)}
                  className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                    'bg-blue-600 text-white'
                  }`}
                >
                  {type.charAt(0).toUpperCase() + type.slice(1)}
                </button>
              ))}
            </div>
          </div>
        </div>

        {/* Transactions Table */}
        <AdminTransactionsTable
          transactions={transactions}
          loading={loading}
          hasMore={hasMore}
          onLoadMore={loadMore}
          onRefresh={refresh}
        />
      </div>
    </div>
  )
}