'use client'

import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { PageLoadingSpinner } from '@/components/ui/LoadingSpinner'
import { useAdminUsersOptimized } from '@/lib/hooks/useAdminUsersOptimized'
import AdminUsersTable from '@/components/admin/AdminUsersTable'
import Link from 'next/link'


export default function AdminUsers() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true} requireAdmin={true}>
      <AdminUsersContent user={user} />
    </FastSessionWrapper>
  )
}

function AdminUsersContent({ user }: { user: any }) {
  const {
    users,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    setSearchTerm,
    loadMore,
    refresh,
    filterByStatus
  } = useAdminUsersOptimized()


  if (loading && users.length === 0) {
    return <PageLoadingSpinner />
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-black">User Management</h1>
              <p className="text-black mt-2">Manage users with optimized performance and real-time updates</p>
            </div>
            <Link
              href="/admin"
              className="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition-colors"
            >
              ← Back to Dashboard
            </Link>
          </div>
        </div>

        {/* User Metrics */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-8">
          <h2 className="text-xl font-semibold text-black mb-4">User Overview</h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{metrics.totalUsers}</div>
              <div className="text-sm text-black">Total Users</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-green-600">{metrics.activeUsers}</div>
              <div className="text-sm text-black">Active (30d)</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-red-600">{metrics.bannedUsers}</div>
              <div className="text-sm text-black">Banned</div>
            </div>
            <div className="text-center p-4 border border-gray-200 rounded-lg">
              <div className="text-2xl font-bold text-purple-600">{metrics.topPerformers}</div>
              <div className="text-sm text-black">Top Performers</div>
            </div>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <input
                type="text"
                placeholder="Search users..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>
            <div className="flex gap-2">
              {['all', 'banned', 'admin', 'kyc_pending', 'kyc_approved'].map((status) => (
                <button
                  key={status}
                  onClick={() => filterByStatus(status)}
                  className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                    'bg-blue-600 text-white'
                  }`}
                >
                  {status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                </button>
              ))}
            </div>
          </div>
        </div>

        {/* Users Table */}
        <AdminUsersTable
          users={users}
          loading={loading}
          hasMore={hasMore}
          onLoadMore={loadMore}
          onRefresh={refresh}
        />
      </div>
    </div>
  )
}