import { NextRequest, NextResponse } from 'next/server'
import { BettingEngine } from '@/lib/betting'

export async function POST(request: NextRequest) {
  try {
    // Get the minimum confidence threshold from request body (default: 70%)
    const body = await request.json().catch(() => ({}))
    const minConfidence = body.minConfidence || 70

    console.log(`Starting automatic market resolution with ${minConfidence}% minimum confidence...`)

    // Call the auto-resolve function
    const result = await BettingEngine.autoResolveExpiredMarkets(minConfidence)

    if (!result.success) {
      return NextResponse.json(
        { 
          success: false, 
          error: result.error,
          processed: result.processed,
          resolved: result.resolved,
          failed: result.failed
        },
        { status: 500 }
      )
    }

    // Log the results
    console.log(`Auto-resolution completed: ${result.resolved}/${result.processed} markets resolved`)

    return NextResponse.json({
      success: true,
      message: `Auto-resolution completed: ${result.resolved}/${result.processed} markets resolved`,
      processed: result.processed,
      resolved: result.resolved,
      failed: result.failed,
      results: result.results
    })

  } catch (error: any) {
    console.error('Error in auto-resolve API:', error)
    
    return NextResponse.json(
      { 
        success: false, 
        error: error.message || 'Internal server error',
        processed: 0,
        resolved: 0,
        failed: 0
      },
      { status: 500 }
    )
  }
}

// Allow GET requests for testing
export async function GET() {
  try {
    console.log('Auto-resolve API endpoint accessed via GET')
    
    const result = await BettingEngine.autoResolveExpiredMarkets(70)
    
    return NextResponse.json({
      success: result.success,
      message: result.success ? 'Auto-resolution completed successfully' : result.error,
      processed: result.processed,
      resolved: result.resolved,
      failed: result.failed,
      results: result.results
    })

  } catch (error: any) {
    console.error('Error in auto-resolve GET API:', error)
    
    return NextResponse.json(
      { 
        success: false, 
        error: error.message || 'Internal server error',
        processed: 0,
        resolved: 0,
        failed: 0
      },
      { status: 500 }
    )
  }
}
