import { NextRequest, NextResponse } from 'next/server'
import { emailService } from '@/lib/services/emailService'

export async function POST(request: NextRequest) {
  try {
    const { 
      to, 
      subject, 
      html, 
      text, 
      type,
      data 
    } = await request.json()

    if (!to || !subject) {
      return NextResponse.json({ 
        error: 'Email address and subject are required' 
      }, { status: 400 })
    }

    let success = false

    // Send different types of emails based on type
    switch (type) {
      case 'welcome':
        const { userName } = data || {}
        success = await emailService.sendWelcomeEmail(to, userName || 'User')
        break

      case 'bet_result':
        const { userName: betUserName, outcome, amount, marketTitle } = data || {}
        success = await emailService.sendBetResultEmail(
          to, 
          betUserName || 'User', 
          outcome, 
          amount, 
          marketTitle
        )
        break

      case 'deposit_confirmation':
        const { userName: depositUserName, amount: depositAmount, transactionId } = data || {}
        success = await emailService.sendDepositConfirmationEmail(
          to, 
          depositUserName || 'User', 
          depositAmount, 
          transactionId
        )
        break

      case 'withdrawal_confirmation':
        const { userName: withdrawalUserName, amount: withdrawalAmount, transactionId: withdrawalTransactionId } = data || {}
        success = await emailService.sendWithdrawalConfirmationEmail(
          to, 
          withdrawalUserName || 'User', 
          withdrawalAmount, 
          withdrawalTransactionId
        )
        break

      case 'market_resolution':
        const { 
          userName: marketUserName, 
          marketTitle: marketTitle, 
          winningOutcome, 
          userOutcome, 
          won, 
          amount: marketAmount 
        } = data || {}
        success = await emailService.sendMarketResolutionEmail(
          to, 
          marketUserName || 'User', 
          marketTitle, 
          winningOutcome, 
          userOutcome, 
          won, 
          marketAmount
        )
        break

      case 'kyc_status':
        const { userName: kycUserName, status, reason } = data || {}
        success = await emailService.sendKYCStatusEmail(
          to, 
          kycUserName || 'User', 
          status, 
          reason
        )
        break

      case 'refund':
        const { 
          userName: refundUserName, 
          amount: refundAmount, 
          betId, 
          outcome, 
          marketTitle: refundMarketTitle 
        } = data || {}
        success = await emailService.sendRefundEmail(
          to, 
          refundUserName || 'User', 
          refundAmount, 
          betId, 
          outcome, 
          refundMarketTitle
        )
        break

      case 'market_refund':
        const { 
          userName: marketRefundUserName, 
          amount: marketRefundAmount, 
          marketId, 
          betCount 
        } = data || {}
        success = await emailService.sendMarketRefundEmail(
          to, 
          marketRefundUserName || 'User', 
          marketRefundAmount, 
          marketId, 
          betCount
        )
        break

      default:
        // Send generic email
        success = await emailService.sendEmail({
          to,
          subject,
          html: html || `<p>${text || 'You have a new notification from BetFusion Zim.'}</p>`,
          text: text || 'You have a new notification from BetFusion Zim.'
        })
        break
    }

    if (success) {
      return NextResponse.json({ 
        success: true, 
        message: 'Email sent successfully' 
      })
    } else {
      return NextResponse.json({ 
        error: 'Failed to send email' 
      }, { status: 500 })
    }

  } catch (error) {
    console.error('Error sending email:', error)
    return NextResponse.json({ 
      error: 'Internal server error' 
    }, { status: 500 })
  }
}
