// API endpoint to trigger notifications for various events
import { NextRequest, NextResponse } from 'next/server'
import { NotificationService } from '@/lib/services/notificationService'
import { supabase } from '@/lib/supabase-client'

export async function POST(request: NextRequest) {
  try {
    const { 
      userId, 
      eventType, 
      eventData,
      targetUsers // For bulk notifications
    } = await request.json()

    if (!eventType) {
      return NextResponse.json({ error: 'Event type is required' }, { status: 400 })
    }

    let result

    if (targetUsers && Array.isArray(targetUsers)) {
      // Bulk notifications
      result = await NotificationService.createBulkNotifications(
        targetUsers,
        eventType,
        eventData
      )
    } else if (userId) {
      // Single user notification
      result = await NotificationService.createEventNotification(
        userId,
        eventType,
        eventData
      )
    } else {
      return NextResponse.json({ error: 'User ID or target users required' }, { status: 400 })
    }

    return NextResponse.json({ 
      success: true, 
      notifications: result,
      message: 'Notifications created successfully' 
    })
  } catch (error) {
    console.error('Error creating notifications:', error)
    return NextResponse.json({ 
      error: 'Failed to create notifications' 
    }, { status: 500 })
  }
}

// Helper function to trigger notifications for bet results
export async function triggerBetResultNotification(
  userId: string, 
  marketId: string, 
  outcome: 'won' | 'lost',
  amount: number,
  marketTitle: string
) {
  const eventType = outcome === 'won' ? 'bet_won' : 'bet_lost'
  const eventData = {
    amount,
    marketTitle,
    marketId
  }

  return NotificationService.createEventNotification(userId, eventType, eventData)
}

// Helper function to trigger notifications for new markets
export async function triggerNewMarketNotification(
  marketId: string,
  marketTitle: string,
  userIds?: string[]
) {
  const eventData = {
    marketTitle,
    marketId
  }

  if (userIds) {
    return NotificationService.createBulkNotifications(
      userIds,
      'new_market_available',
      eventData
    )
  } else {
    // Get all active users to notify about new market
    const { data: users } = await supabase
      .from('profiles')
      .select('id')
      .eq('is_active', true)

    if (users && users.length > 0) {
      const userIds = users.map(user => user.id)
      return NotificationService.createBulkNotifications(
        userIds,
        'new_market_available',
        eventData
      )
    }
  }
}

// Helper function to trigger deposit notifications
export async function triggerDepositNotification(
  userId: string,
  amount: number,
  method: string
) {
  const eventData = {
    amount,
    method
  }

  return NotificationService.createEventNotification(userId, 'deposit_received', eventData)
}

// Helper function to trigger withdrawal notifications
export async function triggerWithdrawalNotification(
  userId: string,
  amount: number,
  status: 'processed' | 'failed'
) {
  const eventData = {
    amount,
    status
  }

  const eventType = status === 'processed' ? 'withdrawal_processed' : 'bet_lost'
  return NotificationService.createEventNotification(userId, eventType, eventData)
}
