'use client'

import { useAuth } from '@/components/auth/ClientAuthProvider'
import SimpleSessionWrapper from '@/components/auth/SimpleSessionWrapper'

export default function ApiPage() {
  const { user } = useAuth()

  const generateApiKey = () => {
    // Generate a random API key
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789'
    let result = ''
    for (let i = 0; i < 32; i++) {
      result += chars.charAt(Math.floor(Math.random() * chars.length))
    }
    return result
  }

  const apiKey = generateApiKey()

  return (
    <SimpleSessionWrapper requireAuth={true}>
      <ApiContent user={user} apiKey={apiKey} />
    </SimpleSessionWrapper>
  )
}

function ApiContent({ user, apiKey }: { user: any, apiKey: string }) {

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-8">
        {/* Page Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-black mb-2">API Documentation</h1>
          <p className="text-black">Access your account data and trading functionality via our REST API</p>
        </div>

        {/* API Key Section */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 mb-8">
          <div className="p-6">
            <h2 className="text-xl font-semibold text-black mb-4">API Key</h2>
            <div className="bg-gray-50 rounded-lg p-4 mb-4">
              <div className="flex items-center justify-between">
                <code className="text-sm font-mono text-black break-all">
                  {apiKey}
                </code>
                <button className="ml-4 bg-blue-600 hover:bg-blue-700 text-white px-3 py-1 rounded text-sm font-medium transition-colors">
                  Copy
                </button>
              </div>
            </div>
            <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
              <div className="flex">
                <svg className="w-5 h-5 text-yellow-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                </svg>
                <div className="ml-3">
                  <p className="text-sm text-yellow-800">
                    Keep your API key secure and never share it publicly. This key provides full access to your account.
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* API Endpoints */}
        <div className="space-y-6">
          {/* Markets Endpoint */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-black">Get Markets</h3>
                <span className="bg-green-100 text-green-800 text-xs font-medium px-2 py-1 rounded-full">
                  GET
                </span>
              </div>
              <div className="bg-gray-900 rounded-lg p-4 mb-4">
                <code className="text-green-400 text-sm">
                  GET /api/markets
                </code>
              </div>
              <p className="text-black text-sm mb-4">
                Retrieve all available prediction markets with current prices and volumes.
              </p>
              <div className="bg-gray-50 rounded-lg p-4">
                <h4 className="font-medium text-black mb-2">Response Example:</h4>
                <pre className="text-xs text-black overflow-x-auto">
{`{
  "markets": [
    {
      "id": "uuid",
      "question": "Will Bitcoin reach $100k by 2024?",
      "category": "crypto",
      "yes_price": 0.65,
      "no_price": 0.35,
      "total_volume": 2500000,
      "end_date": "2024-12-31T23:59:59Z",
      "status": "active"
    }
  ]
}`}
                </pre>
              </div>
            </div>
          </div>

          {/* Portfolio Endpoint */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-black">Get Portfolio</h3>
                <span className="bg-green-100 text-green-800 text-xs font-medium px-2 py-1 rounded-full">
                  GET
                </span>
              </div>
              <div className="bg-gray-900 rounded-lg p-4 mb-4">
                <code className="text-green-400 text-sm">
                  GET /api/portfolio
                </code>
              </div>
              <p className="text-black text-sm mb-4">
                Get your current portfolio including balance, active bets, and transaction history.
              </p>
              <div className="bg-gray-50 rounded-lg p-4">
                <h4 className="font-medium text-black mb-2">Response Example:</h4>
                <pre className="text-xs text-black overflow-x-auto">
{`{
  "balance": 1000.00,
  "total_points": 1500,
  "active_bets": [
    {
      "id": "uuid",
      "market_id": "uuid",
      "outcome": "yes",
      "amount": 100.00,
      "status": "matched"
    }
  ],
  "transactions": [...]
}`}
                </pre>
              </div>
            </div>
          </div>

          {/* Place Bet Endpoint */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-black">Place Bet</h3>
                <span className="bg-blue-100 text-blue-800 text-xs font-medium px-2 py-1 rounded-full">
                  POST
                </span>
              </div>
              <div className="bg-gray-900 rounded-lg p-4 mb-4">
                <code className="text-green-400 text-sm">
                  POST /api/bets
                </code>
              </div>
              <p className="text-black text-sm mb-4">
                Place a new bet on a prediction market.
              </p>
              <div className="bg-gray-50 rounded-lg p-4 mb-4">
                <h4 className="font-medium text-black mb-2">Request Body:</h4>
                <pre className="text-xs text-black overflow-x-auto">
{`{
  "market_id": "uuid",
  "outcome": "yes",
  "amount": 100.00,
  "type": "live"
}`}
                </pre>
              </div>
              <div className="bg-gray-50 rounded-lg p-4">
                <h4 className="font-medium text-black mb-2">Response Example:</h4>
                <pre className="text-xs text-black overflow-x-auto">
{`{
  "success": true,
  "bet_id": "uuid",
  "status": "matched",
  "message": "Bet placed successfully"
}`}
                </pre>
              </div>
            </div>
          </div>

          {/* Wallet Endpoint */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-black">Wallet Transactions</h3>
                <span className="bg-green-100 text-green-800 text-xs font-medium px-2 py-1 rounded-full">
                  GET
                </span>
              </div>
              <div className="bg-gray-900 rounded-lg p-4 mb-4">
                <code className="text-green-400 text-sm">
                  GET /api/wallet/transactions
                </code>
              </div>
              <p className="text-black text-sm mb-4">
                Get your wallet transaction history including deposits and withdrawals.
              </p>
              <div className="bg-gray-50 rounded-lg p-4">
                <h4 className="font-medium text-black mb-2">Response Example:</h4>
                <pre className="text-xs text-black overflow-x-auto">
{`{
  "transactions": [
    {
      "id": "uuid",
      "type": "deposit",
      "amount": 500.00,
      "status": "completed",
      "created_at": "2024-01-01T00:00:00Z"
    }
  ]
}`}
                </pre>
              </div>
            </div>
          </div>
        </div>

        {/* Rate Limits */}
        <div className="bg-blue-50 border border-blue-200 rounded-xl p-6 mt-8">
          <div className="flex">
            <svg className="w-5 h-5 text-blue-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
            <div className="ml-3">
              <h3 className="text-sm font-medium text-blue-800">Rate Limits</h3>
              <div className="mt-2 text-sm text-blue-700">
                <ul className="list-disc list-inside space-y-1">
                  <li>100 requests per minute for authenticated endpoints</li>
                  <li>10 requests per minute for unauthenticated endpoints</li>
                  <li>Rate limit headers are included in all responses</li>
                </ul>
              </div>
            </div>
          </div>
        </div>

        {/* Authentication */}
        <div className="bg-gray-50 border border-gray-200 rounded-xl p-6 mt-6">
          <h3 className="text-lg font-semibold text-black mb-4">Authentication</h3>
          <p className="text-black mb-4">
            All API requests require authentication using your API key. Include it in the Authorization header:
          </p>
          <div className="bg-gray-900 rounded-lg p-4">
            <code className="text-green-400 text-sm">
              Authorization: Bearer {apiKey}
            </code>
          </div>
        </div>
      </div>
    </div>
  )
}
