// Test endpoint to simulate live data updates
import { NextRequest, NextResponse } from 'next/server'
import { supabase } from '@/lib/supabase-client'

export async function POST(request: NextRequest) {
  try {
    const { userId, amount } = await request.json()
    
    if (!userId || !amount) {
      return NextResponse.json({ error: 'Missing userId or amount' }, { status: 400 })
    }

    // Update user balance to test live updates
    const { error } = await supabase
      .from('profiles')
      .update({
        balance: amount,
        updated_at: new Date().toISOString()
      })
      .eq('id', userId)

    if (error) {
      console.error('Error updating balance:', error)
      return NextResponse.json({ error: 'Failed to update balance' }, { status: 500 })
    }

    // Create a test transaction
    const { error: transactionError } = await supabase
      .from('unified_transactions')
      .insert({
        user_id: userId,
        type: 'deposit',
        status: 'completed',
        amount: amount,
        currency: 'USD',
        description: 'Test live update transaction',
        processed_at: new Date().toISOString()
      })

    if (transactionError) {
      console.error('Error creating transaction:', transactionError)
      // Don't fail the request, just log the error
    }

    return NextResponse.json({ 
      success: true, 
      message: 'Balance updated successfully',
      newBalance: amount 
    })
  } catch (error) {
    console.error('Test live update error:', error)
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 })
  }
}
