import { NextRequest, NextResponse } from 'next/server'
import { supabase } from '@/lib/supabase-client'
import { loginTracking } from '@/lib/services/loginTracking'

export async function POST(request: NextRequest) {
  try {
    const { userId, sessionToken } = await request.json()
    
    if (!userId || !sessionToken) {
      return NextResponse.json({
        success: false,
        error: 'Missing userId or sessionToken'
      })
    }
    
    // Get IP address from request headers
    const forwarded = request.headers.get('x-forwarded-for')
    const realIp = request.headers.get('x-real-ip')
    const clientIp = request.headers.get('x-client-ip')
    
    // Try to get the real IP address
    let ipAddress = forwarded?.split(',')[0]?.trim() || 
                   realIp || 
                   clientIp || 
                   '127.0.0.1' // Default to localhost for development
    
    console.log('Track Login API: IP address detected:', ipAddress)
    console.log('Track Login API: Headers:', {
      'x-forwarded-for': forwarded,
      'x-real-ip': realIp,
      'x-client-ip': clientIp
    })
    
    // Track the login with IP address
    console.log('Track Login API: Attempting to track login...', { userId, sessionToken, ipAddress })
    const sessionId = await loginTracking.trackLogin(userId, sessionToken, ipAddress)
    console.log('Track Login API: trackLogin result:', sessionId)
    
    if (sessionId) {
      return NextResponse.json({
        success: true,
        sessionId,
        ipAddress
      })
    } else {
      console.error('Track Login API: trackLogin returned null/undefined')
      return NextResponse.json({
        success: false,
        error: 'Failed to track login session - check if login_sessions table exists and environment variables are set'
      })
    }
    
  } catch (error) {
    console.error('Track Login API error:', error)
    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error'
    })
  }
}
