// PayNow Zim Webhook Handler
import { NextRequest, NextResponse } from 'next/server'
import { payNowZimService } from '@/lib/payments/paynow-zim'
import { supabase } from '@/lib/supabase-client'

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    
    // Process the webhook
    const result = await payNowZimService.processWebhook(body)
    
    if (result.success) {
      // Update transaction status in database
      await updateTransactionStatus(body.transactionId, body.status)
      
      return NextResponse.json({ success: true, message: 'Webhook processed successfully' })
    } else {
      console.error('PayNow Zim webhook processing failed:', result.error)
      return NextResponse.json({ error: result.error }, { status: 400 })
    }
  } catch (error) {
    console.error('PayNow Zim webhook error:', error)
    return NextResponse.json({ error: 'Webhook processing failed' }, { status: 500 })
  }
}

async function updateTransactionStatus(transactionId: string, status: string) {
  try {
    // Update transaction status in database
    const { error } = await supabase
      .from('transactions')
      .update({
        status: status === 'completed' ? 'completed' : 'failed',
        processed_at: new Date().toISOString()
      })
      .eq('external_transaction_id', transactionId)
      .eq('type', 'deposit')

    if (error) {
      console.error('Failed to update transaction status:', error)
    }

    // If completed, update user balance
    if (status === 'completed') {
      const { data: transaction } = await supabase
        .from('transactions')
        .select('user_id, amount')
        .eq('external_transaction_id', transactionId)
        .eq('type', 'deposit')
        .single()

      if (transaction) {
        const { error: balanceError } = await supabase
          .rpc('increment_balance', {
            user_id: transaction.user_id,
            amount: transaction.amount
          })

        if (balanceError) {
          console.error('Failed to update user balance:', balanceError)
        }
      }
    }
  } catch (error) {
    console.error('Error updating transaction status:', error)
  }
}
