'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

interface Bet {
  id: string
  market_id: string
  outcome: 'yes' | 'no'
  amount: number
  seeded_amount: number
  live_amount: number
  price: number
  shares: number
  status: string
  created_at: string
  matched_at: string | null
  markets: {
    title: string
    status: string
    outcome: string | null
    end_date: string
    yes_price: number
    no_price: number
  }
}

export default function BetHistoryPage() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true}>
      <BetHistoryContent user={user} />
    </FastSessionWrapper>
  )
}

function BetHistoryContent({ user }: { user: any }) {
  const router = useRouter()
  const [bets, setBets] = useState<Bet[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState('')
  const [filter, setFilter] = useState<'all' | 'seeded' | 'live' | 'matched' | 'won' | 'lost'>('all')

  const fetchBets = async () => {
    try {
      const { data, error } = await supabase
        .from('bets')
        .select(`
          id,
          market_id,
          outcome,
          amount,
          seeded_amount,
          live_amount,
          price,
          shares,
          status,
          created_at,
          matched_at,
          markets!inner (
            title,
            status,
            outcome,
            end_date,
            yes_price,
            no_price
          )
        `)
        .eq('user_id', user.id)
        .order('created_at', { ascending: false })

      if (error) throw error

      // Transform the data to match our interface
      const transformedData = (data || []).map((bet: any) => ({
        id: bet.id,
        market_id: bet.market_id,
        outcome: bet.outcome,
        amount: bet.amount,
        seeded_amount: bet.seeded_amount,
        live_amount: bet.live_amount,
        price: bet.price,
        shares: bet.shares,
        status: bet.status,
        created_at: bet.created_at,
        matched_at: bet.matched_at,
        markets: bet.markets
      }))

      setBets(transformedData)
    } catch (err) {
      console.error('Error fetching bets:', err)
      setError('Failed to load bet history')
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    if (user) {
      fetchBets()
    }
  }, [user])

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    })
  }

  const formatPrice = (price: number) => {
    return (price * 100).toFixed(1) + '%'
  }

  const getBetStatusColor = (status: string) => {
    switch (status) {
      case 'won': return 'text-green-600 bg-green-100'
      case 'lost': return 'text-red-600 bg-red-100'
      case 'seeded': return 'text-purple-600 bg-purple-100'
      case 'live': return 'text-blue-600 bg-blue-100'
      case 'matched': return 'text-yellow-600 bg-yellow-100'
      default: return 'text-black bg-gray-100'
    }
  }

  const calculatePayout = (bet: Bet) => {
    if (bet.status === 'won' && bet.markets.outcome === bet.outcome) {
      // Calculate payout based on seeded amount and current market prices
      const currentPrice = bet.outcome === 'yes' ? bet.markets.yes_price : bet.markets.no_price
      return bet.seeded_amount / bet.price
    }
    return 0
  }

  const filteredBets = bets.filter(bet => {
    if (filter === 'all') return true
    if (filter === 'won') return bet.status === 'won' || (bet.markets.outcome && bet.markets.outcome === bet.outcome)
    if (filter === 'lost') return bet.status === 'lost' || (bet.markets.outcome && bet.markets.outcome !== bet.outcome)
    return bet.status === filter
  })

  const totalInvested = bets.reduce((sum, bet) => sum + bet.amount + bet.seeded_amount, 0)
  const totalWon = bets.filter(bet => bet.markets.outcome && bet.markets.outcome === bet.outcome)
    .reduce((sum, bet) => sum + calculatePayout(bet), 0)
  const totalLost = bets.filter(bet => bet.markets.outcome && bet.markets.outcome !== bet.outcome)
    .reduce((sum, bet) => sum + bet.amount + bet.seeded_amount, 0)
  const netProfit = totalWon - totalLost

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <LoadingSpinner size="lg" />
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-6xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center space-x-4 mb-4">
            <Link href="/" className="text-black hover:text-black">
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
              </svg>
            </Link>
            <h1 className="text-3xl font-bold text-black">Bet History</h1>
          </div>
          <p className="text-black">
            Track all your betting activity and performance across prediction markets.
          </p>
        </div>

        {error && (
          <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg">
            <div className="flex items-center">
              <svg className="w-5 h-5 text-red-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              <p className="text-red-700 text-sm">{error}</p>
            </div>
          </div>
        )}

        {/* Stats Overview */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Total Invested</div>
            <div className="text-2xl font-bold text-black">
              {formatCurrency(totalInvested)}
            </div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Total Won</div>
            <div className="text-2xl font-bold text-green-600">
              {formatCurrency(totalWon)}
            </div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Total Lost</div>
            <div className="text-2xl font-bold text-red-600">
              {formatCurrency(totalLost)}
            </div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Net Profit</div>
            <div className={`text-2xl font-bold ${netProfit >= 0 ? 'text-green-600' : 'text-red-600'}`}>
              {formatCurrency(netProfit)}
            </div>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
          <div className="flex flex-wrap gap-2">
            {[
              { key: 'all', label: 'All Bets' },
              { key: 'seeded', label: 'Seeded' },
              { key: 'live', label: 'Live' },
              { key: 'matched', label: 'Matched' },
              { key: 'won', label: 'Won' },
              { key: 'lost', label: 'Lost' }
            ].map(({ key, label }) => (
              <button
                key={key}
                onClick={() => setFilter(key as any)}
                className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                  filter === key
                    ? 'bg-blue-600 text-white'
                    : 'bg-gray-100 text-black hover:bg-gray-200'
                }`}
              >
                {label}
              </button>
            ))}
          </div>
        </div>

        {/* Bets List */}
        {filteredBets.length === 0 ? (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8 text-center">
            <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg className="w-8 h-8 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
              </svg>
            </div>
            <h3 className="text-lg font-semibold text-black mb-2">No Bets Found</h3>
            <p className="text-black mb-6">
              {filter === 'all' 
                ? "You haven't placed any bets yet."
                : `No ${filter} bets found.`
              }
            </p>
            <Link 
              href="/"
              className="inline-flex items-center px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white font-medium rounded-lg transition-colors"
            >
              Browse Markets
            </Link>
          </div>
        ) : (
          <div className="space-y-4">
            {filteredBets.map((bet) => (
              <div key={bet.id} className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                <div className="flex items-start justify-between mb-4">
                  <div className="flex-1">
                    <h3 className="text-lg font-semibold text-black mb-2">
                      {bet.markets.title}
                    </h3>
                    <div className="flex items-center space-x-4 text-sm text-black mb-2">
                      <span>Placed: {formatDate(bet.created_at)}</span>
                      {bet.matched_at && (
                        <>
                          <span>•</span>
                          <span>Matched: {formatDate(bet.matched_at)}</span>
                        </>
                      )}
                      <span>•</span>
                      <span>Ends: {formatDate(bet.markets.end_date)}</span>
                    </div>
                  </div>
                  <Link 
                    href={`/markets/${bet.market_id}`}
                    className="text-blue-600 hover:text-blue-700 text-sm font-medium"
                  >
                    View Market →
                  </Link>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
                  {/* Bet Details */}
                  <div className="space-y-3">
                    <div className="text-sm font-medium text-black mb-2">Bet Details</div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Position:</span>
                      <span className={`text-sm font-bold ${bet.outcome === 'yes' ? 'text-green-600' : 'text-red-600'}`}>
                        {bet.outcome.toUpperCase()}
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Total Amount:</span>
                      <span className="text-sm font-bold text-black">
                        {formatCurrency(bet.amount + bet.seeded_amount)}
                      </span>
                    </div>
                    {bet.seeded_amount > 0 && (
                      <div className="flex items-center justify-between">
                        <span className="text-sm text-black">Seeded:</span>
                        <span className="text-sm font-bold text-purple-600">
                          {formatCurrency(bet.seeded_amount)}
                        </span>
                      </div>
                    )}
                    {bet.live_amount > 0 && (
                      <div className="flex items-center justify-between">
                        <span className="text-sm text-black">Live:</span>
                        <span className="text-sm font-bold text-blue-600">
                          {formatCurrency(bet.live_amount)}
                        </span>
                      </div>
                    )}
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Price:</span>
                      <span className="text-sm font-bold text-black">
                        {formatPrice(bet.price)}
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Shares:</span>
                      <span className="text-sm font-bold text-black">
                        {bet.shares.toFixed(2)}
                      </span>
                    </div>
                  </div>

                  {/* Market Status */}
                  <div className="space-y-3">
                    <div className="text-sm font-medium text-black mb-2">Market Status</div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Status:</span>
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        bet.markets.status === 'active' ? 'bg-green-100 text-green-800' :
                        bet.markets.status === 'resolved' ? 'bg-blue-100 text-blue-800' :
                        'bg-gray-100 text-black'
                      }`}>
                        {bet.markets.status}
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Current YES:</span>
                      <span className="text-sm font-bold text-green-600">
                        {formatPrice(bet.markets.yes_price)}
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Current NO:</span>
                      <span className="text-sm font-bold text-red-600">
                        {formatPrice(bet.markets.no_price)}
                      </span>
                    </div>
                    {bet.markets.outcome && (
                      <div className="flex items-center justify-between">
                        <span className="text-sm text-black">Resolved:</span>
                        <span className={`text-sm font-bold ${
                          bet.markets.outcome === 'yes' ? 'text-green-600' : 'text-red-600'
                        }`}>
                          {bet.markets.outcome.toUpperCase()}
                        </span>
                      </div>
                    )}
                  </div>

                  {/* Performance */}
                  <div className="space-y-3">
                    <div className="text-sm font-medium text-black mb-2">Performance</div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Bet Status:</span>
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${getBetStatusColor(bet.status)}`}>
                        {bet.status}
                      </span>
                    </div>
                    {bet.markets.outcome && (
                      <>
                        <div className="flex items-center justify-between">
                          <span className="text-sm text-black">Result:</span>
                          <span className={`text-sm font-bold ${
                            bet.markets.outcome === bet.outcome ? 'text-green-600' : 'text-red-600'
                          }`}>
                            {bet.markets.outcome === bet.outcome ? 'WON' : 'LOST'}
                          </span>
                        </div>
                        {bet.markets.outcome === bet.outcome && (
                          <div className="flex items-center justify-between">
                            <span className="text-sm text-black">Payout:</span>
                            <span className="text-sm font-bold text-green-600">
                              {formatCurrency(calculatePayout(bet))}
                            </span>
                          </div>
                        )}
                      </>
                    )}
                  </div>

                  {/* Actions */}
                  <div className="space-y-3">
                    <div className="text-sm font-medium text-black mb-2">Actions</div>
                    <Link 
                      href={`/markets/${bet.market_id}`}
                      className="block w-full bg-blue-600 hover:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium transition-colors text-sm text-center"
                    >
                      View Market
                    </Link>
                    {bet.status === 'seeded' && (
                      <Link 
                        href="/seeded-bets"
                        className="block w-full bg-purple-600 hover:bg-purple-700 text-white py-2 px-4 rounded-lg font-medium transition-colors text-sm text-center"
                      >
                        Manage Seeded Bet
                      </Link>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  )
}
