'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { LoadingSpinner, ButtonLoadingSpinner, PageLoadingSpinner } from '@/components/ui/LoadingSpinner'
import ImageUpload from '@/components/ui/ImageUpload'
import { useToast } from '@/components/ui/Toast'
import MultiOutcomeCreator from '@/components/markets/MultiOutcomeCreator'

export default function CreateMarketPage() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true}>
      <CreateMarketContent user={user} />
    </FastSessionWrapper>
  )
}

function CreateMarketContent({ user }: { user: any }) {
  const router = useRouter()
  const [loading, setLoading] = useState(false)
  const [pageLoading, setPageLoading] = useState(true)
  const [error, setError] = useState('')
  const [imageError, setImageError] = useState('')
  const [success, setSuccess] = useState(false)
  const { addToast } = useToast()
  const [formData, setFormData] = useState({
    title: '',
    description: '',
    category: 'politics',
    end_date: '',
    image_url: '',
    market_type: 'binary',
    seeding_amount: 0,
    liquidity_amount: 0,
    total_shares: 1000, // Total number of shares to be distributed
    min_bet_amount: 1,
    max_bet_amount: 1000,
    refund_threshold: 0.1, // 10% of seeded amount must be traded
    refund_deadline: 7 // days after market creation
  })

  // Multi-outcome state
  const [outcomes, setOutcomes] = useState<Array<{
    id: string
    name: string
    description: string
    color: string
    icon: string
  }>>([])

  // Handle page loading
  useEffect(() => {
    // Simulate page loading for better UX
    const timer = setTimeout(() => {
      setPageLoading(false)
    }, 500)

    return () => clearTimeout(timer)
  }, [])

  const categories = [
    { value: 'politics', label: 'Politics' },
    { value: 'sports', label: 'Sports' },
    { value: 'crypto', label: 'Cryptocurrency' },
    { value: 'tech', label: 'Technology' },
    { value: 'economics', label: 'Economics' },
    { value: 'science', label: 'Science' },
    { value: 'entertainment', label: 'Entertainment' },
    { value: 'other', label: 'Other' }
  ]

  const marketTypes = [
    { value: 'binary', label: 'Binary (Yes/No)' },
    { value: 'multi_outcome', label: 'Multiple Outcomes' }
  ]

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    const { name, value } = e.target
    setFormData(prev => ({
      ...prev,
      [name]: value
    }))
  }

  const handleImageChange = (url: string) => {
    setFormData(prev => ({
      ...prev,
      image_url: url
    }))
  }

  const handleImageError = (error: string) => {
    setImageError(error)
  }

  // Calculate initial share price based on seeding amount and total shares
  const initialSharePrice = formData.total_shares > 0 
    ? (parseFloat(formData.seeding_amount.toString()) / parseInt(formData.total_shares.toString())).toFixed(4)
    : '0.0000'

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError('')
    setImageError('')

    try {
      // Validate form data
      if (!formData.title.trim()) {
        throw new Error('Market title is required')
      }
      if (!formData.description.trim()) {
        throw new Error('Market description is required')
      }
      if (!formData.end_date) {
        throw new Error('End date is required')
      }

      // Check if end date is in the future
      const endDate = new Date(formData.end_date)
      const now = new Date()
      if (endDate <= now) {
        throw new Error('End date must be in the future')
      }

      // Validate seeding and liquidity amounts
      if (formData.seeding_amount < 0) {
        throw new Error('Seeding amount cannot be negative')
      }
      if (formData.liquidity_amount < 0) {
        throw new Error('Liquidity amount cannot be negative')
      }
      if (formData.seeding_amount === 0 && formData.liquidity_amount === 0) {
        throw new Error('You must provide either seeding amount or liquidity amount to create a trusted market')
      }

      // Validate bet amounts
      if (formData.min_bet_amount <= 0) {
        throw new Error('Minimum bet amount must be greater than 0')
      }
      if (formData.max_bet_amount <= formData.min_bet_amount) {
        throw new Error('Maximum bet amount must be greater than minimum bet amount')
      }

      // Validate total shares
      if (formData.total_shares <= 0) {
        throw new Error('Total shares must be greater than 0')
      }
      if (formData.total_shares < 100) {
        throw new Error('Total shares must be at least 100 for proper price granularity')
      }

      // Check user balance (we'll need to fetch this)
      const { data: profile, error: profileError } = await supabase
        .from('profiles')
        .select('balance')
        .eq('id', user.id)
        .single()

      if (profileError) {
        throw new Error('Failed to fetch user balance')
      }

      const seedingAmount = parseFloat(formData.seeding_amount.toString()) || 0
      const liquidityAmount = parseFloat(formData.liquidity_amount.toString()) || 0
      const totalRequired = seedingAmount + liquidityAmount
      
      if (isNaN(totalRequired) || totalRequired <= 0) {
        throw new Error('Invalid amount values')
      }
      
      if (profile.balance < totalRequired) {
        throw new Error(`Insufficient balance. You need $${totalRequired.toFixed(2)} but only have $${profile.balance.toFixed(2)}`)
      }

      // Create market with seeding and liquidity
      const { data, error: createError } = await supabase
        .from('markets')
        .insert({
          title: formData.title.trim(),
          description: formData.description.trim(),
          category: formData.category,
          end_date: formData.end_date,
          image_url: formData.image_url.trim() || null,
          creator_id: user.id,
          status: 'pending',
          market_type: formData.market_type,
          total_volume: 0,
          total_liquidity: formData.liquidity_amount,
          total_shares: formData.total_shares,
          initial_share_price: parseFloat(initialSharePrice),
          yes_pool: formData.market_type === 'binary' ? formData.seeding_amount / 2 : 0,
          no_pool: formData.market_type === 'binary' ? formData.seeding_amount / 2 : 0,
          yes_price: formData.market_type === 'binary' ? 0.5 : null,
          no_price: formData.market_type === 'binary' ? 0.5 : null,
          seeded_volume: formData.seeding_amount,
          live_volume: 0,
          creator_earnings: 0,
          admin_fee: 0.02, // 2% admin fee
          min_bet_amount: formData.min_bet_amount,
          max_bet_amount: formData.max_bet_amount,
          refund_threshold: formData.refund_threshold,
          refund_deadline: formData.refund_deadline
        })
        .select()
        .single()

      if (createError) {
        throw new Error(createError.message)
      }

      // Deduct seeding and liquidity amounts from user balance
      const { error: balanceError } = await supabase
        .from('profiles')
        .update({
          balance: profile.balance - totalRequired
        })
        .eq('id', user.id)

      if (balanceError) {
        throw new Error('Failed to deduct balance for market creation')
      }

      // If it's a multi-outcome market, create the outcomes
      if (formData.market_type === 'multi_outcome') {
        // Validate outcomes
        if (outcomes.length < 2) {
          throw new Error('Multi-outcome markets require at least 2 outcomes')
        }

        // Validate outcome names
        const emptyOutcomes = outcomes.filter(outcome => !outcome.name.trim())
        if (emptyOutcomes.length > 0) {
          throw new Error('All outcomes must have a name')
        }

        // Check for duplicate names
        const names = outcomes.map(outcome => outcome.name.trim().toLowerCase())
        const uniqueNames = new Set(names)
        if (names.length !== uniqueNames.size) {
          throw new Error('Outcome names must be unique')
        }

        // Create outcomes with equal initial distribution
        const outcomeAmount = formData.seeding_amount / outcomes.length
        const outcomePrice = 1 / outcomes.length

        const outcomesToInsert = outcomes.map((outcome, index) => ({
          market_id: data.id,
          name: outcome.name.trim(),
          description: outcome.description.trim() || null,
          price: outcomePrice,
          pool: outcomeAmount,
          volume: 0,
          sort_order: index + 1,
          color: outcome.color,
          icon: outcome.icon
        }))

        const { error: outcomesError } = await supabase
          .from('market_outcomes')
          .insert(outcomesToInsert)

        if (outcomesError) {
          throw new Error('Failed to create market outcomes')
        }
      }

      // Show success message and redirect
      addToast({
        type: 'success',
        title: 'Market Created Successfully!',
        message: `Your market "${formData.title}" has been created and is now live.`
      })
      setSuccess(true)
      setTimeout(() => {
        router.push(`/markets/${data.id}`)
      }, 1500)
    } catch (err) {
      console.error('Error creating market:', err)
      const errorMessage = err instanceof Error ? err.message : 'Failed to create market'
      setError(errorMessage)
      addToast({
        type: 'error',
        title: 'Market Creation Failed',
        message: errorMessage
      })
    } finally {
      setLoading(false)
    }
  }

  // Show page loading state
  if (pageLoading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading Create Market...</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 relative">
      {loading && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-xl p-8 max-w-md mx-4 text-center">
            <div className="animate-spin rounded-full h-16 w-16 border-b-4 border-blue-600 mx-auto mb-4"></div>
            <h3 className="text-xl font-semibold text-black mb-2">Creating Market...</h3>
            <p className="text-gray-600 text-sm">Please wait while we process your market creation</p>
            <div className="mt-4 bg-blue-50 rounded-lg p-3">
              <p className="text-blue-700 text-xs">This may take a few moments</p>
            </div>
          </div>
        </div>
      )}
      <div className="max-w-2xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center space-x-4 mb-4">
            <Link href="/" className="text-black hover:text-black">
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
              </svg>
            </Link>
            <h1 className="text-3xl font-bold text-black">Create New Market</h1>
          </div>
          <p className="text-black">
            Create a prediction market for others to trade on. Make sure your question is clear and answerable.
          </p>
        </div>

        {/* Form */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 sm:p-8">
          {error && (
            <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg">
              <div className="flex items-center">
                <svg className="w-5 h-5 text-red-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <p className="text-red-700 text-sm">{error}</p>
              </div>
            </div>
          )}

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Market Title */}
            <div>
              <label htmlFor="title" className="block text-sm font-medium text-black mb-2">
                Market Question *
              </label>
              <input
                type="text"
                id="title"
                name="title"
                value={formData.title}
                onChange={handleInputChange}
                placeholder="e.g., Will Bitcoin reach $100,000 by end of 2024?"
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                style={{ color: 'black' }}
                required
              />
              <p className="mt-1 text-sm text-black">
                Make your question clear and specific. It should have a definitive yes/no answer.
              </p>
            </div>

            {/* Description */}
            <div>
              <label htmlFor="description" className="block text-sm font-medium text-black mb-2">
                Description *
              </label>
              <textarea
                id="description"
                name="description"
                value={formData.description}
                onChange={handleInputChange}
                placeholder="Provide more details about your market question..."
                rows={4}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors resize-none"
                style={{ color: 'black' }}
                required
              />
              <p className="mt-1 text-sm text-black">
                Add context, rules, or clarification for your market.
              </p>
            </div>

            {/* Category */}
            <div>
              <label htmlFor="category" className="block text-sm font-medium text-black mb-2">
                Category
              </label>
              <select
                id="category"
                name="category"
                value={formData.category}
                onChange={handleInputChange}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors !text-black"
               style={{ color: 'black' }}>
                {categories.map((category) => (
                  <option key={category.value} value={category.value}>
                    {category.label}
                  </option>
                ))}
              </select>
            </div>

            {/* End Date */}
            <div>
              <label htmlFor="end_date" className="block text-sm font-medium text-black mb-2">
                End Date *
              </label>
              <input
                type="datetime-local"
                id="end_date"
                name="end_date"
                value={formData.end_date}
                onChange={handleInputChange}
                min={new Date().toISOString().slice(0, 16)}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors !text-black"
                required
              style={{ color: 'black' }} />
              <p className="mt-1 text-sm text-black">
                When should this market close? Must be in the future.
              </p>
            </div>

            {/* Market Type */}
            <div>
              <label htmlFor="market_type" className="block text-sm font-medium text-black mb-2">
                Market Type
              </label>
              <select
                id="market_type"
                name="market_type"
                value={formData.market_type}
                onChange={handleInputChange}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors !text-black"
               style={{ color: 'black' }}>
                {marketTypes.map((type) => (
                  <option key={type.value} value={type.value}>
                    {type.label}
                  </option>
                ))}
              </select>
              <p className="mt-1 text-sm text-black">
                Choose between binary (yes/no) or multiple outcomes.
              </p>
            </div>

            {/* Multi-Outcome Creator */}
            {formData.market_type === 'multi_outcome' && (
              <div className="bg-gray-50 border border-gray-200 rounded-lg p-6">
                <MultiOutcomeCreator
                  outcomes={outcomes}
                  onChange={setOutcomes}
                  maxOutcomes={10}
                />
                {outcomes.length < 2 && formData.market_type === 'multi_outcome' && (
                  <div className="mt-4 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                    <p className="text-yellow-800 text-sm">
                      <strong>Note:</strong> Multi-outcome markets require at least 2 outcomes. 
                      For example, "Who will be president?" could have outcomes like "John Doe", "Jane Smith", etc.
                    </p>
                  </div>
                )}
              </div>
            )}

            {/* Seeding Amount */}
            <div>
              <label htmlFor="seeding_amount" className="block text-sm font-medium text-black mb-2">
                Seeding Amount ($) *
              </label>
              <input
                type="number"
                id="seeding_amount"
                name="seeding_amount"
                value={formData.seeding_amount}
                onChange={handleInputChange}
                min="0"
                step="0.01"
                placeholder="0.00"
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors !text-black"
                required
              style={{ color: 'black' }} />
              <p className="mt-1 text-sm text-black">
                Initial liquidity to start the market. This creates the initial price points.
              </p>
            </div>

            {/* Liquidity Amount */}
            <div>
              <label htmlFor="liquidity_amount" className="block text-sm font-medium text-black mb-2">
                Additional Liquidity ($)
              </label>
              <input
                type="number"
                id="liquidity_amount"
                name="liquidity_amount"
                value={formData.liquidity_amount}
                onChange={handleInputChange}
                min="0"
                step="0.01"
                placeholder="0.00"
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors !text-black"
              style={{ color: 'black' }} />
              <p className="mt-1 text-sm text-black">
                Additional liquidity to make the market more attractive to traders.
              </p>
            </div>

            {/* Total Shares */}
            <div>
              <label htmlFor="total_shares" className="block text-sm font-medium text-black mb-2">
                Total Shares *
              </label>
              <input
                type="number"
                id="total_shares"
                name="total_shares"
                value={formData.total_shares}
                onChange={handleInputChange}
                min="100"
                step="1"
                placeholder="1000"
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors !text-black"
              style={{ color: 'black' }}
                required />
              <div className="mt-2 p-3 bg-blue-50 border border-blue-200 rounded-lg">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-blue-800 font-medium">Initial Share Price:</span>
                  <span className="text-blue-900 font-bold">${initialSharePrice}</span>
                </div>
                <p className="mt-1 text-xs text-blue-700">
                  Calculated as: Seeding Amount ÷ Total Shares = ${parseFloat(formData.seeding_amount.toString()).toFixed(2)} ÷ {formData.total_shares} = ${initialSharePrice}
                </p>
              </div>
              <p className="mt-1 text-sm text-black">
                Total number of shares to be distributed. This determines the initial price per share.
              </p>
            </div>

            {/* Bet Limits */}
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
              <div>
                <label htmlFor="min_bet_amount" className="block text-sm font-medium text-black mb-2">
                  Minimum Bet ($)
                </label>
                <input
                  type="number"
                  id="min_bet_amount"
                  name="min_bet_amount"
                  value={formData.min_bet_amount}
                  onChange={handleInputChange}
                  min="0.01"
                  step="0.01"
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors !text-black"
                style={{ color: 'black' }} />
              </div>
              <div>
                <label htmlFor="max_bet_amount" className="block text-sm font-medium text-black mb-2">
                  Maximum Bet ($)
                </label>
                <input
                  type="number"
                  id="max_bet_amount"
                  name="max_bet_amount"
                  value={formData.max_bet_amount}
                  onChange={handleInputChange}
                  min="1"
                  step="1"
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors !text-black"
                style={{ color: 'black' }} />
              </div>
            </div>

            {/* Refund Settings */}
            <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
              <h4 className="text-sm font-medium text-yellow-800 mb-3">Refund Protection</h4>
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <div>
                  <label htmlFor="refund_threshold" className="block text-sm font-medium text-yellow-700 mb-2">
                    Trading Threshold (%)
                  </label>
                  <input
                    type="number"
                    id="refund_threshold"
                    name="refund_threshold"
                    value={formData.refund_threshold}
                    onChange={handleInputChange}
                    min="0"
                    max="1"
                    step="0.01"
                    className="w-full px-4 py-3 border border-yellow-300 rounded-lg focus:ring-2 focus:ring-yellow-500 focus:border-yellow-500 transition-colors !text-black"
                  style={{ color: 'black' }} />
                  <p className="mt-1 text-xs text-yellow-600">
                    Minimum % of seeded amount that must be traded to avoid refund.
                  </p>
                </div>
                <div>
                  <label htmlFor="refund_deadline" className="block text-sm font-medium text-yellow-700 mb-2">
                    Refund Deadline (days)
                  </label>
                  <input
                    type="number"
                    id="refund_deadline"
                    name="refund_deadline"
                    value={formData.refund_deadline}
                    onChange={handleInputChange}
                    min="1"
                    max="30"
                    className="w-full px-4 py-3 border border-yellow-300 rounded-lg focus:ring-2 focus:ring-yellow-500 focus:border-yellow-500 transition-colors !text-black"
                  style={{ color: 'black' }} />
                  <p className="mt-1 text-xs text-yellow-600">
                    Days after creation when refund check occurs.
                  </p>
                </div>
              </div>
            </div>

            {/* Image Upload */}
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Market Image (Optional)
              </label>
              {imageError && (
                <div className="mb-3 p-3 bg-red-50 border border-red-200 rounded-lg">
                  <div className="flex items-center">
                    <svg className="w-4 h-4 text-red-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                    </svg>
                    <p className="text-red-700 text-sm">{imageError}</p>
                  </div>
                </div>
              )}
              <ImageUpload
                value={formData.image_url}
                onChange={handleImageChange}
                onError={handleImageError}
                placeholder="Upload an image or enter URL"
                disabled={loading}
              />
            </div>

            {/* Submit Button */}
            <div className="pt-4">
              <button
                type="submit"
                disabled={loading}
                className="w-full bg-blue-600 hover:bg-blue-700 disabled:bg-blue-400 text-white font-medium py-3 px-6 rounded-lg transition-colors flex items-center justify-center"
              >
                {loading ? (
                  <ButtonLoadingSpinner text="Creating Market..." />
                ) : (
                  'Create Market'
                )}
              </button>
            </div>
          </form>
        </div>

        {/* Guidelines */}
        <div className="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-blue-900 mb-3">Market Creation Guidelines</h3>
          <ul className="space-y-2 text-sm text-blue-800">
            <li className="flex items-start">
              <span className="text-blue-600 mr-2">•</span>
              Questions should be clear, specific, and have a definitive answer
            </li>
            <li className="flex items-start">
              <span className="text-blue-600 mr-2">•</span>
              Avoid subjective questions that can't be objectively resolved
            </li>
            <li className="flex items-start">
              <span className="text-blue-600 mr-2">•</span>
              Set realistic end dates that allow for sufficient trading activity
            </li>
            <li className="flex items-start">
              <span className="text-blue-600 mr-2">•</span>
              <strong>Seeding Amount:</strong> Required to create initial liquidity and price points
            </li>
            <li className="flex items-start">
              <span className="text-blue-600 mr-2">•</span>
              <strong>Total Shares:</strong> Determines initial share price (Seeding Amount ÷ Total Shares)
            </li>
            <li className="flex items-start">
              <span className="text-blue-600 mr-2">•</span>
              <strong>Refund Protection:</strong> If your market doesn't attract enough traders, you'll get refunded
            </li>
            <li className="flex items-start">
              <span className="text-blue-600 mr-2">•</span>
              Markets will be reviewed and may be removed if they violate our terms
            </li>
          </ul>
        </div>

        {/* Share Pricing Info */}
        <div className="mt-6 bg-purple-50 border border-purple-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-purple-900 mb-3">How Share Pricing Works</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm text-purple-800">
            <div>
              <h4 className="font-medium text-purple-900 mb-2">📊 Initial Price Calculation</h4>
              <div className="bg-white p-3 rounded-lg border border-purple-200 mb-3">
                <div className="font-mono text-sm">
                  <div>Initial Share Price =</div>
                  <div className="text-purple-600">Seeding Amount ÷ Total Shares</div>
                </div>
              </div>
              <ul className="space-y-1">
                <li>• Higher share count = Lower price per share</li>
                <li>• More accessible to small traders</li>
                <li>• Better price granularity</li>
                <li>• More trading activity potential</li>
              </ul>
            </div>
            <div>
              <h4 className="font-medium text-purple-900 mb-2">💡 Share Count Guidelines</h4>
              <ul className="space-y-1">
                <li>• <strong>100-500 shares:</strong> High price per share, premium market</li>
                <li>• <strong>500-1000 shares:</strong> Balanced pricing, good accessibility</li>
                <li>• <strong>1000+ shares:</strong> Low price per share, mass market</li>
                <li>• <strong>Minimum 100:</strong> Ensures reasonable price granularity</li>
              </ul>
              <div className="mt-3 p-2 bg-purple-100 rounded text-xs">
                <strong>Tip:</strong> More shares = more traders can participate with smaller amounts
              </div>
            </div>
          </div>
        </div>

        {/* Trust & Liquidity Info */}
        <div className="mt-6 bg-green-50 border border-green-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-green-900 mb-3">Why Seeding & Liquidity Matter</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm text-green-800">
            <div>
              <h4 className="font-medium text-green-900 mb-2">🔒 Trust & Credibility</h4>
              <ul className="space-y-1">
                <li>• Shows you believe in your market</li>
                <li>• Creates initial trading liquidity</li>
                <li>• Attracts serious traders</li>
                <li>• Reduces market manipulation</li>
                <li>• Sets clear initial share pricing</li>
              </ul>
            </div>
            <div>
              <h4 className="font-medium text-green-900 mb-2">💰 Refund Protection</h4>
              <ul className="space-y-1">
                <li>• Get refunded if market fails</li>
                <li>• Based on trading threshold</li>
                <li>• Automatic after deadline</li>
                <li>• Protects your investment</li>
              </ul>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
