'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { uploadFile } from '@/lib/storage'
import { LoadingSpinner, PageLoadingSpinner, ButtonLoadingSpinner } from '@/components/ui/LoadingSpinner'
import Link from 'next/link'

interface KYCApplication {
  id: string
  user_id: string
  status: 'pending' | 'approved' | 'rejected'
  document_type: string
  document_number: string
  first_name: string
  last_name: string
  date_of_birth: string
  address: string
  phone_number: string
  document_front_url: string
  document_back_url: string
  selfie_url: string
  submitted_at: string
  reviewed_at: string | null
  reviewed_by: string | null
  rejection_reason: string | null
}

export default function KYCPage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <KYCContent user={user} />
    </FastSessionWrapper>
  )
}

function KYCContent({ user }: { user: any }) {
  const { profile, refreshProfile } = useAuth()
  const [documents, setDocuments] = useState({
    idFront: null as File | null,
    idBack: null as File | null,
    selfie: null as File | null
  })
  const [formData, setFormData] = useState({
    document_type: 'drivers_license',
    document_number: '',
    first_name: '',
    last_name: '',
    date_of_birth: '',
    address: '',
    phone_number: ''
  })
  const [uploading, setUploading] = useState(false)
  const [uploadProgress, setUploadProgress] = useState(0)
  const [successMessage, setSuccessMessage] = useState('')
  const [errorMessage, setErrorMessage] = useState('')
  const [kycApplication, setKycApplication] = useState<KYCApplication | null>(null)
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    fetchExistingKYCApplication()
  }, [user])

  const fetchExistingKYCApplication = async () => {
    if (!user) return
    
    try {
      const { data, error } = await supabase
        .from('kyc_applications')
        .select('*')
        .eq('user_id', user.id)
        .order('submitted_at', { ascending: false })
        .limit(1)
        .single()

      if (error && error.code !== 'PGRST116') { // PGRST116 is "no rows returned"
        console.error('Error fetching KYC application:', error)
      } else if (data) {
        setKycApplication(data)
        // Pre-fill form with existing data
        setFormData({
          document_type: data.document_type,
          document_number: data.document_number,
          first_name: data.first_name,
          last_name: data.last_name,
          date_of_birth: data.date_of_birth,
          address: data.address,
          phone_number: data.phone_number
        })
        // Refresh profile to sync KYC status
        await refreshProfile()
      }
    } catch (err) {
      console.error('Error fetching KYC application:', err)
    } finally {
      setLoading(false)
    }
  }

  const handleFileChange = (documentType: keyof typeof documents, file: File | null) => {
    setDocuments(prev => ({
      ...prev,
      [documentType]: file
    }))
  }

  const handleInputChange = (field: keyof typeof formData, value: string) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }))
  }


  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setUploading(true)
    setUploadProgress(0)
    setErrorMessage('')
    setSuccessMessage('')

    try {
      // Validate required fields
      if (!formData.first_name || !formData.last_name || !formData.document_number || 
          !formData.date_of_birth || !formData.address || !formData.phone_number) {
        throw new Error('Please fill in all required fields')
      }

      // Validate required documents
      if (!documents.idFront || !documents.idBack || !documents.selfie) {
        throw new Error('Please upload all required documents')
      }

      const documentTypes = Object.keys(documents) as Array<keyof typeof documents>
      const uploadedDocuments: Record<string, string> = {}
      
      // Upload each document
      for (let i = 0; i < documentTypes.length; i++) {
        const docType = documentTypes[i]
        const file = documents[docType]
        
        if (file) {
          // Upload file using the storage utility
          const result = await uploadFile({
            bucket: 'kyc-documents',
            userId: user.id,
            file: file
          })

          if (!result.success) {
            throw new Error(result.error || 'Failed to upload file')
          }

          uploadedDocuments[docType] = result.url!
        }

        // Update progress
        setUploadProgress(((i + 1) / documentTypes.length) * 100)
      }

      // Create or update KYC application
      const kycData = {
        user_id: user.id,
        status: 'pending',
        document_type: formData.document_type,
        document_number: formData.document_number,
        first_name: formData.first_name,
        last_name: formData.last_name,
        date_of_birth: formData.date_of_birth,
        address: formData.address,
        phone_number: formData.phone_number,
        document_front_url: uploadedDocuments.idFront,
        document_back_url: uploadedDocuments.idBack,
        selfie_url: uploadedDocuments.selfie,
        submitted_at: new Date().toISOString()
      }

      let result
      if (kycApplication) {
        // Update existing application
        result = await supabase
          .from('kyc_applications')
          .update(kycData)
          .eq('id', kycApplication.id)
          .select()
          .single()
      } else {
        // Create new application
        result = await supabase
          .from('kyc_applications')
          .insert(kycData)
          .select()
          .single()
      }

      if (result.error) throw result.error

      // Update user profile KYC status
      const { error: profileError } = await supabase
        .from('profiles')
        .update({
          kyc_status: 'pending',
          kyc_verified: false
        })
        .eq('id', user.id)

      if (profileError) {
        console.error('Error updating profile KYC status:', profileError)
      }

      setKycApplication(result.data)
      setSuccessMessage('KYC application submitted successfully! Your verification is now pending.')
      setErrorMessage('')
      
      // Refresh profile to sync KYC status
      await refreshProfile()

    } catch (error) {
      setErrorMessage(error instanceof Error ? error.message : 'Failed to submit KYC application')
      console.error('KYC submission error:', error)
    } finally {
      setUploading(false)
      setUploadProgress(0)
    }
  }

  if (loading) {
    return <PageLoadingSpinner text="Loading KYC information..." />
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-4xl mx-auto px-4">
        <div className="mb-8">
          <div className="flex items-center space-x-4 mb-4">
            <Link href="/settings" className="text-black hover:text-black">
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
              </svg>
            </Link>
            <h1 className="text-3xl font-bold text-black">KYC Verification</h1>
          </div>
          <p className="text-black">Complete your identity verification to access all platform features</p>
        </div>

        {/* Profile KYC Status */}
        <div className="mb-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <svg className="w-5 h-5 mr-2 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
              </svg>
              <div>
                <p className="text-sm font-medium text-blue-700">
                  Profile Status: {profile?.kyc_verified ? 'Verified' : 
                                   profile?.kyc_status === 'pending' ? 'Pending' : 
                                   'Not Verified'}
                </p>
                <p className="text-xs text-gray-500">
                  This is what shows in Settings page
                </p>
              </div>
            </div>
            <div className="flex gap-2">
              <button
                onClick={async () => {
                  await refreshProfile()
                }}
                className="text-blue-600 hover:text-blue-700 text-sm font-medium"
              >
                Refresh Profile
              </button>
              {kycApplication && kycApplication.status === 'approved' && !profile?.kyc_verified && (
                <button
                  onClick={async () => {
                    try {
                      const { error } = await supabase
                        .from('profiles')
                        .update({
                          kyc_status: 'approved',
                          kyc_verified: true,
                          kyc_verified_at: new Date().toISOString()
                        })
                        .eq('id', user.id)
                      
                      if (error) {
                        console.error('Error syncing profile:', error)
                      } else {
                        await refreshProfile()
                      }
                    } catch (err) {
                      console.error('Sync error:', err)
                    }
                  }}
                  className="text-green-600 hover:text-green-700 text-sm font-medium bg-green-50 px-2 py-1 rounded"
                >
                  Sync to Profile
                </button>
              )}
            </div>
          </div>
        </div>

        {/* KYC Application Status */}
        {kycApplication && (
          <div className="mb-6 p-4 border rounded-lg">
            <div className="flex items-center justify-between">
              <div className="flex items-center">
                <svg className={`w-5 h-5 mr-2 ${
                  kycApplication.status === 'approved' ? 'text-green-400' :
                  kycApplication.status === 'rejected' ? 'text-red-400' :
                  'text-blue-400'
                }`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <div>
                  <p className={`text-sm font-medium ${
                    kycApplication.status === 'approved' ? 'text-green-700' :
                    kycApplication.status === 'rejected' ? 'text-red-700' :
                    'text-blue-700'
                  }`}>
                    Application Status: <span className="capitalize">{kycApplication.status}</span>
                  </p>
                  <p className="text-xs text-gray-500">
                    Submitted: {new Date(kycApplication.submitted_at).toLocaleDateString()}
                  </p>
                  {kycApplication.reviewed_at && (
                    <p className="text-xs text-gray-500">
                      Reviewed: {new Date(kycApplication.reviewed_at).toLocaleDateString()}
                    </p>
                  )}
                </div>
              </div>
              <button
                onClick={async () => {
                  await refreshProfile()
                  await fetchExistingKYCApplication()
                }}
                className="text-blue-600 hover:text-blue-700 text-sm font-medium"
              >
                Refresh Status
              </button>
              {kycApplication.status === 'rejected' && kycApplication.rejection_reason && (
                <div className="text-right">
                  <p className="text-xs text-red-600 font-medium">Rejection Reason:</p>
                  <p className="text-xs text-red-600">{kycApplication.rejection_reason}</p>
                </div>
              )}
            </div>
          </div>
        )}

        {/* Success Message */}
        {successMessage && (
          <div className="mb-6 p-4 bg-green-50 border border-green-200 rounded-lg">
            <div className="flex items-center">
              <svg className="w-5 h-5 text-green-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
              </svg>
              <p className="text-green-700 text-sm">{successMessage}</p>
            </div>
          </div>
        )}

        {/* Error Message */}
        {errorMessage && (
          <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg">
            <div className="flex items-center">
              <svg className="w-5 h-5 text-red-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              <p className="text-red-700 text-sm">{errorMessage}</p>
            </div>
          </div>
        )}

        {/* Upload Progress */}
        {uploading && (
          <div className="mb-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
            <div className="flex items-center justify-between mb-2">
              <span className="text-blue-700 text-sm font-medium">Uploading documents...</span>
              <span className="text-blue-700 text-sm">{Math.round(uploadProgress)}%</span>
            </div>
            <div className="w-full bg-blue-200 rounded-full h-2">
              <div 
                className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                style={{ width: `${uploadProgress}%` }}
              ></div>
            </div>
          </div>
        )}

        {/* KYC Form */}
        <form onSubmit={handleSubmit} className="space-y-8">
          {/* Personal Information */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <h2 className="text-xl font-semibold text-black mb-6">Personal Information</h2>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
              <div>
                <label className="block text-sm font-medium text-black mb-2">
                  First Name <span className="text-red-500">*</span>
                </label>
                <input
                  type="text"
                  value={formData.first_name}
                  onChange={(e) => handleInputChange('first_name', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-black"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-black mb-2">
                  Last Name <span className="text-red-500">*</span>
                </label>
                <input
                  type="text"
                  value={formData.last_name}
                  onChange={(e) => handleInputChange('last_name', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-black"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-black mb-2">
                  Date of Birth <span className="text-red-500">*</span>
                </label>
                <input
                  type="date"
                  value={formData.date_of_birth}
                  onChange={(e) => handleInputChange('date_of_birth', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-black"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-black mb-2">
                  Phone Number <span className="text-red-500">*</span>
                </label>
                <input
                  type="tel"
                  value={formData.phone_number}
                  onChange={(e) => handleInputChange('phone_number', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-black"
                  placeholder="+1 (555) 123-4567"
                  required
                />
              </div>
            </div>
            
            <div className="mb-6">
              <label className="block text-sm font-medium text-black mb-2">
                Document Type <span className="text-red-500">*</span>
              </label>
              <select
                value={formData.document_type}
                onChange={(e) => handleInputChange('document_type', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-black"
                required
              >
                <option value="drivers_license">Driver's License</option>
                <option value="passport">Passport</option>
                <option value="national_id">National ID</option>
              </select>
            </div>
            
            <div className="mb-6">
              <label className="block text-sm font-medium text-black mb-2">
                Document Number <span className="text-red-500">*</span>
              </label>
              <input
                type="text"
                value={formData.document_number}
                onChange={(e) => handleInputChange('document_number', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-black"
                placeholder="Enter your document number"
                required
              />
            </div>
            
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Address <span className="text-red-500">*</span>
              </label>
              <textarea
                value={formData.address}
                onChange={(e) => handleInputChange('address', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-black"
                rows={3}
                placeholder="Enter your full address"
                required
              />
            </div>
          </div>

          {/* Documents */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <h2 className="text-xl font-semibold text-black mb-6">Required Documents</h2>
            
            <div className="space-y-6">
              {/* Government ID Front */}
              <div>
                <label className="block text-sm font-medium text-black mb-2">
                  Government ID (Front) <span className="text-red-500">*</span>
                </label>
                <p className="text-sm text-gray-600 mb-3">
                  Upload a clear photo of the front of your government-issued ID (Driver's License, Passport, etc.)
                </p>
                <input
                  type="file"
                  accept="image/*,.pdf"
                  onChange={(e) => handleFileChange('idFront', e.target.files?.[0] || null)}
                  className="block w-full text-sm text-black border border-gray-300 rounded-lg cursor-pointer bg-gray-50 focus:outline-none"
                  style={{ color: 'black' }}
                />
                {documents.idFront && (
                  <p className="mt-2 text-sm text-green-600">
                    ✓ {documents.idFront.name} ({(documents.idFront.size / 1024 / 1024).toFixed(2)} MB)
                  </p>
                )}
              </div>

              {/* Government ID Back */}
              <div>
                <label className="block text-sm font-medium text-black mb-2">
                  Government ID (Back) <span className="text-red-500">*</span>
                </label>
                <p className="text-sm text-gray-600 mb-3">
                  Upload a clear photo of the back of your government-issued ID
                </p>
                <input
                  type="file"
                  accept="image/*,.pdf"
                  onChange={(e) => handleFileChange('idBack', e.target.files?.[0] || null)}
                  className="block w-full text-sm text-black border border-gray-300 rounded-lg cursor-pointer bg-gray-50 focus:outline-none"
                  style={{ color: 'black' }}
                />
                {documents.idBack && (
                  <p className="mt-2 text-sm text-green-600">
                    ✓ {documents.idBack.name} ({(documents.idBack.size / 1024 / 1024).toFixed(2)} MB)
                  </p>
                )}
              </div>

              {/* Selfie */}
              <div>
                <label className="block text-sm font-medium text-black mb-2">
                  Selfie with ID <span className="text-red-500">*</span>
                </label>
                <p className="text-sm text-gray-600 mb-3">
                  Take a selfie holding your government ID next to your face
                </p>
                <input
                  type="file"
                  accept="image/*"
                  onChange={(e) => handleFileChange('selfie', e.target.files?.[0] || null)}
                  className="block w-full text-sm text-black border border-gray-300 rounded-lg cursor-pointer bg-gray-50 focus:outline-none"
                  style={{ color: 'black' }}
                />
                {documents.selfie && (
                  <p className="mt-2 text-sm text-green-600">
                    ✓ {documents.selfie.name} ({(documents.selfie.size / 1024 / 1024).toFixed(2)} MB)
                  </p>
                )}
              </div>

            </div>
          </div>

          {/* Guidelines */}
          <div className="bg-yellow-50 border border-yellow-200 rounded-xl p-6">
            <h3 className="text-lg font-semibold text-black mb-4">📋 Document Guidelines</h3>
            <ul className="space-y-2 text-sm text-black">
              <li>• All documents must be clear and readable</li>
              <li>• Maximum file size: 5MB per document</li>
              <li>• Accepted formats: JPG, PNG, PDF</li>
              <li>• Documents must be valid and not expired</li>
              <li>• Photos must be well-lit and in focus</li>
              <li>• Your face must be clearly visible in the selfie</li>
              <li>• Government ID must show both front and back clearly</li>
            </ul>
          </div>

          {/* Submit Button */}
          <div className="flex justify-center">
            <button
              type="submit"
              disabled={
                uploading || 
                !formData.first_name || !formData.last_name || !formData.document_number || 
                !formData.date_of_birth || !formData.address || !formData.phone_number ||
                !documents.idFront || !documents.idBack || !documents.selfie
              }
              className="px-8 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center"
            >
              {uploading ? (
                <ButtonLoadingSpinner text="Submitting Application..." />
              ) : kycApplication ? (
                'Update KYC Application'
              ) : (
                'Submit KYC Application'
              )}
            </button>
          </div>
        </form>
      </div>
    </div>
  )
}
