'use client'

import { useState } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { useLeaderboard } from '@/lib/hooks/useLeaderboard'
import { useUserStats } from '@/lib/hooks/useUserStats'
import Link from 'next/link'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

export default function LeaderboardPage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <LeaderboardContent user={user} />
    </FastSessionWrapper>
  )
}

function LeaderboardContent({ user }: { user: any }) {
  const [timePeriod, setTimePeriod] = useState<'all' | 'month' | 'week'>('month')
  
  const { leaderboard, loading, error } = useLeaderboard(timePeriod)
  const { userStats, platformStats, loading: statsLoading, error: statsError } = useUserStats(user?.id, timePeriod)

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading leaderboard...</p>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Error Loading Leaderboard</h1>
          <p className="text-black mb-4">{error}</p>
          <Link href="/" className="text-blue-600 hover:text-blue-700">
            Go Home
          </Link>
        </div>
      </div>
    )
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  const getRankBadge = (rank: number) => {
    if (rank === 1) return '🥇'
    if (rank === 2) return '🥈'
    if (rank === 3) return '🥉'
    return `#${rank}`
  }

  const getRankColor = (rank: number) => {
    if (rank === 1) return 'text-yellow-600 bg-yellow-50 border-yellow-200'
    if (rank === 2) return 'text-black bg-gray-50 border-gray-200'
    if (rank === 3) return 'text-orange-600 bg-orange-50 border-orange-200'
    return 'text-black bg-white border-gray-200'
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-6xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-black">Leaderboard</h1>
          <p className="text-black mt-2">
            Top performers {timePeriod === 'all' ? 'all time' : timePeriod === 'week' ? 'this week' : 'this month'}
          </p>
        </div>

        {/* Time Period Selector */}
        <div className="mb-6">
          <div className="flex space-x-1 bg-gray-100 p-1 rounded-lg w-fit">
            <button 
              onClick={() => setTimePeriod('month')}
              className={`px-4 py-2 text-sm font-medium rounded-md transition-colors ${
                timePeriod === 'month'
                  ? 'text-white bg-blue-600'
                  : 'text-black hover:bg-gray-200'
              }`}
            >
              This Month
            </button>
            <button 
              onClick={() => setTimePeriod('all')}
              className={`px-4 py-2 text-sm font-medium rounded-md transition-colors ${
                timePeriod === 'all'
                  ? 'text-white bg-blue-600'
                  : 'text-black hover:bg-gray-200'
              }`}
            >
              All Time
            </button>
            <button 
              onClick={() => setTimePeriod('week')}
              className={`px-4 py-2 text-sm font-medium rounded-md transition-colors ${
                timePeriod === 'week'
                  ? 'text-white bg-blue-600'
                  : 'text-black hover:bg-gray-200'
              }`}
            >
              This Week
            </button>
          </div>
        </div>

        {/* Current User Stats */}
        {user && (
          <div className="mb-8 bg-gradient-to-r from-blue-600 to-purple-600 rounded-xl p-6 text-white">
            <h2 className="text-lg font-semibold mb-4">Your Performance</h2>
            {statsLoading ? (
              <div className="flex justify-center py-4">
                <LoadingSpinner size="md" />
              </div>
            ) : statsError ? (
              <div className="text-center py-4">
                <p className="text-red-200">Error loading your stats</p>
              </div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div className="text-center">
                  <div className="text-2xl font-bold">#{userStats.rank}</div>
                  <div className="text-sm opacity-90">Your Rank</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold">{userStats.points.toLocaleString()}</div>
                  <div className="text-sm opacity-90">Points</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold">{userStats.winRate}%</div>
                  <div className="text-sm opacity-90">Win Rate</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold">{formatCurrency(userStats.profit)}</div>
                  <div className="text-sm opacity-90">Profit</div>
                </div>
              </div>
            )}
          </div>
        )}

        {/* Leaderboard Table */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
          <div className="px-6 py-4 border-b border-gray-200 bg-gray-50">
            <h2 className="text-lg font-semibold text-black">Top Performers</h2>
          </div>
          
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                    Rank
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                    User
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                    Points
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                    Win Rate
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                    Total Bets
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                    Profit
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {leaderboard.map((player) => (
                  <tr key={player.rank} className="hover:bg-gray-50">
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-sm font-medium border ${getRankColor(player.rank)}`}>
                        {getRankBadge(player.rank)}
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center">
                        <div className="w-8 h-8 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white text-sm font-bold mr-3">
                          {player.username.charAt(0).toUpperCase()}
                        </div>
                        <div className="font-medium text-black">{player.username}</div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-black font-medium">
                      {player.points.toLocaleString()}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center">
                        <div className="text-sm font-medium text-black">{player.win_rate}%</div>
                        <div className="ml-2 w-16 bg-gray-200 rounded-full h-2">
                          <div 
                            className="bg-green-500 h-2 rounded-full" 
                            style={{ width: `${player.win_rate}%` }}
                          ></div>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                      {player.total_bets}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-green-600">
                      {formatCurrency(player.profit)}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>

        {/* Achievement Stats */}
        <div className="mt-8 grid grid-cols-1 md:grid-cols-3 gap-6">
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="flex items-center">
              <div className="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center mr-4">
                <span className="text-2xl">🏆</span>
              </div>
              <div>
                {statsLoading ? (
                  <div className="animate-pulse">
                    <div className="h-8 bg-gray-200 rounded w-16 mb-2"></div>
                    <div className="h-4 bg-gray-200 rounded w-24"></div>
                  </div>
                ) : (
                  <>
                    <div className="text-2xl font-bold text-black">{platformStats.totalChampions}</div>
                    <div className="text-sm text-black">Total Champions</div>
                  </>
                )}
              </div>
            </div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="flex items-center">
              <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mr-4">
                <span className="text-2xl">📈</span>
              </div>
              <div>
                {statsLoading ? (
                  <div className="animate-pulse">
                    <div className="h-8 bg-gray-200 rounded w-16 mb-2"></div>
                    <div className="h-4 bg-gray-200 rounded w-24"></div>
                  </div>
                ) : (
                  <>
                    <div className="text-2xl font-bold text-black">{formatCurrency(platformStats.totalProfits)}</div>
                    <div className="text-sm text-black">Total Profits</div>
                  </>
                )}
              </div>
            </div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="flex items-center">
              <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mr-4">
                <span className="text-2xl">🎯</span>
              </div>
              <div>
                {statsLoading ? (
                  <div className="animate-pulse">
                    <div className="h-8 bg-gray-200 rounded w-16 mb-2"></div>
                    <div className="h-4 bg-gray-200 rounded w-24"></div>
                  </div>
                ) : (
                  <>
                    <div className="text-2xl font-bold text-black">{platformStats.averageWinRate}%</div>
                    <div className="text-sm text-black">Average Win Rate</div>
                  </>
                )}
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}