'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/components/auth/ClientAuthProvider'
import { supabase } from '@/lib/supabase-client'
import { logger } from '@/lib/logger'

export default function MonitorPage() {
  const { user, profile } = useAuth()
  const [systemStats, setSystemStats] = useState<any>({})
  const [logs, setLogs] = useState<any[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    const fetchSystemStats = async () => {
      try {
        // Get user count
        const { count: userCount } = await supabase
          .from('profiles')
          .select('*', { count: 'exact', head: true })

        // Get market count
        const { count: marketCount } = await supabase
          .from('markets')
          .select('*', { count: 'exact', head: true })

        // Get bet count
        const { count: betCount } = await supabase
          .from('bets')
          .select('*', { count: 'exact', head: true })

        // Get recent activity
        const { data: recentBets } = await supabase
          .from('bets')
          .select('*')
          .order('created_at', { ascending: false })
          .limit(10)

        setSystemStats({
          userCount,
          marketCount,
          betCount,
          recentBets,
          timestamp: new Date().toISOString(),
        })

        // Get logs
        setLogs(logger.getLogs())

        setLoading(false)
      } catch (error) {
        console.error('Error fetching system stats:', error)
        setLoading(false)
      }
    }

    fetchSystemStats()
  }, [])

  if (!profile?.is_admin) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Access Denied</h1>
          <p className="text-black">You don't have permission to access this page.</p>
        </div>
      </div>
    )
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-black">Loading system stats...</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 p-8">
      <div className="max-w-6xl mx-auto">
        <h1 className="text-3xl font-bold text-black mb-8">System Monitor</h1>
        
        {/* System Stats */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <div className="bg-white rounded-lg shadow p-6">
            <h3 className="text-lg font-semibold text-black mb-2">Users</h3>
            <p className="text-3xl font-bold text-blue-600">{systemStats.userCount || 0}</p>
          </div>
          <div className="bg-white rounded-lg shadow p-6">
            <h3 className="text-lg font-semibold text-black mb-2">Markets</h3>
            <p className="text-3xl font-bold text-green-600">{systemStats.marketCount || 0}</p>
          </div>
          <div className="bg-white rounded-lg shadow p-6">
            <h3 className="text-lg font-semibold text-black mb-2">Bets</h3>
            <p className="text-3xl font-bold text-purple-600">{systemStats.betCount || 0}</p>
          </div>
        </div>

        {/* Recent Activity */}
        <div className="bg-white rounded-lg shadow p-6 mb-8">
          <h2 className="text-xl font-bold text-black mb-4">Recent Activity</h2>
          <div className="space-y-4">
            {systemStats.recentBets?.map((bet: any, index: number) => (
              <div key={index} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                <div>
                  <p className="font-semibold text-black">Bet #{bet.id}</p>
                  <p className="text-sm text-gray-600">Amount: ${bet.amount}</p>
                </div>
                <div className="text-sm text-gray-500">
                  {new Date(bet.created_at).toLocaleString()}
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* System Logs */}
        <div className="bg-white rounded-lg shadow p-6">
          <h2 className="text-xl font-bold text-black mb-4">System Logs</h2>
          <div className="space-y-2 max-h-96 overflow-y-auto">
            {logs.map((log, index) => (
              <div key={index} className={`p-3 rounded-lg text-sm ${
                log.level === 'error' ? 'bg-red-50 text-red-800' :
                log.level === 'warn' ? 'bg-yellow-50 text-yellow-800' :
                log.level === 'info' ? 'bg-blue-50 text-blue-800' :
                'bg-gray-50 text-gray-800'
              }`}>
                <div className="flex items-center justify-between">
                  <span className="font-semibold">[{log.level.toUpperCase()}]</span>
                  <span className="text-xs">{new Date(log.timestamp).toLocaleString()}</span>
                </div>
                <p className="mt-1">{log.message}</p>
                {log.data && (
                  <pre className="mt-2 text-xs bg-black bg-opacity-10 p-2 rounded">
                    {JSON.stringify(log.data, null, 2)}
                  </pre>
                )}
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  )
}
