'use client'

import { useState } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { useNotifications } from '@/lib/hooks/useNotifications'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'
import { useToast } from '@/components/ui/Toast'

export default function NotificationsPage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <NotificationsContent user={user} />
    </FastSessionWrapper>
  )
}

function NotificationsContent({ user }: { user: any }) {
  const { 
    notifications, 
    unreadCount, 
    loading, 
    error, 
    markAsRead, 
    markAllAsRead 
  } = useNotifications(user?.id)
  const { addToast } = useToast()

  const getNotificationIcon = (type: string) => {
    switch (type) {
      case 'success':
        return '✅'
      case 'error':
        return '❌'
      case 'warning':
        return '⚠️'
      case 'info':
      default:
        return '📢'
    }
  }

  const getNotificationColor = (type: string) => {
    switch (type) {
      case 'success':
        return 'bg-green-100 text-green-800 border-green-200'
      case 'error':
        return 'bg-red-100 text-red-800 border-red-200'
      case 'warning':
        return 'bg-yellow-100 text-yellow-800 border-yellow-200'
      case 'info':
      default:
        return 'bg-blue-100 text-blue-800 border-blue-200'
    }
  }

  const handleMarkAllAsRead = async () => {
    try {
      await markAllAsRead()
      addToast({
        type: 'success',
        title: 'All notifications marked as read',
        message: 'You\'ve marked all notifications as read'
      })
    } catch (error) {
      addToast({
        type: 'error',
        title: 'Error',
        message: 'Failed to mark all notifications as read'
      })
    }
  }

  const handleMarkAsRead = async (notificationId: string) => {
    try {
      await markAsRead(notificationId)
    } catch (error) {
      addToast({
        type: 'error',
        title: 'Error',
        message: 'Failed to mark notification as read'
      })
    }
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading notifications...</p>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Error Loading Notifications</h1>
          <p className="text-black mb-4">{error}</p>
          <button 
            onClick={() => window.location.reload()} 
            className="text-blue-600 hover:text-blue-700"
          >
            Try Again
          </button>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-8">
        {/* Page Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-black">Notifications</h1>
              <p className="text-gray-600 mt-2">
                Stay updated with your betting activity and platform news
              </p>
            </div>
            <div className="flex items-center space-x-4">
              {unreadCount > 0 && (
                <div className="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-sm font-medium">
                  {unreadCount} unread
                </div>
              )}
              {notifications.length > 0 && (
                <button
                  onClick={handleMarkAllAsRead}
                  className="text-blue-600 hover:text-blue-700 text-sm font-medium"
                >
                  Mark all as read
                </button>
              )}
            </div>
          </div>
        </div>

        {/* Live Data Indicator */}
        <div className="flex items-center space-x-2 mb-6">
          <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
          <span className="text-sm text-gray-600">Live notifications - Real-time updates</span>
        </div>

        {/* Notifications List */}
        <div className="space-y-4">
          {notifications.length > 0 ? (
            notifications.map((notification) => (
              <div
                key={notification.id}
                className={`bg-white rounded-lg border-l-4 shadow-sm hover:shadow-md transition-shadow cursor-pointer ${
                  !notification.is_read ? getNotificationColor(notification.type) : 'border-gray-200'
                }`}
                onClick={() => handleMarkAsRead(notification.id)}
              >
                <div className="p-6">
                  <div className="flex items-start space-x-4">
                    <div className="flex-shrink-0">
                      <div className={`w-10 h-10 rounded-full flex items-center justify-center text-lg ${
                        notification.type === 'success' ? 'bg-green-100' :
                        notification.type === 'error' ? 'bg-red-100' :
                        notification.type === 'warning' ? 'bg-yellow-100' :
                        'bg-blue-100'
                      }`}>
                        {getNotificationIcon(notification.type)}
                      </div>
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center justify-between">
                        <h3 className={`text-sm font-medium ${
                          !notification.is_read ? 'text-black' : 'text-gray-900'
                        }`}>
                          {notification.title}
                        </h3>
                        <div className="flex items-center space-x-2">
                          {!notification.is_read && (
                            <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
                          )}
                          <span className="text-xs text-gray-500">
                            {new Date(notification.created_at).toLocaleDateString()} at{' '}
                            {new Date(notification.created_at).toLocaleTimeString()}
                          </span>
                        </div>
                      </div>
                      <p className={`text-sm mt-1 ${
                        !notification.is_read ? 'text-gray-700' : 'text-gray-600'
                      }`}>
                        {notification.message}
                      </p>
                      {notification.data && (
                        <div className="mt-2 text-xs text-gray-500">
                          {notification.data.marketTitle && (
                            <span>Market: {notification.data.marketTitle}</span>
                          )}
                          {notification.data.amount && (
                            <span className="ml-2">Amount: ${notification.data.amount.toFixed(2)}</span>
                          )}
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              </div>
            ))
          ) : (
            <div className="text-center py-12">
              <div className="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <svg className="w-12 h-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 17h5l-5 5-5-5h5v-12a7.5 7.5 0 00-15 0v12a7.5 7.5 0 0015 0v-12z" />
                </svg>
              </div>
              <h3 className="text-lg font-medium text-black mb-2">No Notifications Yet</h3>
              <p className="text-gray-600 mb-6">
                You'll receive notifications for bet results, new markets, deposits, and more.
              </p>
              <div className="flex items-center justify-center space-x-4 text-sm text-gray-500">
                <span>🎯 Bet results</span>
                <span>🆕 New markets</span>
                <span>💰 Deposits & withdrawals</span>
                <span>🏆 Achievements</span>
              </div>
            </div>
          )}
        </div>

        {/* Load More Button */}
        {notifications.length >= 50 && (
          <div className="text-center mt-8">
            <button className="bg-gray-100 hover:bg-gray-200 text-black px-6 py-3 rounded-lg font-medium transition-colors">
              Load more notifications
            </button>
          </div>
        )}
      </div>
    </div>
  )
}