'use client'

import { useState, useEffect } from 'react'
import { useHomeMarkets, useLiveMarketUpdates } from '@/lib/hooks/useHomeMarkets'
import { useWatchlist } from '@/lib/hooks/useWatchlist'
import { useAuth } from '@/components/auth/FastAuthProvider'
import { Market } from '@/lib/types'
import UserRedirect from '@/components/UserRedirect'
// Debug components removed
import { supabase } from '@/lib/supabase-client'
import Link from 'next/link'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'
import LoadingLink from '@/components/ui/LoadingLink'
import MultiOutcomeDisplay from '@/components/markets/MultiOutcomeDisplay'
import QuickBetModal from '@/components/markets/QuickBetModal'

export default function HomePage() {
  const { markets, loading, error } = useHomeMarkets(20)
  const { user } = useAuth()
  const { bookmarkedMarkets, toggleBookmark } = useWatchlist(user?.id)
  const [selectedCategory, setSelectedCategory] = useState('Trending')
  const [selectedTag, setSelectedTag] = useState('All')
  const [searchQuery, setSearchQuery] = useState('')
  const [searchField, setSearchField] = useState<'all' | 'title' | 'description' | 'category' | 'status'>('all')
  const [showSuggestions, setShowSuggestions] = useState(false)
  const [activeSuggestionIndex, setActiveSuggestionIndex] = useState<number>(-1)
  const [filteredMarkets, setFilteredMarkets] = useState(markets)
  const [selectedMarket, setSelectedMarket] = useState<Market | null>(null)
  const [showBetModal, setShowBetModal] = useState(false)
  
  // Real-time updates for market data
  const marketIds = markets.map(market => market.id)
  const liveUpdates = useLiveMarketUpdates(marketIds)

  // Build suggestions based on current query
  const suggestions = searchQuery.trim()
    ? markets
        .filter((m) => {
          const q = searchQuery.toLowerCase()
          if (searchField === 'title') return m.title.toLowerCase().includes(q)
          if (searchField === 'description') return (m.description || '').toLowerCase().includes(q)
          if (searchField === 'category') return (m.category || '').toLowerCase().includes(q)
          if (searchField === 'status') return (m.status || '').toLowerCase().includes(q)
          return (
            m.title.toLowerCase().includes(q) ||
            (m.description || '').toLowerCase().includes(q) ||
            (m.category || '').toLowerCase().includes(q)
          )
        })
        .slice(0, 8)
    : []

  // Filter markets based on search, field, category, and bookmarks
  useEffect(() => {
    let filtered = markets

    // Filter by search query
    if (searchQuery.trim()) {
      const q = searchQuery.toLowerCase()
      filtered = filtered.filter((market) => {
        if (searchField === 'title') return market.title.toLowerCase().includes(q)
        if (searchField === 'description') return (market.description || '').toLowerCase().includes(q)
        if (searchField === 'category') return (market.category || '').toLowerCase().includes(q)
        if (searchField === 'status') return (market.status || '').toLowerCase().includes(q)
        return (
          market.title.toLowerCase().includes(q) ||
          (market.description || '').toLowerCase().includes(q) ||
          (market.category || '').toLowerCase().includes(q)
        )
      })
    }

    // Filter by category
    if (selectedCategory !== 'Trending' && selectedCategory !== 'filter' && selectedCategory !== 'bookmark') {
      filtered = filtered.filter(market => market.category === selectedCategory.toLowerCase())
    }

    // Filter by bookmarks
    if (selectedCategory === 'bookmark') {
      filtered = filtered.filter(market => bookmarkedMarkets.includes(market.id))
    }

    // Filter by tag
    if (selectedTag !== 'All') {
      filtered = filtered.filter(market => 
        market.title.toLowerCase().includes(selectedTag.toLowerCase()) ||
        market.description.toLowerCase().includes(selectedTag.toLowerCase())
      )
    }

    setFilteredMarkets(filtered)
  }, [markets, searchQuery, selectedCategory, selectedTag, bookmarkedMarkets])

  // Toggle bookmark wrapper
  const handleToggleBookmark = async (marketId: string, e: React.MouseEvent) => {
    e.preventDefault()
    e.stopPropagation()
    
    if (!user) return
    
    await toggleBookmark(marketId)
  }

  // Use filtered markets
  const displayMarkets = filteredMarkets

  // Show loading state
  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 py-8">
          <div className="text-center">
            <LoadingSpinner size="lg" />
            <p className="mt-4 text-black">Loading markets...</p>
          </div>
        </div>
      </div>
    )
  }

  // Show error state
  if (error) {
    return (
      <div className="min-h-screen bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 py-8">
          <div className="text-center">
            <h1 className="text-2xl font-bold text-black mb-4">Error Loading Markets</h1>
            <p className="text-black mb-4">{error}</p>
            <button 
              onClick={() => window.location.reload()} 
              className="text-blue-600 hover:text-blue-700"
            >
              Try Again
            </button>
          </div>
        </div>
      </div>
    )
  }

  const formatVolume = (volume: number) => {
    if (volume >= 1000000) {
      return `$${(volume / 1000000).toFixed(1)}M`
    } else if (volume >= 1000) {
      return `$${(volume / 1000).toFixed(0)}K`
    }
    return `$${volume}`
  }

  const formatPrice = (price: number) => {
    // Ensure price is between 0 and 1, then convert to percentage
    const normalizedPrice = Math.max(0, Math.min(1, price || 0.5))
    return `${(normalizedPrice * 100).toFixed(0)}%`
  }

  const categories = [
    'Trending', 'Breaking', 'New', 'Politics', 'Sports', 'Crypto', 
    'Earnings', 'Geopolitics', 'Tech', 'Culture', 'World', 'Economy'
  ]

  const tags = [
    'All', 'Gaza', 'Gov Shutdown', 'NYC Mayor', 'Venezuela', 'China', 
    'MLB Playoffs', 'Israel', 'Earnings', 'Fed', 'France'
  ]

  return (
    <div className="min-h-screen bg-white">
      <UserRedirect />
      {/* Category Navigation */}
      <div className="bg-white border-b border-gray-200 sticky top-16 z-40">
        <div className="max-w-7xl mx-auto px-2 sm:px-4 lg:px-8">
          {/* Mobile: Scrollable horizontal tabs */}
          <div className="flex items-center space-x-2 sm:space-x-4 overflow-x-auto py-3 sm:py-4 scrollbar-hide">
            {categories.map((category) => (
              <button
                key={category}
                onClick={() => setSelectedCategory(category)}
                className={`flex-shrink-0 whitespace-nowrap px-2 sm:px-3 py-2 text-xs sm:text-sm font-medium rounded-lg transition-colors ${
                  selectedCategory === category
                    ? 'bg-blue-100 text-blue-700'
                    : 'text-black hover:text-black hover:bg-gray-50'
                }`}
              >
                {category}
              </button>
            ))}
            <button className="flex-shrink-0 flex items-center text-black hover:text-black px-2 sm:px-3 py-2 text-xs sm:text-sm font-medium">
              More
              <svg className="w-3 h-3 sm:w-4 sm:h-4 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
              </svg>
            </button>
          </div>
          
          {/* Mobile: Add scroll indicator */}
          <div className="sm:hidden flex justify-center">
            <div className="w-8 h-1 bg-gray-300 rounded-full mt-1"></div>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Secondary Navigation */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-8">
          <div className="flex items-center space-x-2 sm:space-x-4">
            <div className="relative">
              <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                <svg className="h-5 w-5 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                </svg>
              </div>
              <input
                type="text"
                placeholder="Search markets..."
                value={searchQuery}
                onChange={(e) => {
                  setSearchQuery(e.target.value)
                  setShowSuggestions(true)
                  setActiveSuggestionIndex(-1)
                }}
                onFocus={() => setShowSuggestions(true)}
                onBlur={() => setTimeout(() => setShowSuggestions(false), 120)}
                onKeyDown={(e) => {
                  if (!showSuggestions || suggestions.length === 0) return
                  if (e.key === 'ArrowDown') {
                    e.preventDefault()
                    setActiveSuggestionIndex((prev) => Math.min(prev + 1, suggestions.length - 1))
                  } else if (e.key === 'ArrowUp') {
                    e.preventDefault()
                    setActiveSuggestionIndex((prev) => Math.max(prev - 1, 0))
                  } else if (e.key === 'Enter' && activeSuggestionIndex >= 0) {
                    e.preventDefault()
                    const sel = suggestions[activeSuggestionIndex]
                    setSearchQuery(sel.title)
                    setShowSuggestions(false)
                  }
                }}
                className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-gray-50 text-sm"
                style={{ color: 'black' }}
              />
              {/* Suggestions dropdown */}
              {showSuggestions && suggestions.length > 0 && (
                <div className="absolute z-50 mt-1 w-full bg-white border border-gray-200 rounded-md shadow-lg max-h-72 overflow-auto">
                  {suggestions.map((s, idx) => (
                    <button
                      key={s.id}
                      type="button"
                      onMouseDown={(e) => {
                        e.preventDefault()
                        setSearchQuery(s.title)
                        setShowSuggestions(false)
                      }}
                      className={`w-full text-left px-3 py-2 text-sm ${idx === activeSuggestionIndex ? 'bg-blue-50' : 'bg-white'} text-black hover:bg-blue-50`}
                    >
                      <div className="flex items-center justify-between">
                        <span className="truncate pr-2">{s.title}</span>
                        <span className="ml-2 text-xs text-gray-600 capitalize">{s.category}</span>
                      </div>
                    </button>
                  ))}
                </div>
              )}
            </div>
            {/* Search field filter */}
            <div>
              <select
                value={searchField}
                onChange={(e) => setSearchField(e.target.value as any)}
                className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm text-black bg-white"
                title="Filter field"
              >
                <option value="all">All</option>
                <option value="title">Title</option>
                <option value="description">Description</option>
                <option value="category">Category</option>
                <option value="status">Status</option>
              </select>
            </div>
            <button 
              className={`p-2 rounded-lg transition-colors ${
                selectedCategory === 'filter' 
                  ? 'bg-blue-100 text-blue-600' 
                  : 'text-black hover:text-black hover:bg-gray-50'
              }`}
              onClick={() => setSelectedCategory(selectedCategory === 'filter' ? 'Trending' : 'filter')}
            >
              <svg className="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 10h16M4 14h16M4 18h16" />
              </svg>
            </button>
            <button 
              className={`p-2 rounded-lg transition-colors ${
                selectedCategory === 'bookmark' 
                  ? 'bg-blue-100 text-blue-600' 
                  : 'text-black hover:text-black hover:bg-gray-50'
              }`}
              onClick={() => setSelectedCategory(selectedCategory === 'bookmark' ? 'Trending' : 'bookmark')}
            >
              <svg className="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z" />
              </svg>
            </button>
          </div>
          
          {/* Tag Filters - Moved next to bookmark */}
          <div className="flex items-center space-x-3 overflow-x-auto">
            {tags.map((tag) => (
              <button
                key={tag}
                onClick={() => setSelectedTag(tag)}
                className={`whitespace-nowrap px-4 py-2 text-sm font-medium rounded-full border transition-colors ${
                  selectedTag === tag
                    ? 'bg-blue-600 text-white border-blue-600'
                    : 'bg-white text-black border-gray-300 hover:border-gray-400'
                }`}
              >
                {tag}
              </button>
            ))}
            <button className="flex items-center text-black hover:text-black">
              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
              </svg>
            </button>
          </div>
        </div>

        {/* Live Data Indicator */}
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center space-x-2">
            <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
            <span className="text-sm text-black">Live & Scheduled Markets - Real-time updates every 5 seconds</span>
            {Object.keys(liveUpdates).length > 0 && (
              <span className="text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full">
                {Object.keys(liveUpdates).length} updated
              </span>
            )}
          </div>
          <div className="text-sm text-black">
            {displayMarkets.filter(m => m.status === 'active').length} live • {displayMarkets.filter(m => m.status === 'pending').length} scheduled
            {displayMarkets.filter(m => {
              const endDate = new Date(m.end_date)
              const now = new Date()
              return now > endDate
            }).length > 0 && (
              <span className="text-red-600 ml-2">
                ⚠️ {displayMarkets.filter(m => {
                  const endDate = new Date(m.end_date)
                  const now = new Date()
                  return now > endDate
                }).length} ended markets detected
              </span>
            )}
          </div>
        </div>

        {/* Markets Grid */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
          {displayMarkets.map((market) => (
            <div key={market.id} className="block group h-full">
              <div className="bg-white rounded-xl border border-gray-200 hover:shadow-xl hover:border-blue-200 transition-all duration-300 p-6 cursor-pointer transform hover:-translate-y-1 h-full flex flex-col">
                {/* Market Category Badge */}
                <div className="flex items-center justify-between mb-3">
                  <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800 capitalize">
                    {market.category}
                  </span>
                  <div className="flex items-center space-x-1">
                    {(() => {
                      const endDate = new Date(market.end_date)
                      const now = new Date()
                      const isEnded = now > endDate
                      
                      if (isEnded) {
                        return (
                          <>
                            <div className="w-2 h-2 rounded-full bg-red-500"></div>
                            <span className="text-xs text-red-600">Ended</span>
                          </>
                        )
                      } else if (market.status === 'active') {
                        return (
                          <>
                            <div className="w-2 h-2 rounded-full bg-green-500 animate-pulse"></div>
                            <span className="text-xs text-black">Live</span>
                          </>
                        )
                      } else {
                        return (
                          <>
                            <div className="w-2 h-2 rounded-full bg-yellow-500"></div>
                            <span className="text-xs text-black">Scheduled</span>
                          </>
                        )
                      }
                    })()}
                  </div>
                </div>

                {/* Market Image */}
                <div className="w-full h-32 rounded-lg mb-4 overflow-hidden">
                  {market.image_url ? (
                    <img 
                      src={market.image_url} 
                      alt={market.title}
                      className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                      onError={(e) => {
                        // Fallback to gradient background if image fails to load
                        e.currentTarget.style.display = 'none'
                        const nextElement = e.currentTarget.nextElementSibling as HTMLElement
                        if (nextElement) {
                          nextElement.style.display = 'flex'
                        }
                      }}
                    />
                  ) : null}
                  <div className={`w-full h-full bg-gradient-to-br from-blue-50 to-indigo-100 group-hover:from-blue-100 group-hover:to-indigo-200 transition-all duration-300 flex items-center justify-center ${market.image_url ? 'hidden' : 'flex'}`}>
                    <div className="text-4xl group-hover:scale-110 transition-transform duration-300">
                      {market.category === 'politics' && '🏛️'}
                      {market.category === 'economics' && '📈'}
                      {market.category === 'technology' && '💻'}
                      {market.category === 'sports' && '⚽'}
                      {market.category === 'crypto' && '₿'}
                      {market.category === 'entertainment' && '🎬'}
                      {market.category === 'weather' && '🌤️'}
                      {market.category === 'science' && '🔬'}
                      {!['politics', 'economics', 'technology', 'sports', 'crypto', 'entertainment', 'weather', 'science'].includes(market.category) && '📊'}
                    </div>
                  </div>
                </div>

                {/* Market Title */}
                <h3 className="text-sm font-semibold text-black mb-3 leading-tight line-clamp-2">
                  {market.title}
                </h3>

                {/* Outcome Options */}
                <div className="mb-4 flex-grow">
                  {market.market_type === 'multi_outcome' && market.outcomes ? (
                    // Enhanced multi-outcome market display
                    <MultiOutcomeDisplay 
                      outcomes={market.outcomes} 
                      maxVisible={3}
                      compact={true}
                    />
                  ) : (
                    // Binary market
                    <div className="space-y-2">
                      <div className="flex items-center justify-between">
                        <span className="text-xs text-black">Yes</span>
                        <span className="text-sm font-bold text-green-600">
                          {formatPrice(market.yes_price || 0)}
                        </span>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="text-xs text-black">No</span>
                        <span className="text-sm font-bold text-red-600">
                          {formatPrice(market.no_price || 0)}
                        </span>
                      </div>
                    </div>
                  )}
                </div>

                {/* Action Buttons */}
                <div className="grid grid-cols-2 gap-3 mb-4">
                  <button 
                    onClick={(e) => {
                      e.preventDefault()
                      setSelectedMarket(market)
                      setShowBetModal(true)
                    }}
                    className="bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white py-2.5 px-4 rounded-lg text-sm font-semibold transition-all duration-200 transform hover:scale-105 shadow-lg hover:shadow-xl"
                  >
                    Place Bet
                  </button>
                  <Link 
                    href={`/markets/${market.id}`}
                    className="bg-gradient-to-r from-gray-600 to-gray-700 hover:from-gray-700 hover:to-gray-800 text-white py-2.5 px-4 rounded-lg text-sm font-semibold transition-all duration-200 transform hover:scale-105 shadow-lg hover:shadow-xl text-center"
                  >
                    View Details
                  </Link>
                </div>

                {/* Volume and Actions */}
                <div className="flex items-center justify-between text-xs text-black">
                  <span>Vol. {formatVolume(market.total_volume)}</span>
                  <div className="flex items-center space-x-2">
                    <button 
                      className="text-black hover:text-black"
                      title="Share market"
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8.684 13.342C8.886 12.938 9 12.482 9 12c0-.482-.114-.938-.316-1.342m0 2.684a3 3 0 110-2.684m0 2.684l6.632 3.316m-6.632-6l6.632-3.316m0 0a3 3 0 105.367-2.684 3 3 0 00-5.367 2.684zm0 9.316a3 3 0 105.367 2.684 3 3 0 00-5.367-2.684z" />
                      </svg>
                    </button>
                    <button 
                      onClick={(e) => handleToggleBookmark(market.id, e)}
                      className={`transition-colors ${
                        bookmarkedMarkets.includes(market.id)
                          ? 'text-blue-600 hover:text-blue-700'
                          : 'text-black hover:text-black'
                      }`}
                      title={bookmarkedMarkets.includes(market.id) ? 'Remove from watchlist' : 'Add to watchlist'}
                    >
                      <svg 
                        className="w-4 h-4" 
                        fill={bookmarkedMarkets.includes(market.id) ? 'currentColor' : 'none'} 
                        stroke="currentColor" 
                        viewBox="0 0 24 24"
                      >
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z" />
                      </svg>
                    </button>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>

        {/* Load More */}
        {displayMarkets.length > 0 && (
          <div className="text-center mt-12">
            <button className="bg-gray-100 hover:bg-gray-200 text-black px-6 py-3 rounded-lg font-medium transition-colors">
              Load more markets
            </button>
          </div>
        )}

        {/* Empty State */}
        {displayMarkets.length === 0 && (
          <div className="text-center py-12">
            <div className="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg className="w-12 h-12 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
              </svg>
            </div>
            <h3 className="text-lg font-medium text-black mb-2">No Live Markets Found</h3>
            <p className="text-black mb-6">Only showing live and scheduled markets. Try adjusting your search or filters.</p>
            <LoadingLink 
              href="/create-market"
              className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
            >
              <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
              </svg>
              Create First Market
            </LoadingLink>
          </div>
        )}
      </main>

      {/* Quick Bet Modal */}
      {selectedMarket && (
        <QuickBetModal
          market={selectedMarket}
          isOpen={showBetModal}
          onClose={() => {
            setShowBetModal(false)
            setSelectedMarket(null)
          }}
          onBetPlaced={() => {
            // Refresh markets data
            window.location.reload()
          }}
        />
      )}
    </div>
  )
}