'use client'

import { useState } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { usePortfolioOptimized } from '@/lib/hooks/usePortfolioOptimized'
import Link from 'next/link'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'
import TransactionItem from '@/components/ui/TransactionItem'

export default function PortfolioPage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <PortfolioContent user={user} />
    </FastSessionWrapper>
  )
}

function PortfolioContent({ user }: { user: any }) {
  const { portfolio, loading, error } = usePortfolioOptimized(user?.id)
  const [lastUpdated, setLastUpdated] = useState(new Date())

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading portfolio...</p>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Error Loading Portfolio</h1>
          <p className="text-black mb-4">{error}</p>
          <Link href="/" className="text-blue-600 hover:text-blue-700">
            Go Home
          </Link>
        </div>
      </div>
    )
  }

  if (!portfolio.profile) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Portfolio Not Found</h1>
          <p className="text-black">Unable to load your portfolio information.</p>
        </div>
      </div>
    )
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-6xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-black">Portfolio</h1>
              <p className="text-black mt-2">Manage your investments and track your performance</p>
            </div>
            <div className="flex items-center space-x-4">
              <div className="flex items-center space-x-2 text-sm text-gray-600">
                <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                <span>Live Data</span>
              </div>
              <div className="text-sm text-gray-500">
                Last updated: {lastUpdated.toLocaleTimeString()}
              </div>
            </div>
          </div>
        </div>

        {/* Stats Overview */}
        <div className="grid grid-cols-1 md:grid-cols-6 gap-6 mb-8">
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Total Balance</div>
            <div className="text-3xl font-bold text-black">
              {formatCurrency(portfolio.profile.balance)}
            </div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Total P&L</div>
            <div className={`text-3xl font-bold ${portfolio.totalProfit >= 0 ? 'text-green-600' : 'text-red-600'}`}>
              {formatCurrency(portfolio.totalProfit)}
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">YES Potential</div>
            <div className="text-3xl font-bold text-green-600">
              {formatCurrency(portfolio.yesPotentialWinnings)}
            </div>
            <div className="text-xs text-gray-600 mt-1">
              If YES bets win
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">NO Potential</div>
            <div className="text-3xl font-bold text-red-600">
              {formatCurrency(portfolio.noPotentialWinnings)}
            </div>
            <div className="text-xs text-gray-600 mt-1">
              If NO bets win
            </div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Active Bets</div>
            <div className="text-3xl font-bold text-blue-600">
              {portfolio.activeBets}
            </div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Win Rate</div>
            <div className="text-3xl font-bold text-purple-600">
              {portfolio.winRate}%
            </div>
          </div>
        </div>

        {/* Pool Analysis Section */}
        {portfolio.activeBets > 0 && (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-8">
            <h2 className="text-xl font-semibold text-black mb-4">Active Bet Analysis</h2>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
              <div className="text-center p-4 bg-green-50 rounded-lg">
                <div className="text-sm text-gray-600 mb-1">YES Scenario</div>
                <div className="text-2xl font-bold text-green-600">
                  {formatCurrency(portfolio.yesPotentialWinnings)}
                </div>
                <div className="text-xs text-gray-600 mt-1">
                  If all YES bets win
                </div>
              </div>
              <div className="text-center p-4 bg-red-50 rounded-lg">
                <div className="text-sm text-gray-600 mb-1">NO Scenario</div>
                <div className="text-2xl font-bold text-red-600">
                  {formatCurrency(portfolio.noPotentialWinnings)}
                </div>
                <div className="text-xs text-gray-600 mt-1">
                  If all NO bets win
                </div>
              </div>
              <div className="text-center p-4 bg-gray-50 rounded-lg">
                <div className="text-sm text-gray-600 mb-1">Worst Case</div>
                <div className="text-2xl font-bold text-red-600">
                  -{formatCurrency(portfolio.bets
                    .filter(bet => bet.status === 'seeded' || bet.status === 'live' || bet.status === 'matched')
                    .reduce((sum, bet) => sum + (bet.amount || 0), 0))}
                </div>
                <div className="text-xs text-gray-600 mt-1">
                  If all active bets lose
                </div>
              </div>
              <div className="text-center p-4 bg-blue-50 rounded-lg">
                <div className="text-sm text-gray-600 mb-1">Total Exposure</div>
                <div className="text-2xl font-bold text-blue-600">
                  {formatCurrency(portfolio.bets
                    .filter(bet => bet.status === 'seeded' || bet.status === 'live' || bet.status === 'matched')
                    .reduce((sum, bet) => sum + (bet.amount || 0), 0))}
                </div>
                <div className="text-xs text-gray-600 mt-1">
                  Total amount at risk
                </div>
              </div>
            </div>
          </div>
        )}

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* Recent Transactions */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-xl font-semibold text-black">Recent Transactions</h2>
            </div>
            <div className="p-6">
              {portfolio.transactions.length > 0 ? (
                <div className="space-y-4">
                  {portfolio.transactions.map((transaction) => (
                    <TransactionItem 
                      key={transaction.id} 
                      transaction={transaction}
                      showMarketLink={true}
                    />
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <div className="text-black mb-2">No transactions yet</div>
                  <Link href="/wallet" className="text-blue-600 hover:text-blue-700 text-sm">
                    Make your first deposit
                  </Link>
                </div>
              )}
            </div>
          </div>

          {/* Recent Bets */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-xl font-semibold text-black">Recent Bets</h2>
            </div>
            <div className="p-6">
              {portfolio.bets.length > 0 ? (
                <div className="space-y-4">
                  {portfolio.bets.slice(0, 5).map((bet) => {
                    const market = (bet as any).markets
                    const isActive = bet.status === 'seeded' || bet.status === 'live' || bet.status === 'matched'
                    
                    return (
                      <div key={bet.id} className="py-3 border-b border-gray-100 last:border-b-0">
                        <div className="flex items-center justify-between">
                          <div className="flex-1">
                            <div className="flex items-center gap-2 mb-1">
                              <div className="font-medium text-black">
                                {bet.outcome.toUpperCase()} - {formatCurrency(bet.amount || 0)}
                              </div>
                              <div className={`px-2 py-1 rounded-full text-xs font-medium ${
                                bet.status === 'won' ? 'bg-green-100 text-green-800' :
                                bet.status === 'lost' ? 'bg-red-100 text-red-800' :
                                'bg-blue-100 text-blue-800'
                              }`}>
                                {bet.status}
                              </div>
                            </div>
                            <div className="text-sm text-black mb-1">
                              {(bet as any).markets?.title || 'Market not found'}
                            </div>
                            <div className="text-xs text-gray-600">
                              {bet.seeded_amount ? `Seeded: ${formatCurrency(bet.seeded_amount)}` : ''}
                              {bet.live_amount ? `Live: ${formatCurrency(bet.live_amount)}` : ''}
                            </div>
                          </div>
                          
                          {/* Potential Winnings - Right Side */}
                          <div className="text-right ml-4">
                            <div className="text-xs text-gray-600 mb-1">Potential Win</div>
                            <div className="text-lg font-bold text-green-600">
                              {formatCurrency((bet as any).potentialWinnings || 0)}
                            </div>
                            {market && (
                              <div className="text-xs text-gray-600 mt-1">
                                {bet.outcome === 'yes' ? `${((market.yes_price || 0) * 100).toFixed(1)}%` : `${((market.no_price || 0) * 100).toFixed(1)}%`}
                                {(bet as any).poolShare && ` (${((bet as any).poolShare * 100).toFixed(1)}% pool)`}
                              </div>
                            )}
                          </div>
                        </div>
                      </div>
                    )
                  })}
                </div>
              ) : (
                <div className="text-center py-8">
                  <div className="text-black mb-2">No bets placed yet</div>
                  <Link href="/" className="text-blue-600 hover:text-blue-700 text-sm">
                    Browse markets
                  </Link>
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Quick Actions */}
        <div className="mt-8 bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <h2 className="text-xl font-semibold text-black mb-4">Quick Actions</h2>
          <div className="flex flex-wrap gap-4">
            <Link 
              href="/wallet" 
              className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors"
            >
              Manage Wallet
            </Link>
            <Link 
              href="/" 
              className="bg-gray-600 text-white px-6 py-2 rounded-lg hover:bg-gray-700 transition-colors"
            >
              Browse Markets
            </Link>
            <Link 
              href="/profile" 
              className="bg-gray-600 text-white px-6 py-2 rounded-lg hover:bg-gray-700 transition-colors"
            >
              View Profile
            </Link>
          </div>
        </div>
      </div>
    </div>
  )
}