'use client'

import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { useProfile } from '@/lib/hooks/useProfile'
import { usePortfolio } from '@/lib/hooks/usePortfolio'
import Link from 'next/link'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

export default function ProfilePage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <ProfileContent user={user} />
    </FastSessionWrapper>
  )
}

function ProfileContent({ user }: { user: any }) {
  const { profile, loading: profileLoading, error: profileError } = useProfile(user?.id)
  const { portfolio, loading: portfolioLoading, error: portfolioError } = usePortfolio(user?.id)

  if (profileLoading || portfolioLoading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading profile...</p>
        </div>
      </div>
    )
  }

  if (profileError) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Error Loading Profile</h1>
          <p className="text-black mb-4">{profileError}</p>
          <Link href="/" className="text-blue-600 hover:text-blue-700">
            Go Home
          </Link>
        </div>
      </div>
    )
  }

  if (!profile) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Profile Not Found</h1>
          <p className="text-black">Unable to load your profile information.</p>
        </div>
      </div>
    )
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-black">Profile</h1>
          <p className="text-black mt-2">Manage your account information and preferences</p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Profile Card */}
          <div className="lg:col-span-1">
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
              <div className="text-center">
                <div className="w-20 h-20 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white text-2xl font-bold mx-auto mb-4">
                  {profile.username?.charAt(0).toUpperCase() || user.email?.charAt(0).toUpperCase()}
                </div>
                <h2 className="text-xl font-semibold text-black">{profile.username || 'User'}</h2>
                <p className="text-black">{user.email}</p>
                <div className="mt-4">
                  <Link 
                    href="/settings" 
                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-black bg-white hover:bg-gray-50"
                  >
                    Edit Profile
                  </Link>
                </div>
              </div>
            </div>
          </div>

          {/* Profile Details */}
          <div className="lg:col-span-2 space-y-6">
            {/* Account Stats */}
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-black mb-4">Account Overview</h3>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div className="text-center p-4 bg-gray-50 rounded-lg">
                  <div className="text-2xl font-bold text-green-600">
                    {formatCurrency(profile.balance)}
                  </div>
                  <div className="text-sm text-black">Account Balance</div>
                </div>
                <div className="text-center p-4 bg-gray-50 rounded-lg">
                  <div className="text-2xl font-bold text-blue-600">
                    {profile.total_points || 0}
                  </div>
                  <div className="text-sm text-black">Total Points</div>
                </div>
                <div className="text-center p-4 bg-gray-50 rounded-lg">
                  <div className="text-2xl font-bold">
                    {profile.kyc_verified ? (
                      <span className="text-blue-600">✓</span>
                    ) : profile.kyc_status === 'pending' ? (
                      <span className="text-yellow-600">⏳</span>
                    ) : (
                      <span className="text-red-600">✗</span>
                    )}
                  </div>
                  <div className="text-sm text-black">
                    {profile.kyc_verified ? 'Verified' : 
                     profile.kyc_status === 'pending' ? 'Pending' : 
                     'Not Verified'}
                  </div>
                </div>
              </div>
            </div>

            {/* Trading Stats */}
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-black mb-4">Trading Statistics</h3>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div className="text-center p-4 bg-gray-50 rounded-lg">
                  <div className="text-2xl font-bold text-green-600">
                    {portfolio.totalProfit >= 0 ? '+' : ''}{formatCurrency(portfolio.totalProfit)}
                  </div>
                  <div className="text-sm text-black">Total P&L</div>
                </div>
                <div className="text-center p-4 bg-gray-50 rounded-lg">
                  <div className="text-2xl font-bold text-blue-600">
                    {portfolio.activeBets}
                  </div>
                  <div className="text-sm text-black">Active Bets</div>
                </div>
                <div className="text-center p-4 bg-gray-50 rounded-lg">
                  <div className="text-2xl font-bold text-purple-600">
                    {portfolio.winRate}%
                  </div>
                  <div className="text-sm text-black">Win Rate</div>
                </div>
              </div>
            </div>

            {/* Personal Information */}
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-black mb-4">Personal Information</h3>
              <dl className="space-y-3">
                <div>
                  <dt className="text-sm font-medium text-black">Username</dt>
                  <dd className="text-sm text-black">{profile.username || 'Not set'}</dd>
                </div>
                <div>
                  <dt className="text-sm font-medium text-black">Email</dt>
                  <dd className="text-sm text-black">{user.email}</dd>
                </div>
                <div>
                  <dt className="text-sm font-medium text-black">First Name</dt>
                  <dd className="text-sm text-black">{profile.first_name || 'Not set'}</dd>
                </div>
                <div>
                  <dt className="text-sm font-medium text-black">Last Name</dt>
                  <dd className="text-sm text-black">{profile.last_name || 'Not set'}</dd>
                </div>
                <div>
                  <dt className="text-sm font-medium text-black">Country</dt>
                  <dd className="text-sm text-black">{profile.country || 'Not set'}</dd>
                </div>
                <div>
                  <dt className="text-sm font-medium text-black">Member Since</dt>
                  <dd className="text-sm text-black">
                    {new Date(profile.created_at).toLocaleDateString()}
                  </dd>
                </div>
              </dl>
            </div>

            {/* Account Settings */}
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-black mb-4">Quick Actions</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <Link 
                  href="/settings/profile" 
                  className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
                >
                  <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center mr-3">
                    <svg className="w-5 h-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                    </svg>
                  </div>
                  <div>
                    <div className="font-medium text-black">Edit Profile</div>
                    <div className="text-sm text-black">Update personal information</div>
                  </div>
                </Link>

                <Link 
                  href="/settings/security" 
                  className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
                >
                  <div className="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center mr-3">
                    <svg className="w-5 h-5 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
                    </svg>
                  </div>
                  <div>
                    <div className="font-medium text-black">Security</div>
                    <div className="text-sm text-black">Change password & security</div>
                  </div>
                </Link>

                <Link 
                  href="/portfolio" 
                  className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
                >
                  <div className="w-10 h-10 bg-purple-100 rounded-full flex items-center justify-center mr-3">
                    <svg className="w-5 h-5 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1" />
                    </svg>
                  </div>
                  <div>
                    <div className="font-medium text-black">Portfolio</div>
                    <div className="text-sm text-black">View your investments</div>
                  </div>
                </Link>

                <Link 
                  href="/rewards" 
                  className="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors"
                >
                  <div className="w-10 h-10 bg-yellow-100 rounded-full flex items-center justify-center mr-3">
                    <svg className="w-5 h-5 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11.049 2.927c.3-.921 1.603-.921 1.902 0l1.519 4.674a1 1 0 00.95.69h4.915c.969 0 1.371 1.24.588 1.81l-3.976 2.888a1 1 0 00-.363 1.118l1.518 4.674c.3.922-.755 1.688-1.538 1.118l-3.976-2.888a1 1 0 00-1.176 0l-3.976 2.888c-.783.57-1.838-.197-1.538-1.118l1.518-4.674a1 1 0 00-.363-1.118l-3.976-2.888c-.784-.57-.38-1.81.588-1.81h4.914a1 1 0 00.951-.69l1.519-4.674z" />
                    </svg>
                  </div>
                  <div>
                    <div className="font-medium text-black">Rewards</div>
                    <div className="text-sm text-black">View points & achievements</div>
                  </div>
                </Link>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}