'use client'

import { useState } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { useRewards } from '@/lib/hooks/useRewards'
import { LoadingSpinner, PageLoadingSpinner, ButtonLoadingSpinner } from '@/components/ui/LoadingSpinner'
import Link from 'next/link'

export default function RewardsPage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <RewardsContent user={user} />
    </FastSessionWrapper>
  )
}

function RewardsContent({ user }: { user: any }) {
  const { rewardsData, loading, error, redeemReward } = useRewards(user?.id)
  const [redeemingReward, setRedeemingReward] = useState<string | null>(null)
  const [successMessage, setSuccessMessage] = useState<string>('')

  const handleRedeemReward = async (rewardId: string) => {
    setRedeemingReward(rewardId)
    setSuccessMessage('')
    
    const result = await redeemReward(rewardId)
    
    if (result.success) {
      setSuccessMessage('Reward redeemed successfully!')
      setTimeout(() => setSuccessMessage(''), 3000)
    }
    
    setRedeemingReward(null)
  }

  if (loading) {
    return <PageLoadingSpinner text="Loading rewards..." />
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Error Loading Rewards</h1>
          <p className="text-black mb-4">{error}</p>
          <Link href="/" className="text-blue-600 hover:text-blue-700">
            Go Home
          </Link>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-4xl mx-auto px-4">
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-black">Rewards & Achievements</h1>
          <p className="text-black mt-2">Track your points, achievements, and referral rewards</p>
        </div>

        {/* Success Message */}
        {successMessage && (
          <div className="mb-6 p-4 bg-green-50 border border-green-200 rounded-lg">
            <div className="flex items-center">
              <svg className="w-5 h-5 text-green-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
              </svg>
              <p className="text-green-700 text-sm">{successMessage}</p>
            </div>
          </div>
        )}

        {/* Level Progress */}
        <div className="bg-gradient-to-r from-purple-600 to-blue-600 rounded-xl p-6 mb-8 text-white">
          <div className="flex items-center justify-between mb-4">
            <div>
              <h2 className="text-xl font-semibold">Level Progress</h2>
              <p className="text-purple-100">Current Level: {rewardsData.currentLevel}</p>
            </div>
            <div className="text-right">
              <div className="text-3xl font-bold">{rewardsData.totalPoints.toLocaleString()}</div>
              <div className="text-purple-100">Total Points</div>
            </div>
          </div>
          <div className="w-full bg-purple-200 rounded-full h-3 mb-2">
            <div 
              className="bg-white h-3 rounded-full transition-all duration-500"
              style={{ width: `${Math.min(rewardsData.levelProgress, 100)}%` }}
            ></div>
          </div>
          <div className="text-sm text-purple-100">
            {rewardsData.totalPoints} / {rewardsData.nextLevelPoints} points to next level
          </div>
        </div>

        {/* Points Overview */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">This Month</div>
            <div className="text-3xl font-bold text-green-600">+{rewardsData.monthlyPoints.toLocaleString()}</div>
            <div className="text-xs text-black mt-1">Points earned this month</div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Referrals</div>
            <div className="text-3xl font-bold text-blue-600">{rewardsData.referralCount}</div>
            <div className="text-xs text-black mt-1">Friends referred</div>
          </div>
          
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
            <div className="text-sm text-black mb-1">Referral Earnings</div>
            <div className="text-3xl font-bold text-purple-600">${rewardsData.referralEarnings}</div>
            <div className="text-xs text-black mt-1">Total bonus earned</div>
          </div>
        </div>

        {/* Recent Achievements */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 mb-8">
          <div className="p-6 border-b border-gray-200">
            <h2 className="text-xl font-semibold text-black">Achievements</h2>
            <p className="text-sm text-black mt-1">{rewardsData.achievements.length} achievements unlocked</p>
          </div>
          <div className="p-6">
            {rewardsData.achievements.length > 0 ? (
              <div className="space-y-4">
                {rewardsData.achievements.map((achievement) => (
                  <div key={achievement.id} className="flex items-center space-x-4">
                    <div className="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center">
                      <span className="text-2xl">{achievement.icon}</span>
                    </div>
                    <div className="flex-1">
                      <div className="font-semibold text-black">{achievement.title}</div>
                      <div className="text-sm text-black">{achievement.description}</div>
                      {achievement.unlockedAt && (
                        <div className="text-xs text-gray-500">
                          Unlocked: {new Date(achievement.unlockedAt).toLocaleDateString()}
                        </div>
                      )}
                    </div>
                    <div className="text-sm font-medium text-green-600">
                      +{achievement.pointsAwarded} points
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center py-8">
                <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <span className="text-2xl">🏆</span>
                </div>
                <h3 className="text-lg font-medium text-black mb-2">No achievements yet</h3>
                <p className="text-black">Start betting to unlock your first achievement!</p>
                <Link href="/" className="inline-block mt-4 bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors">
                  Browse Markets
                </Link>
              </div>
            )}
          </div>
        </div>

        {/* Available Rewards */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="p-6 border-b border-gray-200">
            <h2 className="text-xl font-semibold text-black">Available Rewards</h2>
            <p className="text-sm text-black mt-1">{rewardsData.availableRewards.length} rewards available</p>
          </div>
          <div className="p-6">
            {rewardsData.availableRewards.length > 0 ? (
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {rewardsData.availableRewards.map((reward) => (
                  <div key={reward.id} className="border border-gray-200 rounded-lg p-4 relative">
                    <div className="flex items-start justify-between mb-2">
                      <div className="font-semibold text-black">{reward.title}</div>
                      <div className="text-sm font-medium text-purple-600">{reward.pointsRequired} pts</div>
                    </div>
                    <div className="text-sm text-black mb-3">{reward.description}</div>
                    {reward.rewardValue && (
                      <div className="text-sm font-medium text-green-600 mb-3">
                        Value: ${reward.rewardValue}
                      </div>
                    )}
                    <button 
                      onClick={() => handleRedeemReward(reward.id)}
                      disabled={redeemingReward === reward.id || rewardsData.totalPoints < reward.pointsRequired}
                      className={`w-full px-4 py-2 rounded-lg transition-colors flex items-center justify-center ${
                        rewardsData.totalPoints < reward.pointsRequired
                          ? 'bg-gray-300 text-gray-500 cursor-not-allowed'
                          : 'bg-purple-600 text-white hover:bg-purple-700'
                      }`}
                    >
                      {redeemingReward === reward.id ? (
                        <ButtonLoadingSpinner text="Redeeming..." />
                      ) : rewardsData.totalPoints < reward.pointsRequired ? (
                        `Need ${reward.pointsRequired - rewardsData.totalPoints} more points`
                      ) : reward.type === 'bonus' ? (
                        'Redeem'
                      ) : (
                        'Unlock'
                      )}
                    </button>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center py-8">
                <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <span className="text-2xl">🎁</span>
                </div>
                <h3 className="text-lg font-medium text-black mb-2">No rewards available</h3>
                <p className="text-black">Earn more points to unlock rewards!</p>
              </div>
            )}
          </div>
        </div>

        {/* Referral Section */}
        <div className="mt-8 bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <h2 className="text-xl font-semibold text-black mb-4">Referral Program</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <h3 className="font-medium text-black mb-2">Invite Friends</h3>
              <p className="text-sm text-black mb-4">
                Share your referral code and earn $0.10 for each friend who signs up and places their first bet.
              </p>
              <div className="flex items-center space-x-2">
                <input
                  type="text"
                  value={`${typeof window !== 'undefined' ? window.location.origin : 'https://your-domain.com'}/signup?ref=${user?.id || 'your-code'}`}
                  readOnly
                  className="flex-1 px-3 py-2 border border-gray-300 rounded-lg text-sm"
                  style={{ color: 'black' }}
                />
                <button
                  onClick={() => {
                    if (typeof window !== 'undefined' && navigator.clipboard) {
                      const referralUrl = `${window.location.origin}/signup?ref=${user?.id || 'your-code'}`
                      navigator.clipboard.writeText(referralUrl).then(() => {
                        setSuccessMessage('Referral link copied to clipboard!')
                        setTimeout(() => setSuccessMessage(''), 3000)
                      }).catch(() => {
                        // Fallback for older browsers
                        const input = document.createElement('input')
                        input.value = referralUrl
                        document.body.appendChild(input)
                        input.select()
                        document.execCommand('copy')
                        document.body.removeChild(input)
                        setSuccessMessage('Referral link copied to clipboard!')
                        setTimeout(() => setSuccessMessage(''), 3000)
                      })
                    }
                  }}
                  className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors"
                >
                  Copy
                </button>
              </div>
            </div>
            <div>
              <h3 className="font-medium text-black mb-2">Referral Stats</h3>
              <div className="space-y-2">
                <div className="flex justify-between">
                  <span className="text-black">Total Referrals:</span>
                  <span className="font-medium text-black">{rewardsData.referralCount}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-black">Total Earnings:</span>
                  <span className="font-medium text-green-600">${rewardsData.referralEarnings}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-black">Earnings per Referral:</span>
                  <span className="font-medium text-black">$0.10</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}