'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { supabase } from '@/lib/supabase-client'
import { BettingEngine } from '@/lib/betting'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

interface SeededBet {
  id: string
  market_id: string
  outcome: 'yes' | 'no'
  seeded_amount: number
  price: number
  created_at: string
  markets: {
    title: string
    end_date: string
    status: string
    yes_price: number
    no_price: number
  }
}

export default function SeededBetsPage() {
  const { user } = useAuth()

  return (
    <FastSessionWrapper requireAuth={true}>
      <SeededBetsContent user={user} />
    </FastSessionWrapper>
  )
}

function SeededBetsContent({ user }: { user: any }) {
  const router = useRouter()
  const [seededBets, setSeededBets] = useState<SeededBet[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState('')
  const [actionLoading, setActionLoading] = useState<string | null>(null)

  const fetchSeededBets = async () => {
    try {
      const { data, error } = await supabase
        .from('bets')
        .select(`
          id,
          market_id,
          outcome,
          seeded_amount,
          price,
          created_at,
          markets!inner (
            title,
            end_date,
            status,
            yes_price,
            no_price
          )
        `)
        .eq('user_id', user.id)
        .eq('status', 'seeded')
        .order('created_at', { ascending: false })

      if (error) throw error

      // Transform the data to match our interface
      const transformedData = (data || []).map((bet: any) => ({
        id: bet.id,
        market_id: bet.market_id,
        outcome: bet.outcome,
        seeded_amount: bet.seeded_amount,
        price: bet.price,
        created_at: bet.created_at,
        markets: bet.markets
      }))

      setSeededBets(transformedData)
    } catch (err) {
      console.error('Error fetching seeded bets:', err)
      setError('Failed to load seeded bets')
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    if (user) {
      fetchSeededBets()
    }
  }, [user])

  const handleMakeLive = async (betId: string, additionalAmount: string) => {
    setActionLoading(betId)
    try {
      const amount = parseFloat(additionalAmount) || 0
      const result = await BettingEngine.makeBetLive(betId, amount)

      if (!result.success) {
        throw new Error(result.error)
      }

      // Refresh the list
      await fetchSeededBets()
    } catch (err) {
      console.error('Error making bet live:', err)
      setError(err instanceof Error ? err.message : 'Failed to make bet live')
    } finally {
      setActionLoading(null)
    }
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    })
  }

  const formatPrice = (price: number) => {
    return (price * 100).toFixed(1) + '%'
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <LoadingSpinner size="lg" />
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-6xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center space-x-4 mb-4">
            <Link href="/" className="text-black hover:text-black">
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
              </svg>
            </Link>
            <h1 className="text-3xl font-bold text-black">Seeded Bets</h1>
          </div>
          <p className="text-black">
            Manage your seeded bets. Seeded bets allow you to pre-fund positions and get better payout ratios when you win.
          </p>
        </div>

        {error && (
          <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg">
            <div className="flex items-center">
              <svg className="w-5 h-5 text-red-400 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              <p className="text-red-700 text-sm">{error}</p>
            </div>
          </div>
        )}

        {/* Seeded Bets List */}
        {seededBets.length === 0 ? (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8 text-center">
            <div className="w-16 h-16 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg className="w-8 h-8 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            </div>
            <h3 className="text-lg font-semibold text-black mb-2">No Seeded Bets</h3>
            <p className="text-black mb-6">
              You haven't seeded any bets yet. Seeded bets allow you to pre-fund positions for better payout ratios.
            </p>
            <Link 
              href="/"
              className="inline-flex items-center px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white font-medium rounded-lg transition-colors"
            >
              Browse Markets
            </Link>
          </div>
        ) : (
          <div className="space-y-4">
            {seededBets.map((bet) => (
              <div key={bet.id} className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
                <div className="flex items-start justify-between mb-4">
                  <div className="flex-1">
                    <h3 className="text-lg font-semibold text-black mb-2">
                      {bet.markets.title}
                    </h3>
                    <div className="flex items-center space-x-4 text-sm text-black">
                      <span>Seeded: {formatDate(bet.created_at)}</span>
                      <span>•</span>
                      <span>Ends: {formatDate(bet.markets.end_date)}</span>
                      <span>•</span>
                      <span className="capitalize">{bet.markets.status}</span>
                    </div>
                  </div>
                  <Link 
                    href={`/markets/${bet.market_id}`}
                    className="text-blue-600 hover:text-blue-700 text-sm font-medium"
                  >
                    View Market →
                  </Link>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {/* Bet Details */}
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium text-black">Position:</span>
                      <span className={`text-sm font-bold ${bet.outcome === 'yes' ? 'text-green-600' : 'text-red-600'}`}>
                        {bet.outcome.toUpperCase()}
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium text-black">Seeded Amount:</span>
                      <span className="text-sm font-bold text-black">${bet.seeded_amount.toFixed(2)}</span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium text-black">Seeded Price:</span>
                      <span className="text-sm font-bold text-black">{formatPrice(bet.price)}</span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium text-black">Current Price:</span>
                      <span className="text-sm font-bold text-black">
                        {formatPrice(bet.outcome === 'yes' ? bet.markets.yes_price : bet.markets.no_price)}
                      </span>
                    </div>
                  </div>

                  {/* Price Movement */}
                  <div className="space-y-3">
                    <div className="text-sm font-medium text-black mb-2">Price Movement</div>
                    <div className="flex items-center space-x-2">
                      <div className={`w-3 h-3 rounded-full ${bet.outcome === 'yes' ? 'bg-green-500' : 'bg-red-500'}`}></div>
                      <span className="text-sm text-black">
                        {bet.outcome === 'yes' ? 'YES' : 'NO'} price moved from {formatPrice(bet.price)} to{' '}
                        {formatPrice(bet.outcome === 'yes' ? bet.markets.yes_price : bet.markets.no_price)}
                      </span>
                    </div>
                    <div className="text-xs text-black">
                      {bet.markets.status !== 'active' && 'Market is no longer active'}
                    </div>
                  </div>

                  {/* Actions */}
                  <div className="space-y-3">
                    <div className="text-sm font-medium text-black mb-2">Actions</div>
                    {bet.markets.status === 'active' ? (
                      <div className="space-y-2">
                        <button
                          onClick={() => handleMakeLive(bet.id, '0')}
                          disabled={actionLoading === bet.id}
                          className="w-full bg-green-600 hover:bg-green-700 disabled:bg-green-400 text-white py-2 px-4 rounded-lg font-medium transition-colors text-sm"
                        >
                          {actionLoading === bet.id ? 'Activating...' : 'Make Live (No Additional)'}
                        </button>
                        <div className="text-xs text-black text-center">
                          Or add more funds when making live
                        </div>
                      </div>
                    ) : (
                      <div className="text-sm text-black text-center py-2">
                        Market is {bet.markets.status}
                      </div>
                    )}
                  </div>
                </div>

                {/* Additional Amount Input (only show for active markets) */}
                {bet.markets.status === 'active' && (
                  <div className="mt-4 pt-4 border-t border-gray-100">
                    <div className="flex items-center space-x-3">
                      <input
                        type="number"
                        placeholder="Additional amount (optional)"
                        min="0"
                        step="0.01"
                        className="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm"
                        onKeyPress={(e) => {
                          if (e.key === 'Enter') {
                            const target = e.target as HTMLInputElement
                            handleMakeLive(bet.id, target.value)
                          }
                        }}
                        style={{ color: 'black' }}
                      />
                      <button
                        onClick={(e) => {
                          const input = e.currentTarget.previousElementSibling as HTMLInputElement
                          handleMakeLive(bet.id, input.value)
                        }}
                        disabled={actionLoading === bet.id}
                        className="bg-green-600 hover:bg-green-700 disabled:bg-green-400 text-white py-2 px-4 rounded-lg font-medium transition-colors text-sm"
                      >
                        {actionLoading === bet.id ? 'Adding...' : 'Add & Make Live'}
                      </button>
                    </div>
                  </div>
                )}
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  )
}
