'use client'

import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { useProfile } from '@/lib/hooks/useProfile'
import { useLoginSessions } from '@/lib/hooks/useLoginSessions'
import Link from 'next/link'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

export default function SettingsPage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <SettingsContent user={user} />
    </FastSessionWrapper>
  )
}

function SettingsContent({ user }: { user: any }) {
  const { profile, loading, error } = useProfile(user?.id)
  const { sessions } = useLoginSessions(user?.id)

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading settings...</p>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Error Loading Settings</h1>
          <p className="text-black mb-4">{error}</p>
          <Link href="/" className="text-blue-600 hover:text-blue-700">
            Go Home
          </Link>
        </div>
      </div>
    )
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-6xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-black">Settings</h1>
          <p className="text-black mt-2">Manage your account preferences and security</p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
          {/* Settings Navigation */}
          <div className="lg:col-span-1">
            <nav className="space-y-2">
              <Link 
                href="/settings/profile" 
                className="flex items-center px-4 py-3 text-black rounded-lg hover:bg-white hover:shadow-sm transition-colors"
              >
                <svg className="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                </svg>
                Profile
              </Link>
              
              <Link 
                href="/settings/security" 
                className="flex items-center px-4 py-3 text-black rounded-lg hover:bg-white hover:shadow-sm transition-colors"
              >
                <svg className="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
                </svg>
                Security
              </Link>
              
              <Link 
                href="/settings/preferences" 
                className="flex items-center px-4 py-3 text-black rounded-lg hover:bg-white hover:shadow-sm transition-colors"
              >
                <svg className="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0l1.403 5.759c.2.821.8 1.47 1.609 1.739l5.478 1.593c1.761.512 1.761 2.954 0 3.466l-5.478 1.593c-.809.269-1.409.918-1.609 1.739l-1.403 5.759c-.426 1.756-2.924 1.756-3.35 0l-1.403-5.759c-.2-.821-.8-1.47-1.609-1.739l-5.478-1.593c-1.761-.512-1.761-2.954 0-3.466l5.478-1.593c.809-.269 1.409-.918 1.609-1.739l1.403-5.759z" />
                </svg>
                Preferences
              </Link>
            </nav>
          </div>

          {/* Settings Content */}
          <div className="lg:col-span-3">
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
              <h2 className="text-2xl font-semibold text-black mb-6">Account Settings</h2>
              
              <div className="space-y-8">
                {/* Profile Section */}
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Profile Information</h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">Username</label>
                      <input
                        type="text"
                        value={profile?.username || 'Not set'}
                        readOnly
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-black"
                      style={{ color: 'black' }} />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">Email</label>
                      <input
                        type="email"
                        value={user.email}
                        readOnly
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-black"
                      style={{ color: 'black' }} />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">Full Name</label>
                      <input
                        type="text"
                        value={`${profile?.first_name || ''} ${profile?.last_name || ''}`.trim() || 'Not set'}
                        readOnly
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-black"
                      style={{ color: 'black' }} />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-black mb-2">Country</label>
                      <input
                        type="text"
                        value={profile?.country || 'Not set'}
                        readOnly
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-black"
                      style={{ color: 'black' }} />
                    </div>
                  </div>
                  <div className="mt-4">
                    <Link 
                      href="/settings/profile" 
                      className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-black bg-white hover:bg-gray-50"
                    >
                      Edit Profile
                    </Link>
                  </div>
                </div>

                {/* Account Stats */}
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Account Statistics</h3>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div className="text-center p-4 bg-gray-50 rounded-lg">
                      <div className="text-2xl font-bold text-green-600">
                        {formatCurrency(profile?.balance || 0)}
                      </div>
                      <div className="text-sm text-black">Account Balance</div>
                    </div>
                    <div className="text-center p-4 bg-gray-50 rounded-lg">
                      <div className="text-2xl font-bold text-blue-600">
                        {profile?.total_points || 0}
                      </div>
                      <div className="text-sm text-black">Total Points</div>
                    </div>
                    <div className="text-center p-4 bg-gray-50 rounded-lg">
                      <div className="text-2xl font-bold">
                        {profile?.kyc_verified ? (
                          <span className="text-blue-600">✓</span>
                        ) : profile?.kyc_status === 'pending' ? (
                          <span className="text-yellow-600">⏳</span>
                        ) : (
                          <span className="text-red-600">✗</span>
                        )}
                      </div>
                      <div className="text-sm text-black">
                        {profile?.kyc_verified ? 'Verified' : 
                         profile?.kyc_status === 'pending' ? 'Pending' : 
                         'Not Verified'}
                      </div>
                    </div>
                  </div>
                </div>

                {/* Security Section */}
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Security</h3>
                  <div className="space-y-4">
                    {/* KYC Verification Link */}
                    <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                      <div className="flex items-center">
                        <div className="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center mr-4">
                          <svg className="w-5 h-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                          </svg>
                        </div>
                        <div>
                          <h4 className="font-medium text-black">KYC Verification</h4>
                          <p className="text-sm text-gray-600">
                            {profile?.kyc_verified ? 'Identity verified' : 
                             profile?.kyc_status === 'pending' ? 'Verification pending' : 
                             'Complete identity verification'}
                          </p>
                        </div>
                      </div>
                      <Link 
                        href="/kyc" 
                        className="text-blue-600 hover:text-blue-700 font-medium"
                      >
                        {profile?.kyc_verified ? 'View Status' : 'Verify Now'}
                      </Link>
                    </div>
                    
                    <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                      <div>
                        <div className="font-medium text-black">Password</div>
                        <div className="text-sm text-black">Last changed 30 days ago</div>
                      </div>
                      <Link 
                        href="/settings/security" 
                        className="text-blue-600 hover:text-blue-700 text-sm font-medium"
                      >
                        Change
                      </Link>
                    </div>
                    
                    <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                      <div>
                        <div className="font-medium text-black">Two-Factor Authentication</div>
                        <div className="text-sm text-black">Add an extra layer of security</div>
                      </div>
                      <button className="text-blue-600 hover:text-blue-700 text-sm font-medium">
                        Enable
                      </button>
                    </div>
                  </div>
                </div>

                {/* Account Info */}
                <div>
                  <h3 className="text-lg font-medium text-black mb-4">Account Information</h3>
                  <dl className="space-y-3">
                    <div>
                      <dt className="text-sm font-medium text-black">Member Since</dt>
                      <dd className="text-sm text-black">
                        {profile?.created_at ? new Date(profile.created_at).toLocaleDateString() : 'Unknown'}
                      </dd>
                    </div>
                    <div>
                      <dt className="text-sm font-medium text-black">Account Status</dt>
                      <dd className="text-sm text-black capitalize">
                        {profile?.account_status || 'Active'}
                      </dd>
                    </div>
                    <div>
                      <dt className="text-sm font-medium text-black">Last Login</dt>
                      <dd className="text-sm text-black">
                        {sessions && sessions.length > 0 
                          ? new Date(sessions[0].login_time).toLocaleDateString() 
                          : profile?.last_login 
                            ? new Date(profile.last_login).toLocaleDateString() 
                            : 'Unknown'
                        }
                      </dd>
                    </div>
                    <div>
                      <dt className="text-sm font-medium text-black">Total Sessions</dt>
                      <dd className="text-sm text-black">
                        {sessions?.length || 0}
                      </dd>
                    </div>
                    <div>
                      <dt className="text-sm font-medium text-black">Active Sessions</dt>
                      <dd className="text-sm text-black">
                        {sessions?.filter(session => session.is_active).length || 0}
                      </dd>
                    </div>
                  </dl>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}