'use client'

import { useAuth } from '@/components/auth/ClientAuthProvider'
import FastSettingsWrapper from '@/components/auth/FastSettingsWrapper'
import { useUserPreferencesSimple } from '@/lib/hooks/useUserPreferencesSimple'
import { useToast } from '@/components/ui/Toast'
import Link from 'next/link'
import { useState, useEffect } from 'react'

export default function PreferencesSettingsPage() {
  const { user } = useAuth()
  
  return (
    <FastSettingsWrapper requireAuth={true}>
      <PreferencesSettingsContent user={user} />
    </FastSettingsWrapper>
  )
}

function PreferencesSettingsContent({ user }: { user: any }) {
  const { preferences, loading, error, updatePreferences } = useUserPreferencesSimple(user?.id)
  const { addToast } = useToast()
  const [saving, setSaving] = useState(false)
  const [formData, setFormData] = useState({
    email_notifications: true,
    email_market_updates: true,
    email_bet_results: true,
    email_security_alerts: true,
    push_notifications: false,
    theme: 'light',
    language: 'en',
    timezone: 'UTC',
    public_profile: false,
    show_betting_activity: true
  })

  useEffect(() => {
    if (preferences) {
      setFormData({
        email_notifications: preferences.email_notifications,
        email_market_updates: preferences.email_market_updates,
        email_bet_results: preferences.email_bet_results,
        email_security_alerts: preferences.email_security_alerts,
        push_notifications: preferences.push_notifications,
        theme: preferences.theme,
        language: preferences.language,
        timezone: preferences.timezone,
        public_profile: preferences.public_profile,
        show_betting_activity: preferences.show_betting_activity
      })
    }
  }, [preferences])

  const handleSave = async (e: React.FormEvent) => {
    e.preventDefault()
    
    try {
      setSaving(true)
      await updatePreferences({
        ...formData,
        theme: formData.theme as 'light' | 'dark' | 'system'
      })
      addToast({
        type: 'success',
        title: 'Preferences Updated',
        message: 'Your preferences have been saved successfully.'
      })
    } catch (err) {
      addToast({
        type: 'error',
        title: 'Update Failed',
        message: 'Failed to update preferences. Please try again.'
      })
    } finally {
      setSaving(false)
    }
  }

  const handleInputChange = (field: string, value: any) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }))
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-black">Loading preferences...</p>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <p className="text-red-600 mb-4">Error loading preferences: {error}</p>
          <button 
            onClick={() => window.location.reload()} 
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700"
          >
            Retry
          </button>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center space-x-4 mb-4">
            <Link href="/settings" className="text-black hover:text-black">
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
              </svg>
            </Link>
            <h1 className="text-3xl font-bold text-black">Preferences</h1>
          </div>
          <p className="text-black">Customize your experience and notification settings</p>
        </div>

        <form onSubmit={handleSave} className="space-y-6">
          {/* Notifications */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <h2 className="text-xl font-semibold text-black mb-6">Notifications</h2>
            <div className="space-y-6">
              <div>
                <h3 className="text-lg font-medium text-black mb-4">Email Notifications</h3>
                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <div className="font-medium text-black">Market Updates</div>
                      <div className="text-sm text-black">Get notified when markets you're watching change</div>
                    </div>
                    <label className="relative inline-flex items-center cursor-pointer">
                      <input 
                        type="checkbox" 
                        className="sr-only peer" 
                        checked={formData.email_market_updates}
                        onChange={(e) => handleInputChange('email_market_updates', e.target.checked)}
                      />
                      <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                    </label>
                  </div>
                  
                  <div className="flex items-center justify-between">
                    <div>
                      <div className="font-medium text-black">Bet Results</div>
                      <div className="text-sm text-black">Receive notifications when your bets are resolved</div>
                    </div>
                    <label className="relative inline-flex items-center cursor-pointer">
                      <input 
                        type="checkbox" 
                        className="sr-only peer" 
                        checked={formData.email_bet_results}
                        onChange={(e) => handleInputChange('email_bet_results', e.target.checked)}
                      />
                      <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                    </label>
                  </div>
                  
                  <div className="flex items-center justify-between">
                    <div>
                      <div className="font-medium text-black">Account Security</div>
                      <div className="text-sm text-black">Important security alerts and login notifications</div>
                    </div>
                    <label className="relative inline-flex items-center cursor-pointer">
                      <input 
                        type="checkbox" 
                        className="sr-only peer" 
                        checked={formData.email_security_alerts}
                        onChange={(e) => handleInputChange('email_security_alerts', e.target.checked)}
                      />
                      <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                    </label>
                  </div>
                </div>
              </div>

              <div>
                <h3 className="text-lg font-medium text-black mb-4">Push Notifications</h3>
                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <div className="font-medium text-black">Mobile Push</div>
                      <div className="text-sm text-black">Receive push notifications on your mobile device</div>
                    </div>
                    <label className="relative inline-flex items-center cursor-pointer">
                      <input 
                        type="checkbox" 
                        className="sr-only peer"
                        checked={formData.push_notifications}
                        onChange={(e) => handleInputChange('push_notifications', e.target.checked)}
                      />
                      <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                    </label>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Appearance */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <h2 className="text-xl font-semibold text-black mb-6">Appearance</h2>
            <div className="space-y-6">
              <div>
                <label className="block text-sm font-medium text-black mb-2">Theme</label>
                <div className="flex space-x-4">
                  <label className="flex items-center">
                    <input 
                      type="radio" 
                      name="theme" 
                      value="light" 
                      className="mr-2" 
                      checked={formData.theme === 'light'}
                      onChange={(e) => handleInputChange('theme', e.target.value)}
                    />
                    Light
                  </label>
                  <label className="flex items-center">
                    <input 
                      type="radio" 
                      name="theme" 
                      value="dark" 
                      className="mr-2"
                      checked={formData.theme === 'dark'}
                      onChange={(e) => handleInputChange('theme', e.target.value)}
                    />
                    Dark
                  </label>
                  <label className="flex items-center">
                    <input 
                      type="radio" 
                      name="theme" 
                      value="system" 
                      className="mr-2"
                      checked={formData.theme === 'system'}
                      onChange={(e) => handleInputChange('theme', e.target.value)}
                    />
                    System
                  </label>
                </div>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-black mb-2">Language</label>
                <select 
                  className="w-full md:w-64 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  value={formData.language}
                  onChange={(e) => handleInputChange('language', e.target.value)}
                >
                  <option value="en">English</option>
                  <option value="es">Español</option>
                  <option value="fr">Français</option>
                  <option value="de">Deutsch</option>
                </select>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-black mb-2">Timezone</label>
                <select 
                  className="w-full md:w-64 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  value={formData.timezone}
                  onChange={(e) => handleInputChange('timezone', e.target.value)}
                >
                  <option value="UTC">UTC</option>
                  <option value="America/New_York">Eastern Time</option>
                  <option value="America/Chicago">Central Time</option>
                  <option value="America/Denver">Mountain Time</option>
                  <option value="America/Los_Angeles">Pacific Time</option>
                  <option value="Europe/London">London</option>
                  <option value="Europe/Paris">Paris</option>
                  <option value="Asia/Tokyo">Tokyo</option>
                </select>
              </div>
            </div>
          </div>

          {/* Privacy */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <h2 className="text-xl font-semibold text-black mb-6">Privacy</h2>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <div>
                  <div className="font-medium text-black">Public Profile</div>
                  <div className="text-sm text-black">Allow others to see your profile information</div>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input 
                    type="checkbox" 
                    className="sr-only peer"
                    checked={formData.public_profile}
                    onChange={(e) => handleInputChange('public_profile', e.target.checked)}
                  />
                  <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                </label>
              </div>
              
              <div className="flex items-center justify-between">
                <div>
                  <div className="font-medium text-black">Show Betting Activity</div>
                  <div className="text-sm text-black">Display your betting activity on your profile</div>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input 
                    type="checkbox" 
                    className="sr-only peer"
                    checked={formData.show_betting_activity}
                    onChange={(e) => handleInputChange('show_betting_activity', e.target.checked)}
                  />
                  <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                </label>
              </div>
            </div>
          </div>

          {/* Save Button */}
          <div className="flex justify-end">
            <button 
              type="submit"
              disabled={saving}
              className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {saving ? 'Saving...' : 'Save Preferences'}
            </button>
          </div>
        </form>
      </div>
    </div>
  )
}