'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/components/auth/ClientAuthProvider'
import FastSettingsWrapper from '@/components/auth/FastSettingsWrapper'
import { useProfileOptimized } from '@/lib/hooks/useProfileOptimized'
import { supabase } from '@/lib/supabase-client'
import { uploadFile } from '@/lib/storage'
import { useToast } from '@/components/ui/Toast'
import Link from 'next/link'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

export default function ProfileSettingsPage() {
  const { user } = useAuth()
  
  return (
    <FastSettingsWrapper requireAuth={true}>
      <ProfileSettingsContent user={user} />
    </FastSettingsWrapper>
  )
}

function ProfileSettingsContent({ user }: { user: any }) {
  console.log('ProfileSettingsContent: user data:', user)
  const { profile, loading, error, updateProfile } = useProfileOptimized(user?.id)
  console.log('ProfileSettingsContent: profile data:', { profile, loading, error })
  const { addToast } = useToast()
  const [formData, setFormData] = useState({
    username: '',
    first_name: '',
    last_name: '',
    phone: '',
    country: 'ZW',
    date_of_birth: '',
    bio: ''
  })
  const [isSubmitting, setIsSubmitting] = useState(false)

  // Initialize form data when profile loads
  useEffect(() => {
    console.log('ProfileSettingsContent: useEffect triggered', { profile })
    if (profile) {
      console.log('ProfileSettingsContent: Initializing form data with profile:', profile)
      setFormData({
        username: profile.username || '',
        first_name: profile.first_name || '',
        last_name: profile.last_name || '',
        phone: profile.phone || '',
        country: profile.country || 'ZW',
        date_of_birth: profile.date_of_birth || '',
        bio: profile.bio || ''
      })
    }
  }, [profile])

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target
    setFormData(prev => ({
      ...prev,
      [name]: value
    }))
  }

  const handleAvatarUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]
    if (!file || !user) return

    try {
      setIsSubmitting(true)

      // Upload using the storage utility
      const result = await uploadFile({
        bucket: 'avatars',
        userId: user.id,
        file: file,
        expiresIn: 60 * 60 * 24 * 365 // 1 year expiry for avatars
      })

      if (!result.success) {
        throw new Error(result.error || 'Failed to upload avatar')
      }

      // Update profile with new avatar URL
      const updateResult = await updateProfile({
        avatar_url: result.url
      })

      if (updateResult?.success) {
        addToast({
          type: 'success',
          title: 'Profile Picture Updated',
          message: 'Your profile picture has been updated successfully.'
        })
      } else {
        addToast({
          type: 'error',
          title: 'Upload Failed',
          message: updateResult?.error || 'Failed to update profile picture.'
        })
      }
    } catch (error) {
      console.error('Error uploading avatar:', error)
      addToast({
        type: 'error',
        title: 'Upload Failed',
        message: error instanceof Error ? error.message : 'Error uploading profile picture'
      })
    } finally {
      setIsSubmitting(false)
    }
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsSubmitting(true)

    try {
      const result = await updateProfile({
        username: formData.username,
        first_name: formData.first_name,
        last_name: formData.last_name,
        phone: formData.phone,
        country: formData.country,
        date_of_birth: formData.date_of_birth || null,
        bio: formData.bio
      })

      if (result?.success) {
        addToast({
          type: 'success',
          title: 'Profile Updated',
          message: 'Your profile has been updated successfully.'
        })
      } else {
        addToast({
          type: 'error',
          title: 'Update Failed',
          message: result?.error || 'Failed to update profile.'
        })
      }
    } catch (err) {
      addToast({
        type: 'error',
        title: 'Update Failed',
        message: 'An unexpected error occurred.'
      })
    } finally {
      setIsSubmitting(false)
    }
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading profile settings...</p>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Error Loading Profile</h1>
          <p className="text-black mb-4">{error}</p>
          <Link href="/settings" className="text-blue-600 hover:text-blue-700">
            Back to Settings
          </Link>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center space-x-4 mb-4">
            <Link href="/settings" className="text-black hover:text-black">
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
              </svg>
            </Link>
            <h1 className="text-3xl font-bold text-black">Profile Settings</h1>
          </div>
          <p className="text-black">Update your personal information and profile details</p>
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            {/* Profile Picture */}
            <div className="mb-8">
              <label className="block text-sm font-medium text-black mb-4">Profile Picture</label>
              <div className="flex items-center space-x-6">
                <div className="w-20 h-20 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white text-2xl font-bold overflow-hidden">
                  {profile?.avatar_url ? (
                    <img 
                      src={profile.avatar_url} 
                      alt="Profile" 
                      className="w-full h-full object-cover"
                    />
                  ) : (
                    formData.username?.charAt(0).toUpperCase() || user?.email?.charAt(0).toUpperCase()
                  )}
                </div>
                <div>
                  <input
                    type="file"
                    id="avatar-upload"
                    accept="image/*"
                    className="hidden"
                    onChange={handleAvatarUpload}
                    disabled={isSubmitting}
                  />
                  <label 
                    htmlFor="avatar-upload" 
                    className={`px-4 py-2 rounded-lg transition-colors cursor-pointer inline-block ${
                      isSubmitting 
                        ? 'bg-gray-400 text-white cursor-not-allowed' 
                        : 'bg-blue-600 text-white hover:bg-blue-700'
                    }`}
                  >
                    {isSubmitting ? 'Uploading...' : 'Upload Photo'}
                  </label>
                  <p className="text-sm text-black mt-1">JPG, PNG up to 2MB</p>
                </div>
              </div>
            </div>

            {/* Personal Information */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-black mb-2">First Name</label>
                <input
                  type="text"
                  name="first_name"
                  value={formData.first_name}
                  onChange={handleInputChange}
                  placeholder="Enter your first name"
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  disabled={isSubmitting}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-black mb-2">Last Name</label>
                <input
                  type="text"
                  name="last_name"
                  value={formData.last_name}
                  onChange={handleInputChange}
                  placeholder="Enter your last name"
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  disabled={isSubmitting}
                />
              </div>
            </div>

            <div className="mt-6">
              <label className="block text-sm font-medium text-black mb-2">Username</label>
              <input
                type="text"
                name="username"
                value={formData.username}
                onChange={handleInputChange}
                placeholder="Choose a unique username"
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                disabled={isSubmitting}
              />
            </div>

            <div className="mt-6">
              <label className="block text-sm font-medium text-black mb-2">Email</label>
              <input
                type="email"
                value={user?.email || ''}
                readOnly
                className="w-full px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-black"
              />
              <p className="text-sm text-black mt-1">Email cannot be changed. Contact support if needed.</p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
              <div>
                <label className="block text-sm font-medium text-black mb-2">Phone Number</label>
                <input
                  type="tel"
                  name="phone"
                  value={formData.phone}
                  onChange={handleInputChange}
                  placeholder="+263 77 123 4567"
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  disabled={isSubmitting}
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-black mb-2">Country</label>
                <select 
                  name="country"
                  value={formData.country}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  disabled={isSubmitting}
                >
                  <option value="ZW">🇿🇼 Zimbabwe</option>
                </select>
              </div>
            </div>

            <div className="mt-6">
              <label className="block text-sm font-medium text-black mb-2">Date of Birth</label>
              <input
                type="date"
                name="date_of_birth"
                value={formData.date_of_birth}
                onChange={handleInputChange}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                disabled={isSubmitting}
              />
            </div>

            <div className="mt-6">
              <label className="block text-sm font-medium text-black mb-2">Bio</label>
              <textarea
                rows={4}
                name="bio"
                value={formData.bio}
                onChange={handleInputChange}
                placeholder="Tell us about yourself..."
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                disabled={isSubmitting}
              />
            </div>

            {/* Save Button */}
            <div className="flex justify-end space-x-4 mt-8">
              <Link 
                href="/settings" 
                className="px-6 py-2 border border-gray-300 text-black rounded-lg hover:bg-gray-50 transition-colors"
              >
                Cancel
              </Link>
              <button 
                type="submit"
                disabled={isSubmitting}
                className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center"
              >
                {isSubmitting && <div className="mr-2"><LoadingSpinner size="sm" /></div>}
                Save Changes
              </button>
            </div>
          </div>
        </form>
      </div>
    </div>
  )
}