'use client'

import { useAuth } from '@/components/auth/ClientAuthProvider'
import FastSettingsWrapper from '@/components/auth/FastSettingsWrapper'
import { useUserPreferencesSimple } from '@/lib/hooks/useUserPreferencesSimple'
import { useToast } from '@/components/ui/Toast'
import LoginActivity from '@/components/ui/LoginActivity'
import Link from 'next/link'
import React, { useState } from 'react'
import { supabase } from '@/lib/supabase-client'
import { loginTracking } from '@/lib/services/loginTracking'
import DeleteAccountModal from '@/components/ui/DeleteAccountModal'

export default function SecuritySettingsPage() {
  const { user } = useAuth()
  
  return (
    <FastSettingsWrapper requireAuth={true}>
      <SecuritySettingsContent user={user} />
    </FastSettingsWrapper>
  )
}

function SecuritySettingsContent({ user }: { user: any }) {
  console.log('SecuritySettingsContent: Component rendered', { user, userId: user?.id })
  const { preferences, updatePreferences } = useUserPreferencesSimple(user?.id)
  const { addToast } = useToast()
  
  // Fallback: get user from Supabase if auth provider fails
  const [fallbackUser, setFallbackUser] = useState<any>(null)
  
  React.useEffect(() => {
    if (!user) {
      const getCurrentUser = async () => {
        try {
          const { data: { user: currentUser } } = await supabase.auth.getUser()
          console.log('SecuritySettingsContent: Fallback user fetch', { currentUser })
          setFallbackUser(currentUser)
        } catch (error) {
          console.error('SecuritySettingsContent: Error fetching fallback user', error)
        }
      }
      getCurrentUser()
    }
  }, [user])
  
  const effectiveUser = user || fallbackUser
  const [passwordForm, setPasswordForm] = useState({
    currentPassword: '',
    newPassword: '',
    confirmPassword: ''
  })
  const [changingPassword, setChangingPassword] = useState(false)
  const [signingOutAll, setSigningOutAll] = useState(false)
  const [showDeleteModal, setShowDeleteModal] = useState(false)

  const handlePasswordChange = async (e: React.FormEvent) => {
    e.preventDefault()
    
    if (passwordForm.newPassword !== passwordForm.confirmPassword) {
      addToast({
        type: 'error',
        title: 'Password Mismatch',
        message: 'New password and confirm password do not match.'
      })
      return
    }

    if (passwordForm.newPassword.length < 6) {
      addToast({
        type: 'error',
        title: 'Password Too Short',
        message: 'Password must be at least 6 characters long.'
      })
      return
    }

    try {
      setChangingPassword(true)
      
      const { error } = await supabase.auth.updateUser({
        password: passwordForm.newPassword
      })

      if (error) {
        throw error
      }

      addToast({
        type: 'success',
        title: 'Password Updated',
        message: 'Your password has been changed successfully.'
      })

      setPasswordForm({
        currentPassword: '',
        newPassword: '',
        confirmPassword: ''
      })
    } catch (err) {
      addToast({
        type: 'error',
        title: 'Password Update Failed',
        message: err instanceof Error ? err.message : 'Failed to update password. Please try again.'
      })
    } finally {
      setChangingPassword(false)
    }
  }

  const handleTwoFactorToggle = async (method: 'sms' | 'authenticator') => {
    try {
      await updatePreferences({
        two_factor_enabled: !preferences?.two_factor_enabled,
        two_factor_method: method
      })
      
      addToast({
        type: 'success',
        title: 'Two-Factor Authentication',
        message: `${preferences?.two_factor_enabled ? 'Disabled' : 'Enabled'} ${method.toUpperCase()} authentication.`
      })
    } catch (err) {
      addToast({
        type: 'error',
        title: 'Update Failed',
        message: 'Failed to update two-factor authentication settings.'
      })
    }
  }

  const handleSignOutAll = async () => {
    try {
      console.log('SecuritySettingsContent: handleSignOutAll called', { user, effectiveUser })
      
      if (!effectiveUser?.id) {
        console.error('SecuritySettingsContent: User not authenticated', { user, effectiveUser })
        addToast({
          type: 'error',
          title: 'Authentication Error',
          message: 'You must be logged in to sign out all devices.'
        })
        return
      }
      
      setSigningOutAll(true)

      // End all login sessions in database
      console.log('SecuritySettingsContent: Ending all login sessions...', { userId: effectiveUser.id })
      const sessionsEnded = await loginTracking.endAllSessions(effectiveUser.id)
      console.log('SecuritySettingsContent: Sessions ended result:', sessionsEnded)
      
      if (!sessionsEnded) {
        console.warn('SecuritySettingsContent: Failed to end login sessions, but continuing with auth sign out')
        // Don't throw error, just continue with auth sign out
      }

      // Sign out from Supabase auth
      console.log('SecuritySettingsContent: Signing out from Supabase auth...')
      const { error } = await supabase.auth.signOut({ scope: 'global' })
      
      if (error) {
        console.error('SecuritySettingsContent: Supabase sign out error:', error)
        throw error
      }
      
      console.log('SecuritySettingsContent: Successfully signed out from all devices')

      addToast({
        type: 'success',
        title: 'Signed Out',
        message: 'You have been signed out of all devices.'
      })
      
      // Redirect to home page after successful sign out
      setTimeout(() => {
        window.location.href = '/'
      }, 2000)
    } catch (err) {
      console.error('Sign out error:', err)
      addToast({
        type: 'error',
        title: 'Sign Out Failed',
        message: 'Failed to sign out of all devices. Please try again.'
      })
    } finally {
      setSigningOutAll(false)
    }
  }

  const handleDeleteAccountSuccess = () => {
    // Redirect to home page after account deletion
    window.location.href = '/'
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-8">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center space-x-4 mb-4">
            <Link href="/settings" className="text-black hover:text-black">
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
              </svg>
            </Link>
            <h1 className="text-3xl font-bold text-black">Security Settings</h1>
          </div>
          <p className="text-black">Manage your password and security preferences</p>
        </div>

        <div className="space-y-6">
          {/* Change Password */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <h2 className="text-xl font-semibold text-black mb-6">Change Password</h2>
            <form onSubmit={handlePasswordChange} className="space-y-6">
              <div>
                <label className="block text-sm font-medium text-black mb-2">New Password</label>
                <input
                  type="password"
                  placeholder="Enter your new password"
                  value={passwordForm.newPassword}
                  onChange={(e) => setPasswordForm(prev => ({ ...prev, newPassword: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-black mb-2">Confirm New Password</label>
                <input
                  type="password"
                  placeholder="Confirm your new password"
                  value={passwordForm.confirmPassword}
                  onChange={(e) => setPasswordForm(prev => ({ ...prev, confirmPassword: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  required
                />
              </div>
              
              <button 
                type="submit"
                disabled={changingPassword}
                className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {changingPassword ? 'Updating...' : 'Update Password'}
              </button>
            </form>
          </div>

          {/* Two-Factor Authentication */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <h2 className="text-xl font-semibold text-black mb-6">Two-Factor Authentication</h2>
            <div className="space-y-4">
              <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                <div>
                  <div className="font-medium text-black">SMS Authentication</div>
                  <div className="text-sm text-black">Receive codes via text message</div>
                  {preferences?.two_factor_enabled && preferences?.two_factor_method === 'sms' && (
                    <div className="text-xs text-green-600 mt-1">✓ Currently enabled</div>
                  )}
                </div>
                <button 
                  onClick={() => handleTwoFactorToggle('sms')}
                  className={`px-4 py-2 rounded-lg transition-colors ${
                    preferences?.two_factor_enabled && preferences?.two_factor_method === 'sms'
                      ? 'bg-red-600 hover:bg-red-700 text-white'
                      : 'bg-green-600 hover:bg-green-700 text-white'
                  }`}
                >
                  {preferences?.two_factor_enabled && preferences?.two_factor_method === 'sms' ? 'Disable' : 'Enable'}
                </button>
              </div>
              
              <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                <div>
                  <div className="font-medium text-black">Authenticator App</div>
                  <div className="text-sm text-black">Use Google Authenticator or similar</div>
                  {preferences?.two_factor_enabled && preferences?.two_factor_method === 'authenticator' && (
                    <div className="text-xs text-green-600 mt-1">✓ Currently enabled</div>
                  )}
                </div>
                <button 
                  onClick={() => handleTwoFactorToggle('authenticator')}
                  className={`px-4 py-2 rounded-lg transition-colors ${
                    preferences?.two_factor_enabled && preferences?.two_factor_method === 'authenticator'
                      ? 'bg-red-600 hover:bg-red-700 text-white'
                      : 'bg-blue-600 hover:bg-blue-700 text-white'
                  }`}
                >
                  {preferences?.two_factor_enabled && preferences?.two_factor_method === 'authenticator' ? 'Disable' : 'Setup'}
                </button>
              </div>
            </div>
          </div>

          {/* Login History */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <h2 className="text-xl font-semibold text-black mb-6">Recent Login Activity</h2>
            <LoginActivity userId={effectiveUser?.id} />
          </div>

          {/* Account Actions */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <h2 className="text-xl font-semibold text-black mb-6">Account Actions</h2>
            <div className="space-y-4">
              <div className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                <div>
                  <div className="font-medium text-black">Sign out of all devices</div>
                  <div className="text-sm text-black">This will sign you out of all other sessions</div>
                </div>
                <button 
                  onClick={handleSignOutAll}
                  disabled={signingOutAll || !effectiveUser?.id}
                  className="bg-yellow-600 text-white px-4 py-2 rounded-lg hover:bg-yellow-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {signingOutAll ? 'Signing Out...' : 'Sign Out All'}
                </button>
              </div>
              
            </div>
          </div>

          {/* Delete Account */}
          <div className="bg-white rounded-xl shadow-sm border border-red-200 p-8">
            <h2 className="text-xl font-semibold text-red-600 mb-6">Delete Account</h2>
            <div className="space-y-4">
              <div className="bg-red-50 border border-red-200 rounded-lg p-4">
                <div className="flex items-start">
                  <svg className="w-5 h-5 text-red-600 mr-3 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z" />
                  </svg>
                  <div>
                    <h3 className="font-medium text-red-800">Danger Zone</h3>
                    <p className="text-red-700 text-sm mt-1">
                      Once you delete your account, there is no going back. This action will permanently delete all your data, including:
                    </p>
                    <ul className="text-red-700 text-sm mt-2 space-y-1">
                      <li>• All betting history and transactions</li>
                      <li>• Profile information and preferences</li>
                      <li>• Login sessions and security settings</li>
                      <li>• Any pending bets or withdrawals</li>
                    </ul>
                  </div>
                </div>
              </div>
              
              <div className="flex items-center justify-between p-4 border border-red-200 rounded-lg bg-red-50">
                <div>
                  <div className="font-medium text-red-800">Delete Account</div>
                  <div className="text-sm text-red-600">This action cannot be undone</div>
                </div>
                <button 
                  onClick={() => setShowDeleteModal(true)}
                  className="bg-red-600 text-white px-4 py-2 rounded-lg hover:bg-red-700 transition-colors"
                >
                  Delete Account
                </button>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Delete Account Modal */}
      <DeleteAccountModal
        isOpen={showDeleteModal}
        onClose={() => setShowDeleteModal(false)}
        userId={user?.id}
        onSuccess={handleDeleteAccountSuccess}
      />
    </div>
  )
}