'use client'

import { useState } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { useToast } from '@/components/ui/Toast'

export default function TestEmailPage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <TestEmailContent user={user} />
    </FastSessionWrapper>
  )
}

function TestEmailContent({ user }: { user: any }) {
  const [loading, setLoading] = useState(false)
  const [emailType, setEmailType] = useState('welcome')
  const [testEmail, setTestEmail] = useState('')
  const { addToast } = useToast()

  const testEmailTypes = [
    { value: 'welcome', label: 'Welcome Email', description: 'New user welcome email' },
    { value: 'bet_result', label: 'Bet Result Email', description: 'Bet won/lost notification' },
    { value: 'deposit_confirmation', label: 'Deposit Confirmation', description: 'Deposit received email' },
    { value: 'withdrawal_confirmation', label: 'Withdrawal Confirmation', description: 'Withdrawal processed email' },
    { value: 'market_resolution', label: 'Market Resolution', description: 'Market closed email' },
    { value: 'kyc_status', label: 'KYC Status', description: 'KYC approval/rejection email' },
    { value: 'refund', label: 'Refund Email', description: 'Bet refund notification' },
    { value: 'market_refund', label: 'Market Refund Email', description: 'Market refund notification' },
    { value: 'generic', label: 'Generic Email', description: 'Custom email content' }
  ]

  const handleSendTestEmail = async () => {
    if (!testEmail) {
      addToast({
        type: 'error',
        title: 'Error',
        message: 'Please enter an email address'
      })
      return
    }

    setLoading(true)

    try {
      const response = await fetch('/api/email/send', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          to: testEmail,
          type: emailType,
          subject: `Test ${testEmailTypes.find(t => t.value === emailType)?.label}`,
          data: getTestDataForType(emailType)
        })
      })

      const result = await response.json()

      if (result.success) {
        addToast({
          type: 'success',
          title: 'Email Sent!',
          message: `Test ${testEmailTypes.find(t => t.value === emailType)?.label} sent successfully`
        })
      } else {
        addToast({
          type: 'error',
          title: 'Email Failed',
          message: result.error || 'Failed to send email'
        })
      }
    } catch (error) {
      console.error('Error sending test email:', error)
      addToast({
        type: 'error',
        title: 'Error',
        message: 'Failed to send test email'
      })
    } finally {
      setLoading(false)
    }
  }

  const getTestDataForType = (type: string) => {
    switch (type) {
      case 'welcome':
        return { userName: 'Test User' }
      case 'bet_result':
        return { 
          userName: 'Test User', 
          outcome: 'won', 
          amount: 150.00, 
          marketTitle: 'Test Market: Will it rain tomorrow?' 
        }
      case 'deposit_confirmation':
        return { 
          userName: 'Test User', 
          amount: 500.00, 
          transactionId: 'test_deposit_123' 
        }
      case 'withdrawal_confirmation':
        return { 
          userName: 'Test User', 
          amount: 250.00, 
          transactionId: 'test_withdrawal_456' 
        }
      case 'market_resolution':
        return { 
          userName: 'Test User', 
          marketTitle: 'Test Market: Will it rain tomorrow?', 
          winningOutcome: 'Yes', 
          userOutcome: 'Yes', 
          won: true, 
          amount: 150.00 
        }
      case 'kyc_status':
        return { 
          userName: 'Test User', 
          status: 'approved' 
        }
      default:
        return {}
    }
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <div className="mb-6">
            <h1 className="text-2xl font-bold text-black mb-2">Test Email System</h1>
            <p className="text-gray-600">Send test emails to verify the email notification system is working correctly.</p>
          </div>

          <div className="space-y-6">
            {/* Email Type Selection */}
            <div>
              <label className="block text-sm font-medium text-black mb-3">
                Select Email Type
              </label>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                {testEmailTypes.map((type) => (
                  <div
                    key={type.value}
                    className={`border rounded-lg p-3 cursor-pointer transition-colors ${
                      emailType === type.value
                        ? 'border-blue-500 bg-blue-50'
                        : 'border-gray-200 hover:border-gray-300'
                    }`}
                    onClick={() => setEmailType(type.value)}
                  >
                    <div className="flex items-center space-x-3">
                      <input
                        type="radio"
                        name="emailType"
                        value={type.value}
                        checked={emailType === type.value}
                        onChange={() => setEmailType(type.value)}
                        className="text-blue-600 focus:ring-blue-500"
                      />
                      <div>
                        <div className="font-medium text-black">{type.label}</div>
                        <div className="text-sm text-gray-600">{type.description}</div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Email Address Input */}
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Test Email Address
              </label>
              <input
                type="email"
                value={testEmail}
                onChange={(e) => setTestEmail(e.target.value)}
                placeholder="Enter email address to send test to"
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
              <p className="text-xs text-gray-600 mt-1">
                Enter a valid email address to receive the test email
              </p>
            </div>

            {/* Test Data Preview */}
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Test Data Preview
              </label>
              <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
                <pre className="text-sm text-gray-700 whitespace-pre-wrap">
                  {JSON.stringify(getTestDataForType(emailType), null, 2)}
                </pre>
              </div>
            </div>

            {/* Send Button */}
            <div className="flex justify-end">
              <button
                onClick={handleSendTestEmail}
                disabled={loading || !testEmail}
                className="bg-blue-600 hover:bg-blue-700 disabled:bg-gray-400 disabled:cursor-not-allowed text-white font-medium py-2 px-6 rounded-lg transition-colors flex items-center space-x-2"
              >
                {loading ? (
                  <>
                    <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                    <span>Sending...</span>
                  </>
                ) : (
                  <>
                    <span>📧</span>
                    <span>Send Test Email</span>
                  </>
                )}
              </button>
            </div>
          </div>

          {/* Configuration Info */}
          <div className="mt-8 p-4 bg-blue-50 border border-blue-200 rounded-lg">
            <h3 className="font-medium text-blue-900 mb-2">📧 Email Configuration</h3>
            <div className="text-sm text-blue-800 space-y-1">
              <p><strong>SMTP Host:</strong> {process.env.NEXT_PUBLIC_SMTP_HOST || 'Not configured'}</p>
              <p><strong>SMTP Port:</strong> {process.env.NEXT_PUBLIC_SMTP_PORT || 'Not configured'}</p>
              <p><strong>SMTP User:</strong> {process.env.NEXT_PUBLIC_SMTP_USER || 'Not configured'}</p>
              <p className="text-xs text-blue-600 mt-2">
                Configure SMTP settings in your environment variables to enable email sending.
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
