'use client'

import { useState } from 'react'
import { unifiedPaymentService } from '@/lib/payments'
import { useToast } from '@/components/ui/Toast'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

export default function TestPaymentsPage() {
  const [loading, setLoading] = useState(false)
  const [selectedProvider, setSelectedProvider] = useState<'paynow_zim' | 'pesapal'>('paynow_zim')
  const [amount, setAmount] = useState('100')
  const { addToast } = useToast()
  
  // USD-only implementation
  const currency = 'USD'

  const handleTestPayment = async () => {
    setLoading(true)
    
    try {
      const paymentRequest = {
        provider: selectedProvider,
        amount: parseFloat(amount),
        currency: currency,
        description: `Test payment via ${selectedProvider}`,
        reference: `test_${Date.now()}`,
        customerEmail: 'test@betfusionzim.com',
        customerPhone: '+263123456789',
        returnUrl: `${window.location.origin}/wallet?deposit=success`,
        cancelUrl: `${window.location.origin}/wallet?deposit=cancelled`
      }

      const response = await unifiedPaymentService.createPayment(paymentRequest)

      if (response.success) {
        addToast({
          type: 'success',
          title: 'Payment Created!',
          message: `Test payment created successfully with ${selectedProvider}`
        })
        
        if (response.paymentUrl) {
          // In a real scenario, this would redirect to the payment provider
          console.log('Payment URL:', response.paymentUrl)
          addToast({
            type: 'info',
            title: 'Payment URL Generated',
            message: 'Check console for payment URL (in production, user would be redirected)'
          })
        }
      } else {
        throw new Error(response.error || 'Payment creation failed')
      }
    } catch (error) {
      addToast({
        type: 'error',
        title: 'Payment Test Failed',
        message: error instanceof Error ? error.message : 'Unknown error occurred'
      })
    } finally {
      setLoading(false)
    }
  }

  const supportedMethods = unifiedPaymentService.getSupportedPaymentMethods(currency)

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-2xl mx-auto px-4">
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
          <h1 className="text-2xl font-bold text-black mb-6">Payment Integration Test</h1>
          
          <div className="space-y-6">
            {/* USD-only notice */}
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-3">
              <div className="flex items-center">
                <div className="text-blue-600 mr-2">💵</div>
                <div className="text-sm text-blue-800">
                  <strong>USD Only:</strong> All transactions are processed in US Dollars
                </div>
              </div>
            </div>

            {/* Amount */}
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Amount
              </label>
              <input
                type="number"
                value={amount}
                onChange={(e) => setAmount(e.target.value)}
                className="block w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-black"
                placeholder="100"
              />
            </div>

            {/* Payment Provider Selection */}
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Payment Provider
              </label>
              <div className="space-y-3">
                {supportedMethods.map((provider) => {
                  const methodInfo = unifiedPaymentService.getPaymentMethodInfo(provider)
                  return (
                    <div
                      key={provider}
                      className={`border-2 rounded-lg p-4 cursor-pointer transition-colors ${
                        selectedProvider === provider
                          ? 'border-blue-500 bg-blue-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                      onClick={() => setSelectedProvider(provider)}
                    >
                      <div className="flex items-center space-x-3">
                        <div className="text-2xl">{methodInfo.icon}</div>
                        <div className="flex-1">
                          <div className="flex items-center space-x-2">
                            <input
                              type="radio"
                              name="paymentProvider"
                              value={provider}
                              checked={selectedProvider === provider}
                              onChange={() => setSelectedProvider(provider)}
                              className="text-blue-600 focus:ring-blue-500"
                            />
                            <h3 className="font-medium text-black">{methodInfo.name}</h3>
                          </div>
                          <p className="text-sm text-gray-600 mt-1">{methodInfo.description}</p>
                          <p className="text-xs text-gray-500 mt-1">
                            Processing time: {methodInfo.processingTime} • Currencies: {methodInfo.supportedCurrencies.join(', ')}
                          </p>
                        </div>
                      </div>
                    </div>
                  )
                })}
              </div>
            </div>

            {/* Test Button */}
            <button
              onClick={handleTestPayment}
              disabled={loading}
              className="w-full bg-blue-600 hover:bg-blue-700 disabled:bg-gray-300 text-white font-medium py-3 px-4 rounded-lg transition-colors flex items-center justify-center"
            >
              {loading ? (
                <LoadingSpinner size="sm" />
              ) : (
                `Test Payment with ${unifiedPaymentService.getPaymentMethodInfo(selectedProvider).name}`
              )}
            </button>

            {/* Information */}
            <div className="bg-yellow-50 rounded-lg p-4">
              <div className="flex items-center space-x-2 mb-2">
                <svg className="w-5 h-5 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <span className="text-sm font-medium text-yellow-800">Test Mode</span>
              </div>
              <p className="text-sm text-yellow-700">
                This is a test page for the payment integration. In production, users would be redirected to the actual payment provider's secure page.
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
