'use client'

import { useState, useEffect } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { useWalletOptimized } from '@/lib/hooks/useWalletOptimized'
import Link from 'next/link'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'
import DepositModal from '@/components/wallet/DepositModal'
import WithdrawModal from '@/components/wallet/WithdrawModal'
import { useToast } from '@/components/ui/Toast'
import TransactionItem from '@/components/ui/TransactionItem'

export default function WalletPage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <WalletContent user={user} />
    </FastSessionWrapper>
  )
}

function WalletContent({ user }: { user: any }) {
  const { profile, transactions, paymentMethods, dailyLimits, loading, error } = useWalletOptimized(user?.id)
  const [showDepositModal, setShowDepositModal] = useState(false)
  const [showWithdrawModal, setShowWithdrawModal] = useState(false)
  const { addToast } = useToast()

  // Handle URL parameters for deposit success/cancellation
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search)
    const depositStatus = urlParams.get('deposit')
    
    if (depositStatus === 'success') {
      addToast({
        type: 'success',
        title: 'Deposit Successful!',
        message: 'Your deposit has been processed successfully and your balance has been updated.'
      })
      // Clean up URL
      window.history.replaceState({}, '', '/wallet')
    } else if (depositStatus === 'cancelled') {
      addToast({
        type: 'warning',
        title: 'Deposit Cancelled',
        message: 'Your deposit was cancelled. No charges have been made to your account.'
      })
      // Clean up URL
      window.history.replaceState({}, '', '/wallet')
    }
  }, [addToast])

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading wallet...</p>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Error Loading Wallet</h1>
          <p className="text-black mb-4">{error}</p>
          <Link href="/" className="text-blue-600 hover:text-blue-700">
            Go Home
          </Link>
        </div>
      </div>
    )
  }

  if (!profile) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Profile Not Found</h1>
          <p className="text-black">Unable to load your profile information.</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4 sm:py-6 lg:py-8">
        {/* Header */}
        <div className="mb-6 sm:mb-8">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
            <div>
              <h1 className="text-2xl sm:text-3xl lg:text-4xl font-bold text-black">Wallet</h1>
              <p className="text-black mt-1 sm:mt-2 text-sm sm:text-base">Manage your funds and transactions</p>
            </div>
            <div className="flex flex-col sm:flex-row sm:items-center space-y-2 sm:space-y-0 sm:space-x-4">
              <div className="flex items-center space-x-2 text-xs sm:text-sm text-gray-600">
                <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                <span>Live Data</span>
              </div>
              <div className="text-xs sm:text-sm text-gray-500">
                Real-time updates enabled
              </div>
            </div>
          </div>
        </div>

        {/* Balance Card */}
        <div className="bg-gradient-to-br from-blue-600 to-purple-600 rounded-xl shadow-lg text-white p-4 sm:p-6 lg:p-8 mb-6 sm:mb-8">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
            <div className="flex-1">
              <h2 className="text-base sm:text-lg font-medium opacity-90">Account Balance</h2>
              <div className="text-2xl sm:text-3xl lg:text-4xl font-bold mt-1 sm:mt-2">{formatCurrency(profile.balance)}</div>
              <div className="text-xs sm:text-sm opacity-75 mt-1">Available for betting</div>
            </div>
            <div className="w-12 h-12 sm:w-16 sm:h-16 bg-white bg-opacity-20 rounded-full flex items-center justify-center flex-shrink-0">
              <svg className="w-6 h-6 sm:w-8 sm:h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1" />
              </svg>
            </div>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-4 sm:gap-6 lg:gap-8">
          {/* Deposit/Withdraw */}
          <div className="space-y-4 sm:space-y-6">
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-4 sm:p-6">
              <h3 className="text-base sm:text-lg font-semibold text-black mb-3 sm:mb-4">Deposit Funds</h3>
              <div className="space-y-3 sm:space-y-4">
                <div className="bg-blue-50 rounded-lg p-3 sm:p-4">
                  <div className="flex items-center space-x-2 mb-2">
                    <span className="text-xl sm:text-2xl">🇿🇼</span>
                    <span className="text-xl sm:text-2xl">🌍</span>
                  </div>
                  <p className="text-xs sm:text-sm text-blue-800 mb-2 sm:mb-3">
                    Deposit using PayNow Zim or Pesapal. Choose from multiple payment methods including mobile money, bank transfers, and digital wallets.
                  </p>
                  <div className="flex flex-wrap gap-1 sm:gap-2 text-xs text-blue-700">
                    <span className="bg-blue-100 px-2 py-1 rounded">PayNow Zim</span>
                    <span className="bg-blue-100 px-2 py-1 rounded">Pesapal</span>
                    <span className="bg-blue-100 px-2 py-1 rounded">USD Only</span>
                  </div>
                </div>
                <button 
                  onClick={() => setShowDepositModal(true)}
                  className="w-full bg-green-600 text-white py-2.5 sm:py-3 px-3 sm:px-4 rounded-lg hover:bg-green-700 transition-colors font-medium flex items-center justify-center space-x-2 text-sm sm:text-base"
                >
                  <svg className="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6v6m0 0v6m0-6h6m-6 0H6" />
                  </svg>
                  <span>Deposit Funds</span>
                </button>
              </div>
            </div>

            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-4 sm:p-6">
              <h3 className="text-base sm:text-lg font-semibold text-black mb-3 sm:mb-4">Withdraw Funds</h3>
              <div className="space-y-3 sm:space-y-4">
                <div className="bg-orange-50 rounded-lg p-3 sm:p-4">
                  <div className="flex items-center space-x-2 mb-2">
                    <svg className="w-4 h-4 sm:w-5 sm:h-5 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z" />
                    </svg>
                    <span className="text-xs sm:text-sm font-medium text-orange-800">Processing Time</span>
                  </div>
                  <p className="text-xs sm:text-sm text-orange-700 mb-2">
                    Withdrawals are processed within 1-3 business days. Minimum withdrawal amount is $25.
                  </p>
                </div>
                <button 
                  onClick={() => setShowWithdrawModal(true)}
                  className="w-full bg-red-600 text-white py-2.5 sm:py-3 px-3 sm:px-4 rounded-lg hover:bg-red-700 transition-colors font-medium flex items-center justify-center space-x-2 text-sm sm:text-base"
                >
                  <svg className="w-4 h-4 sm:w-5 sm:h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M20 12H4m8-8l-8 8 8 8" />
                  </svg>
                  <span>Withdraw Funds</span>
                </button>
              </div>
            </div>
          </div>

          {/* Transaction History */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-4 sm:p-6">
            <h3 className="text-base sm:text-lg font-semibold text-black mb-3 sm:mb-4">Recent Transactions</h3>
            <div className="space-y-2 sm:space-y-3">
              {transactions.length > 0 ? (
                transactions.slice(0, 15).map((transaction) => (
                  <TransactionItem 
                    key={transaction.id} 
                    transaction={transaction}
                    showMarketLink={true}
                  />
                ))
              ) : (
                <div className="text-center py-6 sm:py-8">
                  <div className="text-black mb-2 text-sm sm:text-base">No transactions yet</div>
                  <p className="text-xs sm:text-sm text-black">Your transaction history will appear here</p>
                </div>
              )}
            </div>
            
            <div className="mt-4 sm:mt-6">
              <Link 
                href="/portfolio" 
                className="text-blue-600 hover:text-blue-700 text-xs sm:text-sm font-medium"
              >
                View All Transactions →
              </Link>
            </div>
          </div>
        </div>

        {/* Limits & Settings */}
        <div className="mt-6 sm:mt-8 bg-white rounded-xl shadow-sm border border-gray-200 p-4 sm:p-6">
          <h3 className="text-base sm:text-lg font-semibold text-black mb-3 sm:mb-4">Account Limits</h3>
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4 lg:gap-6">
            <div className="text-center p-3 sm:p-4 bg-gray-50 rounded-lg">
              <div className="text-lg sm:text-xl lg:text-2xl font-bold text-blue-600">$5,000</div>
              <div className="text-xs sm:text-sm text-black">Daily Deposit Limit</div>
            </div>
            <div className="text-center p-3 sm:p-4 bg-gray-50 rounded-lg">
              <div className="text-lg sm:text-xl lg:text-2xl font-bold text-purple-600">$5,000</div>
              <div className="text-xs sm:text-sm text-black">Daily Withdrawal Limit</div>
            </div>
            <div className="text-center p-3 sm:p-4 bg-gray-50 rounded-lg sm:col-span-2 lg:col-span-1">
              <div className="text-lg sm:text-xl lg:text-2xl font-bold text-green-600">$50,000</div>
              <div className="text-xs sm:text-sm text-black">Monthly Limit</div>
            </div>
          </div>
          <div className="mt-3 sm:mt-4">
            <Link 
              href="/settings" 
              className="text-blue-600 hover:text-blue-700 text-xs sm:text-sm font-medium"
            >
              Adjust Limits →
            </Link>
          </div>
        </div>
      </div>

      {/* Modals */}
      {showDepositModal && (
        <DepositModal
          profile={profile}
          paymentMethods={paymentMethods}
          onClose={() => setShowDepositModal(false)}
        />
      )}

      {showWithdrawModal && (
        <WithdrawModal
          profile={profile}
          paymentMethods={paymentMethods}
          onClose={() => setShowWithdrawModal(false)}
        />
      )}
    </div>
  )
}