'use client'

import { useAuth } from '@/components/auth/FastAuthProvider'
import FastSessionWrapper from '@/components/auth/FastSessionWrapper'
import { useWatchlist } from '@/lib/hooks/useWatchlist'
import Link from 'next/link'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

interface Market {
  id: string
  title: string
  description: string
  category: string
  status: string
  yes_price: number
  no_price: number
  total_volume: number
  end_date: string
  created_at: string
  creator_id: string
  creator: {
    email: string
  }
}

interface WatchlistItem {
  id: string
  market_id: string
  created_at: string
  markets: Market
}

export default function WatchlistPage() {
  const { user } = useAuth()
  
  return (
    <FastSessionWrapper requireAuth={true}>
      <WatchlistContent user={user} />
    </FastSessionWrapper>
  )
}

function WatchlistContent({ user }: { user: any }) {
  const { 
    watchlistItems, 
    loading, 
    error, 
    removeFromWatchlist 
  } = useWatchlist(user?.id)

  // Format price
  const formatPrice = (price: number) => {
    return `$${(price * 100).toFixed(0)}`
  }

  // Format date
  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString()
  }

  // Get status color
  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
        return 'text-green-600 bg-green-100'
      case 'resolved':
        return 'text-blue-600 bg-blue-100'
      case 'cancelled':
        return 'text-red-600 bg-red-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading your watchlist...</p>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="text-red-600 text-lg font-medium mb-2">Error</div>
          <p className="text-black">{error}</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-6xl mx-auto px-4">
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-black">My Watchlist</h1>
          <p className="text-black mt-2">
            {watchlistItems.length} market{watchlistItems.length !== 1 ? 's' : ''} in your watchlist
          </p>
        </div>

        {watchlistItems.length === 0 ? (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
            <div className="text-center">
              <svg className="mx-auto h-16 w-16 text-black" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z" />
              </svg>
              <h3 className="mt-4 text-lg font-medium text-black">No markets in your watchlist yet</h3>
              <p className="mt-2 text-black">
                Start watching markets to track their performance and get notified of updates.
              </p>
              <div className="mt-6">
                <Link 
                  href="/" 
                  className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                >
                  Browse Markets
                </Link>
              </div>
            </div>
          </div>
        ) : (
          <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
            {watchlistItems.map((item) => {
              const market = item.markets
              if (!market) return null

              return (
                <div key={item.id} className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow h-full flex flex-col">
                  <div className="flex justify-between items-start mb-4">
                    <div className="flex-1">
                      <Link 
                        href={`/markets/${market.id}`}
                        className="text-lg font-semibold text-black hover:text-blue-600 transition-colors line-clamp-2"
                      >
                        {market.title}
                      </Link>
                      <p className="text-sm text-black mt-1 line-clamp-2">
                        {market.description}
                      </p>
                    </div>
                    <button
                      onClick={() => removeFromWatchlist(market.id)}
                      className="ml-2 p-1 text-gray-400 hover:text-red-600 transition-colors"
                      title="Remove from watchlist"
                    >
                      <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24">
                        <path d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z" />
                      </svg>
                    </button>
                  </div>

                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Yes</span>
                      <span className="font-semibold text-green-600">
                        {formatPrice(market.yes_price)}
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">No</span>
                      <span className="font-semibold text-red-600">
                        {formatPrice(market.no_price)}
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-black">Volume</span>
                      <span className="font-semibold text-black">
                        ${market.total_volume.toLocaleString()}
                      </span>
                    </div>
                  </div>

                  <div className="mt-4 pt-4 border-t border-gray-200">
                    <div className="flex items-center justify-between">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${getStatusColor(market.status)}`}>
                        {market.status.charAt(0).toUpperCase() + market.status.slice(1)}
                      </span>
                      <span className="text-xs text-black">
                        Ends {formatDate(market.end_date)}
                      </span>
                    </div>
                    <div className="mt-2 text-xs text-black">
                      Added {formatDate(item.created_at)}
                    </div>
                  </div>
                </div>
              )
            })}
          </div>
        )}
      </div>
    </div>
  )
}