'use client'

import { useState } from 'react'
import { useAuth } from './auth/ClientAuthProvider'
import { supabase } from '@/lib/supabase-client'

export default function AuthTestPanel() {
  const { user, profile, loading } = useAuth()
  const [testResults, setTestResults] = useState<any>({})
  const [isExpanded, setIsExpanded] = useState(false)

  const runQuickTests = async () => {
    const results: any = {}

    // Test 1: Session check
    try {
      const { data, error } = await supabase.auth.getSession()
      results.session = {
        success: !error,
        hasSession: !!data.session,
        error: error?.message || 'None'
      }
    } catch (err) {
      results.session = {
        success: false,
        error: err instanceof Error ? err.message : 'Unknown error'
      }
    }

    // Test 2: Database query
    try {
      const { data, error } = await supabase
        .from('profiles')
        .select('id')
        .limit(1)
      results.database = {
        success: !error,
        error: error?.message || 'None'
      }
    } catch (err) {
      results.database = {
        success: false,
        error: err instanceof Error ? err.message : 'Unknown error'
      }
    }

    // Test 3: Auth state listener
    try {
      const { data: { subscription } } = supabase.auth.onAuthStateChange(
        (event, session) => {
          console.log('Quick test auth change:', event)
        }
      )
      subscription.unsubscribe()
      results.authListener = {
        success: true,
        message: 'Auth listener works'
      }
    } catch (err) {
      results.authListener = {
        success: false,
        error: err instanceof Error ? err.message : 'Unknown error'
      }
    }

    setTestResults(results)
  }

  if (process.env.NODE_ENV !== 'development') {
    return null
  }

  return (
    <div className="fixed bottom-4 left-4 bg-black text-white p-4 rounded-lg text-xs max-w-sm z-50">
      <div className="flex items-center justify-between mb-2">
        <h3 className="font-bold">Auth Debug</h3>
        <button
          onClick={() => setIsExpanded(!isExpanded)}
          className="text-white hover:text-gray-300"
        >
          {isExpanded ? '−' : '+'}
        </button>
      </div>
      
      <div className="space-y-1">
        <div>Loading: {loading ? 'Yes' : 'No'}</div>
        <div>User: {user ? user.email : 'None'}</div>
        <div>Profile: {profile ? 'Loaded' : 'None'}</div>
        <div>Admin: {profile?.is_admin ? 'Yes' : 'No'}</div>
      </div>

      {isExpanded && (
        <div className="mt-4 space-y-2">
          <button
            onClick={runQuickTests}
            className="bg-blue-600 text-white px-2 py-1 rounded text-xs hover:bg-blue-700"
          >
            Run Tests
          </button>
          
          {Object.keys(testResults).length > 0 && (
            <div className="space-y-1">
              {Object.entries(testResults).map(([test, result]: [string, any]) => (
                <div key={test} className={`p-1 rounded text-xs ${result.success ? 'bg-green-600' : 'bg-red-600'}`}>
                  {test}: {result.success ? '✓' : '✗'}
                </div>
              ))}
            </div>
          )}

          <div className="pt-2 border-t border-gray-600">
            <a href="/debug" className="text-blue-400 hover:text-blue-300 text-xs">
              Full Debug Page →
            </a>
          </div>
        </div>
      )}
    </div>
  )
}
