'use client'

import React, { memo, useState } from 'react'
import { AdminBet } from '@/lib/hooks/useAdminBetsOptimized'
import { LoadingSpinner, ButtonLoadingSpinner } from '@/components/ui/LoadingSpinner'
import { useToast } from '@/components/ui/Toast'
import { supabase } from '@/lib/supabase-client'

interface AdminBetsTableProps {
  bets: AdminBet[]
  loading: boolean
  hasMore: boolean
  onLoadMore: () => void
  onRefresh: () => void
}

const BetRow = memo(({ bet, onAction }: { 
  bet: AdminBet, 
  onAction: (betId: string, action: string) => Promise<void> 
}) => {
  const [actionLoading, setActionLoading] = useState(false)

  const handleAction = async (action: string) => {
    setActionLoading(true)
    try {
      await onAction(bet.id, action)
    } finally {
      setActionLoading(false)
    }
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'seeded':
        return 'bg-blue-100 text-blue-800'
      case 'live':
        return 'bg-green-100 text-green-800'
      case 'matched':
        return 'bg-purple-100 text-purple-800'
      case 'won':
        return 'bg-green-100 text-green-800'
      case 'lost':
        return 'bg-red-100 text-red-800'
      case 'cancelled':
        return 'bg-gray-100 text-black'
      case 'refunded':
        return 'bg-yellow-100 text-yellow-800'
      default:
        return 'bg-gray-100 text-black'
    }
  }

  const getRiskColor = (riskScore: number) => {
    if (riskScore < 30) return 'text-green-600 bg-green-100'
    if (riskScore < 60) return 'text-yellow-600 bg-yellow-100'
    return 'text-red-600 bg-red-100'
  }

  return (
    <tr className="hover:bg-gray-50">
      <td className="px-6 py-4">
        <div className="space-y-1">
          <div className="text-sm font-medium text-black">
            ${bet.amount.toLocaleString()} on {bet.outcome}
          </div>
          <div className="text-sm text-black">
            Price: ${bet.price.toFixed(2)} | Shares: {bet.shares}
          </div>
          <div className="text-xs text-black">
            {new Date(bet.created_at).toLocaleString()}
          </div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap">
        <div>
          <div className="text-sm font-medium text-black">{bet.user_profile?.username || 'Unknown'}</div>
          <div className="text-sm text-black">{bet.user_profile?.email}</div>
        </div>
      </td>
      <td className="px-6 py-4">
        <div className="max-w-xs">
          <div className="text-sm font-medium text-black truncate">{bet.market?.title || 'Unknown Market'}</div>
          <div className="text-sm text-black">
            {bet.market?.status} | ${bet.market?.total_liquidity?.toLocaleString() || 0} liquidity
          </div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
        <div className="space-y-1">
          <div className="text-black">Efficiency: {bet.algorithm_metrics?.market_efficiency.toFixed(1)}%</div>
          <div className="text-black">Price Impact: {bet.algorithm_metrics?.price_impact.toFixed(1)}%</div>
          <div className={`text-xs px-1 py-0.5 rounded ${getRiskColor(bet.algorithm_metrics?.risk_score || 0)}`}>
            Risk: {bet.algorithm_metrics?.risk_score.toFixed(1)}/100
          </div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap">
        <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(bet.status)}`}>
          {bet.status}
        </span>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
        <div className="flex flex-wrap gap-1">
          {bet.status === 'seeded' && (
            <button
              onClick={() => handleAction('match')}
              disabled={actionLoading}
              className="text-green-600 hover:text-green-900 text-xs"
            >
              {actionLoading ? <ButtonLoadingSpinner /> : 'Match'}
            </button>
          )}

          {(bet.status === 'seeded' || bet.status === 'live') && (
            <button
              onClick={() => handleAction('cancel')}
              disabled={actionLoading}
              className="text-red-600 hover:text-red-900 text-xs"
            >
              {actionLoading ? <ButtonLoadingSpinner /> : 'Cancel'}
            </button>
          )}

          {bet.status === 'matched' && (
            <>
              <button
                onClick={() => handleAction('resolve_win')}
                disabled={actionLoading}
                className="text-green-600 hover:text-green-900 text-xs"
              >
                {actionLoading ? <ButtonLoadingSpinner /> : 'Win'}
              </button>
              <button
                onClick={() => handleAction('resolve_loss')}
                disabled={actionLoading}
                className="text-red-600 hover:text-red-900 text-xs"
              >
                {actionLoading ? <ButtonLoadingSpinner /> : 'Loss'}
              </button>
            </>
          )}

          {/* Refund button - only show for certain statuses */}
          {(bet.status === 'seeded' || bet.status === 'live' || bet.status === 'matched') && (
            <button
              onClick={() => handleAction('refund')}
              disabled={actionLoading}
              className="text-yellow-600 hover:text-yellow-900 text-xs"
            >
              {actionLoading ? <ButtonLoadingSpinner /> : 'Refund'}
            </button>
          )}
        </div>
      </td>
    </tr>
  )
})

BetRow.displayName = 'BetRow'

const LoadingRow = memo(() => (
  <tr>
    <td colSpan={6} className="px-6 py-4 text-center">
      <LoadingSpinner size="sm" />
    </td>
  </tr>
))

LoadingRow.displayName = 'LoadingRow'

const EmptyRow = memo(() => (
  <tr>
    <td colSpan={6} className="px-6 py-12 text-center text-gray-500">
      No bets found matching default criteria.
    </td>
  </tr>
))

EmptyRow.displayName = 'EmptyRow'

function AdminBetsTable({ bets, loading, hasMore, onLoadMore, onRefresh }: AdminBetsTableProps) {
  const { addToast } = useToast()

  const handleBetAction = async (betId: string, action: string) => {
    try {
      let updateData: any = {}
      let message = ''

      switch (action) {
        case 'match':
          updateData = { status: 'matched' }
          message = 'Bet matched successfully'
          break
        case 'cancel':
          updateData = { status: 'cancelled' }
          message = 'Bet cancelled'
          break
        case 'refund':
          await handleRefund(betId)
          message = 'Bet refunded successfully'
          break
        case 'resolve_win':
          updateData = { status: 'won' }
          message = 'Bet resolved as winner'
          break
        case 'resolve_loss':
          updateData = { status: 'lost' }
          message = 'Bet resolved as loser'
          break
        default:
          throw new Error(`Unknown action: ${action}`)
      }

      // For non-refund actions, update the bet status
      if (action !== 'refund') {
        const { error } = await supabase
          .from('bets')
          .update(updateData)
          .eq('id', betId)

        if (error) {
          throw error
        }
      }

      addToast({
        type: 'success',
        title: 'Action Completed',
        message: message
      })

      onRefresh()

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to perform action'
      addToast({
        type: 'error',
        title: 'Action Failed',
        message: errorMessage
      })
    }
  }

  const handleRefund = async (betId: string) => {
    try {
      // Get the bet details
      const { data: bet, error: betError } = await supabase
        .from('bets')
        .select(`
          *,
          profiles!bets_user_id_fkey (
            id,
            email,
            username
          )
        `)
        .eq('id', betId)
        .single()

      if (betError || !bet) {
        throw new Error('Bet not found')
      }

      // Check if bet can be refunded
      if (bet.status === 'refunded') {
        throw new Error('Bet is already refunded')
      }
      
      if (bet.status === 'won' || bet.status === 'lost') {
        throw new Error('Cannot refund resolved bets (won/lost)')
      }
      
      if (bet.status === 'cancelled') {
        throw new Error('Cannot refund cancelled bets')
      }

      // Calculate refund amount (original bet amount)
      const refundAmount = bet.amount

      // Start a transaction
      const { data: userProfile, error: userError } = await supabase
        .from('profiles')
        .select('balance')
        .eq('id', bet.user_id)
        .single()

      if (userError || !userProfile) {
        throw new Error('User profile not found')
      }

      // Update user balance
      const newBalance = userProfile.balance + refundAmount
      const { error: balanceError } = await supabase
        .from('profiles')
        .update({ balance: newBalance })
        .eq('id', bet.user_id)

      if (balanceError) {
        throw new Error('Failed to update user balance')
      }

      // Create refund transaction record
      const { error: transactionError } = await supabase
        .from('transactions')
        .insert({
          user_id: bet.user_id,
          type: 'refund',
          amount: refundAmount,
          status: 'completed',
          description: `Refund for bet on ${bet.outcome}`,
          reference: `BET_REFUND_${betId}`,
          created_at: new Date().toISOString()
        })

      if (transactionError) {
        console.error('Failed to create refund transaction:', transactionError)
        // Don't throw error here as the refund is still processed
      }

      // Update bet status to refunded
      const { error: betUpdateError } = await supabase
        .from('bets')
        .update({ 
          status: 'refunded',
          updated_at: new Date().toISOString()
        })
        .eq('id', betId)

      if (betUpdateError) {
        throw new Error('Failed to update bet status')
      }

      // Send refund notification email
      try {
        const { emailService } = await import('@/lib/services/emailService')
        await emailService.sendRefundEmail(
          bet.profiles?.email || '',
          bet.profiles?.username || 'User',
          refundAmount,
          betId,
          bet.outcome,
          bet.market?.title || 'Unknown Market'
        )
      } catch (notificationError) {
        console.error('Failed to send refund notification:', notificationError)
        // Don't throw error here as the refund is still processed
      }

    } catch (error) {
      console.error('Refund error:', error)
      throw error
    }
  }

  return (
    <div className="bg-white rounded-lg shadow-sm overflow-hidden">
      <div className="overflow-x-auto">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Bet Details</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">User</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Market</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Algorithm Metrics</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Status</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Actions</th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {bets.map((bet) => (
              <BetRow 
                key={bet.id} 
                bet={bet} 
                onAction={handleBetAction}
              />
            ))}
            
            {loading && <LoadingRow />}
            
            {!loading && bets.length === 0 && <EmptyRow />}
          </tbody>
        </table>
      </div>
      
      {hasMore && !loading && (
        <div className="px-6 py-4 bg-gray-50 border-t">
          <button
            onClick={onLoadMore}
            className="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Load More Bets
          </button>
        </div>
      )}
    </div>
  )
}

export default memo(AdminBetsTable)
