'use client'

import { useState } from 'react'
import { Market, Profile, Bet } from '@/lib/types'
import CreateMarketForm from './CreateMarketForm'
import MarketsTable from './MarketsTable'
import UsersTable from './UsersTable'
import BetsTable from './BetsTable'
import AdminStats from './AdminStats'

interface AdminDashboardProps {
  markets: Market[]
  profiles: Profile[]
  bets: Bet[]
  stats: {
    totalMarkets: number
    activeMarkets: number
    totalUsers: number
    totalVolume: number
    totalBets: number
  }
}

export default function AdminDashboard({ markets, profiles, bets, stats }: AdminDashboardProps) {
  const [activeTab, setActiveTab] = useState('overview')

  const tabs = [
    { id: 'overview', name: 'Overview', icon: '📊' },
    { id: 'markets', name: 'Markets', icon: '📈' },
    { id: 'users', name: 'Users', icon: '👥' },
    { id: 'bets', name: 'Bets', icon: '💰' },
    { id: 'create', name: 'Create Market', icon: '➕' },
  ]

  return (
    <div className="space-y-6">
      {/* Stats Cards */}
      <AdminStats stats={stats} />

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="-mb-px flex space-x-8 px-6">
            {tabs.map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors ${
                  activeTab === tab.id
                    ? 'border-blue-500 text-blue-600'
                    : 'border-transparent text-black hover:text-black hover:border-gray-300'
                }`}
              >
                <span className="mr-2">{tab.icon}</span>
                {tab.name}
              </button>
            ))}
          </nav>
        </div>

        <div className="p-6">
          {activeTab === 'overview' && (
            <div className="space-y-6">
              <div>
                <h3 className="text-lg font-semibold mb-4">Recent Activity</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="bg-gray-50 rounded-lg p-4">
                    <h4 className="font-medium text-black mb-2">Latest Markets</h4>
                    <div className="space-y-2">
                      {markets.slice(0, 5).map((market) => (
                        <div key={market.id} className="flex items-center justify-between text-sm">
                          <span className="truncate">{market.title}</span>
                          <span className={`px-2 py-1 text-xs rounded-full ${
                            market.status === 'active' ? 'bg-green-100 text-green-800' :
                            market.status === 'closed' ? 'bg-yellow-100 text-yellow-800' :
                            'bg-blue-100 text-blue-800'
                          }`}>
                            {market.status}
                          </span>
                        </div>
                      ))}
                    </div>
                  </div>
                  
                  <div className="bg-gray-50 rounded-lg p-4">
                    <h4 className="font-medium text-black mb-2">Recent Bets</h4>
                    <div className="space-y-2">
                      {bets.slice(0, 5).map((bet) => (
                        <div key={bet.id} className="flex items-center justify-between text-sm">
                          <span className={`px-2 py-1 text-xs rounded-full ${
                            bet.outcome === 'yes' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                          }`}>
                            {bet.outcome}
                          </span>
                          <span>${bet.amount.toFixed(2)}</span>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}

          {activeTab === 'markets' && (
            <MarketsTable markets={markets} />
          )}

          {activeTab === 'users' && (
            <UsersTable profiles={profiles} />
          )}

          {activeTab === 'bets' && (
            <BetsTable bets={bets} />
          )}

          {activeTab === 'create' && (
            <CreateMarketForm />
          )}
        </div>
      </div>
    </div>
  )
}
