'use client'

import React, { memo, useState } from 'react'
import Link from 'next/link'
import { supabase } from '@/lib/supabase-client'
import { useToast } from '@/lib/hooks/useToast'
import { MarketWithStats } from '@/lib/types'

interface AdminMarketsTableProps {
  markets: MarketWithStats[]
  loading: boolean
  hasMore: boolean
  onLoadMore: () => void
  onRefresh: () => void
}

const MarketRow = ({ market, onAction }: { 
  market: MarketWithStats, 
  onAction: (marketId: string, action: string, outcome?: string, notes?: string) => Promise<void> 
}) => {
  const [actionLoading, setActionLoading] = useState(false)
  const [showNotes, setShowNotes] = useState(false)
  const [notes, setNotes] = useState('')

  const handleAction = async (action: string) => {
    setActionLoading(true)
    try {
      if (action === 'resolve') {
        setShowNotes(true)
        return
      }
      await onAction(market.id, action, undefined, notes || undefined)
    } finally {
      setActionLoading(false)
    }
  }

  const handleResolve = async (outcome: 'yes' | 'no') => {
    setActionLoading(true)
    try {
      await onAction(market.id, 'resolve', outcome, notes || undefined)
      setShowNotes(false)
      setNotes('')
    } finally {
      setActionLoading(false)
    }
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    })
  }

  const formatPrice = (price: number) => {
    return (price * 100).toFixed(1) + '%'
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
        return 'bg-green-100 text-green-800'
      case 'resolved':
        return 'bg-blue-100 text-blue-800'
      case 'closed':
        return 'bg-gray-100 text-gray-800'
      default:
        return 'bg-gray-100 text-gray-800'
    }
  }

  const getAdminStatusColor = (adminStatus: string) => {
    switch (adminStatus) {
      case 'pending':
        return 'bg-yellow-100 text-yellow-800'
      case 'approved':
        return 'bg-green-100 text-green-800'
      case 'paused':
        return 'bg-orange-100 text-orange-800'
      case 'denied':
        return 'bg-red-100 text-red-800'
      case 'resolved':
        return 'bg-blue-100 text-blue-800'
      default:
        return 'bg-gray-100 text-gray-800'
    }
  }

  return (
    <tr className="hover:bg-gray-50">
      <td className="px-6 py-4">
        <div>
          <div className="text-sm font-medium text-gray-900 truncate max-w-xs">
            {market.title}
          </div>
          <div className="text-sm text-gray-500 line-clamp-2">
            {market.description}
          </div>
          <div className="text-xs text-gray-400">
            ID: {market.id.slice(0, 8)}...
          </div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap">
        <div className="text-sm text-gray-900">{market.creator_email}</div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap">
        <div className="space-y-2">
          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(market.status)}`}>
            {market.status}
          </span>
          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getAdminStatusColor(market.admin_status)}`}>
            {market.admin_status_display}
          </span>
          {market.outcome && (
            <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
              market.outcome === 'yes' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
            }`}>
              {market.outcome.toUpperCase()}
            </span>
          )}
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
        <div className="space-y-1">
          <div className="font-medium">{formatCurrency(market.total_volume)}</div>
          <div className="text-xs text-gray-500">
            YES {formatPrice(market.yes_price)} • NO {formatPrice(market.no_price)}
          </div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
        <div className="space-y-1">
          <div className="text-blue-600">
            Creator: {formatCurrency(market.creator_earnings)}
          </div>
          <div className="text-purple-600">
            Admin: {formatCurrency(market.admin_fee)}
          </div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
        <div className="space-y-1">
          <div>Created: {formatDate(market.created_at)}</div>
          <div>Ends: {formatDate(market.end_date)}</div>
          {market.resolved_at && (
            <div className="text-blue-600">
              Resolved: {formatDate(market.resolved_at)}
            </div>
          )}
          {market.admin_action_at && (
            <div className="text-gray-600">
              Admin Action: {formatDate(market.admin_action_at)}
            </div>
          )}
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
        <div className="flex flex-col space-y-2">
          {/* Admin Actions */}
          <div className="flex flex-wrap gap-1">
            {market.admin_status === 'pending' && (
              <>
                <button
                  onClick={() => handleAction('approve')}
                  disabled={actionLoading}
                  className="text-green-600 hover:text-green-900 text-xs px-2 py-1 border border-green-300 rounded"
                >
                  {actionLoading ? '...' : 'Approve'}
                </button>
                <button
                  onClick={() => handleAction('deny')}
                  disabled={actionLoading}
                  className="text-red-600 hover:text-red-900 text-xs px-2 py-1 border border-red-300 rounded"
                >
                  {actionLoading ? '...' : 'Deny'}
                </button>
              </>
            )}

            {market.admin_status === 'approved' && (
              <button
                onClick={() => handleAction('pause')}
                disabled={actionLoading}
                className="text-orange-600 hover:text-orange-900 text-xs px-2 py-1 border border-orange-300 rounded"
              >
                {actionLoading ? '...' : 'Pause'}
              </button>
            )}

            {market.admin_status === 'paused' && (
              <button
                onClick={() => handleAction('approve')}
                disabled={actionLoading}
                className="text-green-600 hover:text-green-900 text-xs px-2 py-1 border border-green-300 rounded"
              >
                {actionLoading ? '...' : 'Resume'}
              </button>
            )}

            {(market.admin_status === 'approved' || market.admin_status === 'paused') && (
              <button
                onClick={() => handleAction('resolve')}
                disabled={actionLoading}
                className="text-blue-600 hover:text-blue-900 text-xs px-2 py-1 border border-blue-300 rounded"
              >
                {actionLoading ? '...' : 'Resolve'}
              </button>
            )}

            {/* Market Refund Button - only for active markets with bets */}
            {(market.admin_status === 'approved' || market.admin_status === 'paused') && (
              <button
                onClick={() => handleAction('refund')}
                disabled={actionLoading}
                className="text-yellow-600 hover:text-yellow-900 text-xs px-2 py-1 border border-yellow-300 rounded"
              >
                {actionLoading ? '...' : 'Refund Market'}
              </button>
            )}
          </div>

          {/* Standard Actions */}
          <div className="flex gap-1">
            <Link
              href={`/markets/${market.id}`}
              className="text-blue-600 hover:text-blue-900 text-xs"
            >
              View
            </Link>
            <Link
              href={`/admin/resolve-markets`}
              className="text-green-600 hover:text-green-900 text-xs"
            >
              Resolve
            </Link>
          </div>
        </div>

        {/* Resolution Modal */}
        {showNotes && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white p-6 rounded-lg max-w-md w-full mx-4">
              <h3 className="text-lg font-semibold mb-4">Resolve Market</h3>
              <div className="mb-4">
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Admin Notes (Optional)
                </label>
                <textarea
                  value={notes}
                  onChange={(e) => setNotes(e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  rows={3}
                  placeholder="Add notes about the resolution..."
                />
              </div>
              <div className="flex gap-2">
                <button
                  onClick={() => handleResolve('yes')}
                  disabled={actionLoading}
                  className="flex-1 bg-green-600 text-white py-2 px-4 rounded-lg hover:bg-green-700 disabled:opacity-50"
                >
                  {actionLoading ? 'Resolving...' : 'Resolve YES'}
                </button>
                <button
                  onClick={() => handleResolve('no')}
                  disabled={actionLoading}
                  className="flex-1 bg-red-600 text-white py-2 px-4 rounded-lg hover:bg-red-700 disabled:opacity-50"
                >
                  {actionLoading ? 'Resolving...' : 'Resolve NO'}
                </button>
                <button
                  onClick={() => {
                    setShowNotes(false)
                    setNotes('')
                  }}
                  className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
                >
                  Cancel
                </button>
              </div>
            </div>
          </div>
        )}
      </td>
    </tr>
  )
}

MarketRow.displayName = 'MarketRow'

const LoadingRow = memo(() => (
  <tr>
    <td colSpan={7} className="px-6 py-4 text-center">
      <div className="animate-pulse">Loading markets...</div>
    </td>
  </tr>
))

LoadingRow.displayName = 'LoadingRow'

const EmptyRow = memo(() => (
  <tr>
    <td colSpan={7} className="px-6 py-12 text-center text-gray-500">
      No markets found matching the current filters.
    </td>
  </tr>
))

EmptyRow.displayName = 'EmptyRow'

function AdminMarketsTableEnhanced({ markets, loading, hasMore, onLoadMore, onRefresh }: AdminMarketsTableProps) {
  const { addToast } = useToast()

  const handleMarketAction = async (marketId: string, action: string, outcome?: string, notes?: string) => {
    try {
      const { data: { user } } = await supabase.auth.getUser()
      if (!user) {
        throw new Error('You must be logged in to perform admin actions')
      }

      let result
      if (action === 'resolve') {
        result = await supabase.rpc('admin_resolve_market', {
          market_id: marketId,
          outcome: outcome,
          admin_user_id: user.id,
          notes: notes || null
        })
      } else if (action === 'refund') {
        await handleMarketRefund(marketId, user.id)
        result = { error: null } // Success handled in handleMarketRefund
      } else {
        result = await supabase.rpc('admin_market_action', {
          market_id: marketId,
          action_type: action,
          admin_user_id: user.id,
          notes: notes || null
        })
      }

      if (result.error) {
        throw result.error
      }

      const actionMessages = {
        'approve': 'Market approved successfully',
        'pause': 'Market paused successfully',
        'deny': 'Market denied successfully',
        'resolve': `Market resolved as ${outcome?.toUpperCase()} successfully`,
        'refund': 'Market refunded successfully - all eligible bets have been refunded to users'
      }

      addToast({
        type: 'success',
        title: 'Action Completed',
        message: actionMessages[action as keyof typeof actionMessages] || 'Action completed successfully'
      })

      onRefresh()

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to perform action'
      addToast({
        type: 'error',
        title: 'Action Failed',
        message: errorMessage
      })
    }
  }

  const handleMarketRefund = async (marketId: string, adminUserId: string) => {
    try {
      console.log(`Starting market refund for market ${marketId} by admin ${adminUserId}`)
      
      // First, check if the market exists
      const { data: market, error: marketError } = await supabase
        .from('markets')
        .select('id, title, status, admin_status')
        .eq('id', marketId)
        .single()

      if (marketError) {
        console.error('Market fetch error:', marketError)
        throw new Error(`Market not found: ${marketError.message}`)
      }

      console.log('Market found:', market)

      // Get all bets for this market (including all statuses for market refund)
      const { data: bets, error: betsError } = await supabase
        .from('bets')
        .select(`
          *,
          profiles!bets_user_id_fkey (
            id,
            email,
            username
          )
        `)
        .eq('market_id', marketId)

      if (betsError) {
        throw new Error('Failed to fetch market bets')
      }

      // Filter for eligible bets (not already refunded, won, or lost)
      const eligibleBets = (bets || []).filter(bet => 
        !['refunded', 'won', 'lost'].includes(bet.status)
      )

      console.log(`Market refund for ${marketId}: Found ${bets?.length || 0} total bets, ${eligibleBets.length} eligible bets`)

      if (eligibleBets.length === 0) {
        // If no eligible bets, still process the market refund for consistency
        console.log('No eligible bets found, processing market refund anyway')
        
        // Update market status to refunded even without bets
        const updateData = {
          status: 'refunded',
          admin_status: 'refunded',
          admin_action_at: new Date().toISOString(),
          admin_action_by: adminUserId,
          admin_notes: 'Market refunded - no eligible bets found'
        }
        
        console.log('Updating market with data:', updateData)
        
        const { error: marketUpdateError } = await supabase
          .from('markets')
          .update(updateData)
          .eq('id', marketId)

        if (marketUpdateError) {
          console.error('Market update error:', marketUpdateError)
          
          // Try a simpler update as fallback
          console.log('Trying simpler market update...')
          const { error: simpleUpdateError } = await supabase
            .from('markets')
            .update({ 
              status: 'refunded',
              admin_notes: 'Market refunded - no eligible bets found'
            })
            .eq('id', marketId)
          
          if (simpleUpdateError) {
            console.error('Simple update also failed:', simpleUpdateError)
            throw new Error(`Failed to update market status: ${marketUpdateError.message}`)
          } else {
            console.log('Simple update succeeded')
          }
        }

        console.log('Market refund completed - no eligible bets to refund')
        return // Exit early if no eligible bets
      }

      // Group eligible bets by user for efficient processing
      const userBets = new Map<string, any[]>()
      eligibleBets.forEach(bet => {
        if (!userBets.has(bet.user_id)) {
          userBets.set(bet.user_id, [])
        }
        userBets.get(bet.user_id)!.push(bet)
      })

      // Process refunds for each user
      for (const [userId, userBetList] of userBets) {
        const totalRefundAmount = userBetList.reduce((sum, bet) => sum + bet.amount, 0)
        
        // Get user's current balance
        const { data: userProfile, error: userError } = await supabase
          .from('profiles')
          .select('balance')
          .eq('id', userId)
          .single()

        if (userError || !userProfile) {
          console.error(`Failed to get user profile for ${userId}:`, userError)
          continue
        }

        // Update user balance
        const newBalance = userProfile.balance + totalRefundAmount
        const { error: balanceError } = await supabase
          .from('profiles')
          .update({ balance: newBalance })
          .eq('id', userId)

        if (balanceError) {
          console.error(`Failed to update balance for user ${userId}:`, balanceError)
          continue
        }

        // Create refund transaction record
        const { error: transactionError } = await supabase
          .from('transactions')
          .insert({
            user_id: userId,
            type: 'market_refund',
            amount: totalRefundAmount,
            status: 'completed',
            description: `Market refund for ${userBetList.length} bet(s)`,
            reference: `MARKET_REFUND_${marketId}`,
            created_at: new Date().toISOString()
          })

        if (transactionError) {
          console.error(`Failed to create refund transaction for user ${userId}:`, transactionError)
        }

        // Update all bets to refunded status
        const betIds = userBetList.map(bet => bet.id)
        const { error: betsUpdateError } = await supabase
          .from('bets')
          .update({ 
            status: 'refunded',
            updated_at: new Date().toISOString()
          })
          .in('id', betIds)

        if (betsUpdateError) {
          console.error(`Failed to update bet statuses for user ${userId}:`, betsUpdateError)
        }

        // Send refund notification email
        try {
          const { emailService } = await import('@/lib/services/emailService')
          await emailService.sendMarketRefundEmail(
            userBetList[0].profiles?.email || '',
            userBetList[0].profiles?.username || 'User',
            totalRefundAmount,
            marketId,
            userBetList.length
          )
        } catch (emailError) {
          console.error(`Failed to send refund email for user ${userId}:`, emailError)
        }
      }

      // Update market status to refunded
      const updateData = {
        status: 'refunded',
        admin_status: 'refunded',
        admin_action_at: new Date().toISOString(),
        admin_action_by: adminUserId,
        admin_notes: 'Market refunded - all bets returned to users'
      }
      
      console.log('Updating market with data:', updateData)
      
      const { error: marketUpdateError } = await supabase
        .from('markets')
        .update(updateData)
        .eq('id', marketId)

      if (marketUpdateError) {
        console.error('Market update error:', marketUpdateError)
        
        // Try a simpler update as fallback
        console.log('Trying simpler market update...')
        const { error: simpleUpdateError } = await supabase
          .from('markets')
          .update({ 
            status: 'refunded',
            admin_notes: 'Market refunded - all bets returned to users'
          })
          .eq('id', marketId)
        
        if (simpleUpdateError) {
          console.error('Simple update also failed:', simpleUpdateError)
          throw new Error(`Failed to update market status: ${marketUpdateError.message}`)
        } else {
          console.log('Simple update succeeded')
        }
      }

    } catch (error) {
      console.error('Market refund error:', error)
      throw error
    }
  }

  return (
    <div className="bg-white rounded-lg shadow-sm overflow-hidden">
      <div className="overflow-x-auto">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Market
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Creator
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Status & Admin Status
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Volume & Prices
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Earnings
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Dates
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Admin Actions
              </th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {markets.map((market) => (
              <MarketRow 
                key={market.id} 
                market={market} 
                onAction={handleMarketAction}
              />
            ))}
            
            {loading && <LoadingRow />}
            
            {!loading && markets.length === 0 && <EmptyRow />}
          </tbody>
        </table>
      </div>

      {hasMore && (
        <div className="px-6 py-4 border-t border-gray-200 bg-gray-50">
          <button
            onClick={onLoadMore}
            disabled={loading}
            className="w-full bg-blue-600 text-white py-2 px-4 rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {loading ? 'Loading...' : 'Load More Markets'}
          </button>
        </div>
      )}
    </div>
  )
}

export default memo(AdminMarketsTableEnhanced)
